#include <QTime>
#include <QAction>
#include <QMenuBar>
#include <QDebug>

#include "qflipclock.h"

// minimum padding for noBackground mode
const qreal MIN_PAD = 0.01;

#ifdef QFLIPCLOCK_WIDGET
QFlipClock::QFlipClock(QWidget *parent)
    : QWidget(parent)
    , _transition(Transition_Flip)
#else
QFlipClock::QFlipClock(QMainWindow *parent)
        : QMainWindow(parent)
        , _transition(Transition_Flip)
#endif
{
    _theme = new Theme();
    reloadTheme();

    bool timeformat(true);

#ifdef Q_WS_MAEMO_5
    // handle 12/24 time settings
    is24 = new GConfItem("/apps/clock/time-format");
    timeformat = is24->value().toBool();

    QObject::connect(is24, SIGNAL(valueChanged()), this, SLOT(setHoursMode()));
#endif

    _hour = new Hour(this, _theme, timeformat);
    _hour->show();
    _minute = new Sheet(this, _theme);
    _minute->show();

    QTime t = QTime::currentTime();
    _hour->setNumber(t.hour());
    _minute->setNumber(t.minute());

    updateTime();
    startTimer();
}

QFlipClock::~QFlipClock()
{
#ifdef Q_WS_MAEMO_5
    delete is24;
#endif
    delete _hour;
    delete _minute;
}

void QFlipClock::updateTime(bool force)
{
    QTime t = QTime::currentTime();
    _hour->flipTo(t.hour(), force);
    _minute->flipTo(t.minute(), force);
}

void QFlipClock::setTransition(TransitionsEnum i)
{
    _hour->setTransition(i);
    _minute->setTransition(i);
    updateTime();
}

void QFlipClock::setHoursMode()
{
#ifdef Q_WS_MAEMO_5
    _hour->setHoursMode(is24->value().toBool());
    updateTime(true);
#endif
}

void QFlipClock::toggleShowBackground(bool value)
{
    _showBackground = value;
}

void QFlipClock::toggleZeroSign(bool value)
{
    _hour->setAddZero(value);
}

int QFlipClock::getHeight(int width)
{
    return width/2 * _aspect;
}

void QFlipClock::resizeEvent(QResizeEvent*)
{
    int pad = width() / 2 * ((_showBackground) ? _xpad : MIN_PAD);
    int gap = width() / 2 * _gap / 2;
    int digitsWidth = width() / 2 - gap - pad;
    int digitsHeight = digitsWidth * _aspect;
#ifdef QFLIPCLOCK_STANDALONE
    if (digitsHeight > MAX_HEIGHT)
    {
       digitsHeight = MAX_HEIGHT;
       digitsWidth = digitsHeight / _aspect;
       pad = (width() - digitsWidth*2 - gap);
    }
#endif

    int y = (height() - digitsHeight) / 2;

    _hour->resize(digitsWidth, digitsHeight);
    _hour->move(pad, y);

    _minute->resize(digitsWidth, digitsHeight);
    _minute->move(width() / 2 + gap, y);
}

void QFlipClock::timerEvent(QTimerEvent * event)
{
    if (event->timerId() == _ticker.timerId())
    {
        updateTime();
    }
}

void QFlipClock::updateTheme()
{
    _theme->updateTheme();
    reloadTheme();
    _hour->reloadTheme();
    _minute->reloadTheme();
    emit(resizeEvent(0));
    updateTime();
}

void QFlipClock::reloadTheme()
{
    _aspect = _theme->getAspect();
    _xpad = _theme->getXPad();
    _gap = _theme->getGap();
    _hasSemicolon = _theme->hasSemicolon();
}

/*
 * Redraw Background
 */
void QFlipClock::paintEvent(QPaintEvent *)
{
    QPainter p(this);
    drawBackground(&p);
    drawSemicolon(&p);
    p.end();
}

void QFlipClock::drawBackground(QPainter * p)
{
    if (!_showBackground)
        return;
    p->setBrush(_theme->getBgBrush());
    p->setPen(Qt::NoPen);
    p->drawRoundedRect(rect(), 25, 25);
}

void QFlipClock::drawSemicolon(QPainter * p)
{
    if (!_hasSemicolon)
        return;

    QPixmap pixmap = _theme->drawText(0.4, ":", rect(), true);
    p->drawPixmap(0, 0, pixmap);
}

void QFlipClock::stopTimer()
{
    if (_ticker.isActive())
        _ticker.stop();
}

void QFlipClock::startTimer()
{
    if (!_ticker.isActive())
        _ticker.start(1000, this);
}
