#ifndef TRANSFERITEM_H
#define TRANSFERITEM_H

#include "enums.h"
#include "serviceplugin.h"
#include "recaptchaplugin.h"
#include "decaptchaplugin.h"
#include <QObject>
#include <QUrl>
#include <QFile>
#include <QTime>
#include <QNetworkRequest>
#include <QProcess>

class Session;
class QNetworkAccessManager;
class QNetworkReply;
class QPixmap;

class TransferItem : public QObject
{
    Q_OBJECT

    Q_PROPERTY(QUrl webUrl
               READ webUrl
               CONSTANT)
    Q_PROPERTY(QString service
               READ service
               CONSTANT)
    Q_PROPERTY(QString iconName
               READ iconName
               CONSTANT)
    Q_PROPERTY(QString fileName
               READ fileName
               WRITE setFileName
               NOTIFY fileNameChanged)
    Q_PROPERTY(Transfers::Status status
               READ status
               WRITE setStatus
               NOTIFY statusChanged)
    Q_PROPERTY(QString statusText
               READ statusText
               NOTIFY statusChanged)
    Q_PROPERTY(QString statusInfo
               READ statusInfo
               NOTIFY statusInfoChanged)
    Q_PROPERTY(Transfers::Priority priority
               READ priority
               WRITE setPriority
               NOTIFY priorityChanged)
    Q_PROPERTY(QString priorityText
               READ priorityText
               NOTIFY priorityChanged)
    Q_PROPERTY(QString category
               READ category
               WRITE setCategory
               NOTIFY categoryChanged)
    Q_PROPERTY(int progress
               READ progress
               NOTIFY progressChanged)
    Q_PROPERTY(double speed
               READ speed
               NOTIFY speedChanged)
    Q_PROPERTY(qint64 size
               READ size
               NOTIFY sizeChanged)
    Q_PROPERTY(bool convertibleToAudio
               READ convertibleToAudio
               NOTIFY convertibleToAudioChanged)
    Q_PROPERTY(bool saveAsAudio
               READ saveAsAudio
               WRITE setSaveAsAudio
               NOTIFY saveAsAudioChanged)

public:
    explicit TransferItem(QObject *parent = 0);
    explicit TransferItem(QString id, const QUrl &webUrl, const QString &service, QString fileName = QString(), Transfers::Status status = Transfers::Queued, Transfers::Priority priority = Transfers::NormalPriority, QObject *parent = 0);
    ~TransferItem();
    inline Session* session() const { return m_session; }
    inline void setSession(Session *session) { m_session = session; }
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QString id() const { return m_id; }
    inline void setId(const QString &id) { m_id = id; }
    inline QUrl webUrl() const { return m_webUrl; }
    inline void setWebUrl(const QUrl &url) { m_webUrl = url; }
    inline QNetworkRequest downloadRequest() const { return m_downloadRequest; }
    inline void setDownloadRequest(const QNetworkRequest &request) { m_downloadRequest = request; }
    inline QByteArray downloadPostData() const { return m_downloadData; }
    inline void setDownloadPostData(const QByteArray &data) { m_downloadData = data; }
    inline QString service() const { return m_service; }
    inline void setService(const QString &service) { m_service = service; }
    inline QString category() const { return m_category; }
    inline QString fileName() const { return m_filename; }
    void setFileName(const QString &name);
    inline QString tempFileName() const { return m_tempFileName; }
    inline void setTempFileName(const QString &name) { m_tempFileName = name; }
    inline QString downloadPath() const { return m_downloadPath; }
    inline void setDownloadPath(const QString &path) { m_downloadPath = path; }
    inline Transfers::Status status() const { return m_status; }
    QString statusText() const;
    inline QString statusInfo() const { return m_statusInfo; }
    inline Transfers::Priority priority() const { return m_priority; }
    inline int progress() const { return m_progress; }
    inline void setProgress(int progress) { m_progress = progress; emit progressChanged(progress); }
    inline double speed() const { return m_speed; }
    inline void setSpeed(double speed) { m_speed = speed; emit speedChanged(speed); }
    void setStatus(Transfers::Status status);
    void setStatusInfo(const QString &info);
    QString priorityText() const;
    inline qint64 size() const { return m_size; }
    void setSize(qint64 size);
    inline ServicePlugin* servicePlugin() const { return m_servicePlugin; }
    void setServicePlugin(ServicePlugin *plugin);
    inline RecaptchaPlugin* recaptchaPlugin() const { return m_recaptchaPlugin; }
    void setRecaptchaPlugin(RecaptchaPlugin *plugin);
    inline DecaptchaPlugin* decaptchaPlugin() const { return m_decaptchaPlugin; }
    void setDecaptchaPlugin(DecaptchaPlugin *plugin);
    inline QString iconName() const { return !servicePlugin() ? QString() : servicePlugin()->iconName(); }
    inline uint row() const { return m_row; }
    void setRow(uint row);
    bool convertibleToAudio() const;
    inline bool saveAsAudio() const { return m_saveAsAudio; }

public slots:
    void startDownload();
    void setCategory(const QString &category);
    void setPriority(Transfers::Priority priority);
    void setSaveAsAudio(bool saveAsAudio);

private:
    void pauseDownload();
    void cancelDownload();
    void showCaptchaDialog(const QByteArray &imageData);
    void convertVideoToAudio();

private slots:
    void onServicePluginError(ServicePlugin::ErrorType errorType);
    void onServicePluginWaiting(int msecs);
    void onServicePluginStatusChanged(ServicePlugin::Status status);
    void onRecaptchaPluginError(RecaptchaPlugin::ErrorType errorType);
    void onDecaptchaPluginError(DecaptchaPlugin::ErrorType errorType);
    void performDownload(QNetworkRequest request, const QByteArray &data = QByteArray());
    void onDownloadMetadataChanged();
    void onDownloadReadyRead();
    void onDownloadProgressChanged(qint64 received, qint64 total);
    void onDownloadFinished();
    void onCaptchaRequested();
    void onCaptchaImageReceived(const QByteArray &imageData);
    void onCaptchaResponseReceived(const QString &response);
    void onCaptchaRejectedByUser();
    void onCaptchaIncorrect();
    void onConvertToAudioFinished(int exitCode, QProcess::ExitStatus exitStatus);

signals:
    void progressChanged(int progress);
    void speedChanged(double speed);
    void categoryChanged(const QString &category);
    void statusChanged(Transfers::Status status);
    void statusInfoChanged(const QString &info);
    void priorityChanged(Transfers::Priority priority);
    void sizeChanged(qint64 size);
    void fileNameChanged(const QString &fileName);
    void rowChanged(uint row);
    void convertibleToAudioChanged(bool convertible);
    void saveAsAudioChanged(bool saveAsAudio);

private:
    Session *m_session;
    ServicePlugin *m_servicePlugin;
    RecaptchaPlugin *m_recaptchaPlugin;
    DecaptchaPlugin *m_decaptchaPlugin;
    QNetworkAccessManager *m_nam;
    QNetworkReply *m_reply;
    QProcess *m_process;
    QFile m_file;
    QString m_id;
    QUrl m_webUrl;
    QNetworkRequest m_downloadRequest;
    QByteArray m_downloadData;
    QString m_service;
    QString m_category;
    QString m_filename;
    QString m_tempFileName;
    QString m_downloadPath;
    Transfers::Status m_status;
    QString m_statusText;
    QString m_statusInfo;
    Transfers::Priority m_priority;
    int m_progress;
    double m_speed;
    qint64 m_size;
    QString m_priorityText;
    int m_retries;
    QTime m_downloadTime;
    uint m_row;
    mutable bool m_convertible;
    mutable bool m_checkedIfConvertible;
    bool m_saveAsAudio;
};

#endif // TRANSFERITEM_H
