#include "database.h"
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSqlError>
#include <QVariant>
#include <QDir>
#include <QDesktopServices>
#include <QDebug>

Database::Database(QObject *parent) :
    QObject(parent)
{
    QString path(QDesktopServices::storageLocation(QDesktopServices::HomeLocation) + "/.QDL/");
    QDir dir;
    m_database = QSqlDatabase::addDatabase("QSQLITE");

    if (dir.mkpath(path)) {
        m_database.setDatabaseName(path + "qdl.sqlite");
    }
    else {
        m_database.setDatabaseName("qdl.sqlite");
    }

    this->initialize();
}

void Database::initialize() {
    if (m_database.open()) {
        m_database.exec("CREATE TABLE IF NOT EXISTS downloads (id TEXT UNIQUE, webUrl TEXT, service TEXT, downloadPath TEXT, tempFileName TEXT, fileName TEXT, category TEXT)");
        m_database.exec("ALTER TABLE downloads ADD COLUMN priority INTEGER");
        m_database.exec("ALTER TABLE downloads ADD COLUMN rowNumber INTEGER");
        m_database.exec("ALTER TABLE downloads ADD COLUMN saveAsAudio INTEGER");
        m_database.exec("CREATE TABLE IF NOT EXISTS categories (name TEXT UNIQUE, path TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS accounts (serviceName TEXT UNIQUE, username TEXT, password TEXT)");
        this->addCategory(tr("Default"), "");
    }
    else {
        qDebug() << m_database.lastError().text();
    }    
}

bool Database::storeDownload(QSharedPointer<TransferItem> transfer) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR IGNORE INTO downloads VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        query.addBindValue(transfer.data()->id());
        query.addBindValue(transfer.data()->webUrl().toString());
        query.addBindValue(transfer.data()->service());
        query.addBindValue(transfer.data()->downloadPath());
        query.addBindValue(transfer.data()->tempFileName());
        query.addBindValue(transfer.data()->fileName());
        query.addBindValue(transfer.data()->category());
        query.addBindValue(Transfers::Priority(transfer.data()->priority()));
        query.addBindValue(transfer.data()->row());
        query.addBindValue(transfer.data()->saveAsAudio());
        bool success = query.exec();
        m_database.close();

        return success;
    }    

    return false;
}

bool Database::removeStoredDownload(const QString &id) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM downloads WHERE id = ?");
        query.addBindValue(id);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

bool Database::updateStoredDownload(const QString &id, const QString &key, const QVariant &value) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare(QString("UPDATE downloads SET %1 = ? WHERE id = ?").arg(key));
        query.addBindValue(value);
        query.addBindValue(id);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

QList< QSharedPointer<TransferItem> > Database::getStoredDownloads() {
    QList< QSharedPointer<TransferItem> > transfers;

    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT * FROM downloads ORDER BY rowNumber ASC");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem);
                transfer.data()->setId(query.value(0).toString());
                transfer.data()->setWebUrl(query.value(1).toUrl());
                transfer.data()->setService(query.value(2).toString());
                transfer.data()->setDownloadPath(query.value(3).toString());
                transfer.data()->setTempFileName(query.value(4).toString());
                transfer.data()->setFileName(query.value(5).toString());
                transfer.data()->setCategory(query.value(6).toString());
                transfer.data()->setPriority(static_cast<Transfers::Priority>(query.value(7).toInt()));
                transfer.data()->setRow(query.value(8).toUInt());
                transfer.data()->setSaveAsAudio(query.value(9).toBool());
                transfers.append(transfer);
            }
        }

        m_database.close();
    }

    return transfers;
}

bool Database::addCategory(const QString &name, const QString &path) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO categories VALUES (?, ?)");
        query.addBindValue(name);
        query.addBindValue((path.endsWith('/')) || (path.isEmpty()) ? path : path + '/');
        bool success = query.exec();
        m_database.close();

        if (success) {
            emit categoriesChanged();
        }

        return success;
    }

    return false;
}

bool Database::removeCategory(const QString &name) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM categories WHERE name = ?");
        query.addBindValue(name);
        bool success = query.exec();
        m_database.close();

        if (success) {
            emit categoriesChanged();
        }

        return success;
    }

    return false;
}

QString Database::getCategoryPath(const QString &name) {
    QString path;

    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        QSqlRecord record;
        query.prepare("SELECT path FROM categories WHERE name = ?");
        query.addBindValue(name);
        query.exec();
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                path = query.value(0).toString();
            }
        }

        m_database.close();
    }

    return path;
}

QList< QPair<QString, QString> > Database::getCategories() {
    QList< QPair<QString, QString> > categories;

    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT * FROM categories ORDER BY name ASC");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                QPair<QString, QString> category;
                category.first = query.value(0).toString();
                category.second = query.value(1).toString();
                categories.append(category);
            }
        }

        m_database.close();
    }

    return categories;
}

bool Database::addAccount(const QString &serviceName, const QString &username, const QString &password) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO accounts VALUES (?, ?, ?)");
        query.addBindValue(serviceName);
        query.addBindValue(username);
        query.addBindValue(password);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

bool Database::removeAccount(const QString &serviceName) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM accounts WHERE serviceName = ?");
        query.addBindValue(serviceName);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

QPair<QString, QString> Database::getAccount(const QString &serviceName) {
    QPair<QString, QString> account;

    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        QSqlRecord record;
        query.prepare("SELECT username, password FROM accounts WHERE serviceName = ?");
        query.addBindValue(serviceName);
        query.exec();
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                account.first = query.value(0).toString();
                account.second = query.value(1).toString();
            }
        }

        m_database.close();
    }

    return account;
}
