#include "mainwindow.h"
#include "transfertitledelegate.h"
#include "transfercategorydelegate.h"
#include "transferprioritydelegate.h"
#include "transfersizedelegate.h"
#include "transferprogressdelegate.h"
#include "transferstatusdelegate.h"
#include "addurlsdialog.h"
#include "settingsdialog.h"
#include "aboutdialog.h"
#include "checkurlsdialog.h"
#include "treeview.h"
#include "../shared/session.h"
#include <QMenu>
#include <QNetworkAccessManager>
#include <QHeaderView>
#include <QToolBar>
#include <QLineEdit>
#include <QFileDialog>
#include <QDesktopServices>
#include <QFile>
#include <QMenuBar>

MainWindow::MainWindow(Session *session, QWidget *parent) :
    QMainWindow(parent),
    m_session(session),
    m_urlEdit(new QLineEdit(this)),
    m_view(new TreeView(this)),
    m_contextMenu(new QMenu(this)),
    m_saveAsAudioAction(m_contextMenu->addAction(tr("Save as audio file"), this, SLOT(setSaveTransferAsAudioFile()))),
    m_contextPauseResumeAction(m_contextMenu->addAction(tr("Pause"), this, SLOT(pauseResumeTransfer()))),
    m_contextCancelAction(m_contextMenu->addAction(tr("Cancel"), this, SLOT(cancelTransfer()))),
    m_priorityMenu(m_contextMenu->addMenu(tr("Priority"))),
    m_highPriorityAction(m_priorityMenu->addAction(tr("High"), this, SLOT(setTransferHighPriority()))),
    m_normalPriorityAction(m_priorityMenu->addAction(tr("Normal"), this, SLOT(setTransferNormalPriority()))),
    m_lowPriorityAction(m_priorityMenu->addAction(tr("Low"), this, SLOT(setTransferLowPriority()))),
    m_categoryMenu(m_contextMenu->addMenu(tr("Category"))),
    m_checkDialog(new CheckUrlsDialog(m_session, this))
{
    this->setWindowTitle(tr("QDL"));
    this->setCentralWidget(m_view);
    this->setMinimumSize(1100, 600);

    QToolBar *toolBar = new QToolBar(this);
    toolBar->setAllowedAreas(Qt::TopToolBarArea);
    QAction *addUrlsAction = toolBar->addAction(QIcon::fromTheme("list-add"), tr("Add URLs"), this, SLOT(showAddUrlsDialog()));
    QAction *textFileAction = toolBar->addAction(QIcon::fromTheme("document-import"), tr("Import URLs"), this, SLOT(showTextFileDialog()));
    QAction *settingsAction = toolBar->addAction(QIcon::fromTheme("preferences-system"), tr("Set preferences"), this, SLOT(showSettingsDialog()));
    addUrlsAction->setToolTip(tr("Add multiple URLs"));
    textFileAction->setToolTip(tr("Import URLs from a text file"));
    toolBar->addSeparator();
    QAction *resumeAction = toolBar->addAction(QIcon::fromTheme("media-playback-start"), "", this, SLOT(resumeSeletctedTransfers()));
    QAction *pauseAction = toolBar->addAction(QIcon::fromTheme("media-playback-pause"), "", this, SLOT(pauseSelectedTransfers()));
    QAction *cancelAction = toolBar->addAction(QIcon::fromTheme("dialog-close"), "", this, SLOT(cancelSelectedTransfers()));
    resumeAction->setToolTip(tr("Resume selected downloads"));
    pauseAction->setToolTip(tr("Pause selected downloads"));
    cancelAction->setToolTip(tr("Cancel selected downloads"));
    toolBar->addSeparator();
    toolBar->addWidget(m_urlEdit);
    this->addToolBar(toolBar);

    QMenu *urlsMenu = menuBar()->addMenu(tr("URLs"));
    urlsMenu->addAction(addUrlsAction);
    urlsMenu->addAction(textFileAction);

    QMenu *optionsMenu = menuBar()->addMenu(tr("Options"));
    optionsMenu->addAction(settingsAction);

    QMenu *helpMenu = menuBar()->addMenu(tr("Help"));
    helpMenu->addAction(tr("About QDL"), this, SLOT(showAboutDialog()));

    QActionGroup *priorityGroup = new QActionGroup(m_priorityMenu);
    priorityGroup->setExclusive(true);

    m_highPriorityAction->setActionGroup(priorityGroup);
    m_highPriorityAction->setCheckable(true);
    m_normalPriorityAction->setActionGroup(priorityGroup);
    m_normalPriorityAction->setCheckable(true);
    m_lowPriorityAction->setActionGroup(priorityGroup);
    m_lowPriorityAction->setCheckable(true);
    m_saveAsAudioAction->setCheckable(true);

    m_urlEdit->setPlaceholderText(tr("Add URL"));

    m_view->setModel(m_session->transferManager());
    m_view->setAlternatingRowColors(true);
    m_view->setRootIsDecorated(false);
    m_view->setDragEnabled(true);
    m_view->setDragDropMode(QTreeView::InternalMove);
    m_view->setDropIndicatorShown(true);
    m_view->header()->setDefaultAlignment(Qt::AlignCenter);
    m_view->setColumnWidth(0, 450);
    m_view->setColumnWidth(1, 120);
    m_view->setColumnWidth(2, 100);
    m_view->setColumnWidth(3, 100);
    m_view->setColumnWidth(4, 150);
    m_view->setColumnWidth(5, 150);
    m_view->setMouseTracking(true);
    m_view->setSelectionBehavior(QTreeView::SelectRows);
    m_view->setContextMenuPolicy(Qt::CustomContextMenu);
    m_view->setEditTriggers(QTreeView::NoEditTriggers);
    m_view->setExpandsOnDoubleClick(false);
    m_view->setItemsExpandable(false);
    m_view->setItemDelegateForColumn(0, new TransferTitleDelegate(m_view));
    m_view->setItemDelegateForColumn(1, new TransferCategoryDelegate(m_view));
    m_view->setItemDelegateForColumn(2, new TransferPriorityDelegate(m_view));
    m_view->setItemDelegateForColumn(3, new TransferSizeDelegate(m_view));
    m_view->setItemDelegateForColumn(4, new TransferProgressDelegate(m_view));
    m_view->setItemDelegateForColumn(5, new TransferStatusDelegate(m_view));

    this->connect(m_urlEdit, SIGNAL(returnPressed()), this, SLOT(addTransferFromUrlEdit()));
    this->connect(m_view, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    this->connect(m_view, SIGNAL(rowEntered(int)), m_session->transferManager(), SLOT(setRowEntered(int)));
    this->connect(m_view, SIGNAL(rowClicked(int)), m_session->transferManager(), SLOT(toggleRowSelected(int)));
    this->connect(m_session->database(), SIGNAL(categoriesChanged()), this, SLOT(setCategoryContextActions()));
}

void MainWindow::setCategoryContextActions() {    
    m_categoryMenu->clear();

    QActionGroup *group = new QActionGroup(m_categoryMenu);
    group->setExclusive(true);

    QList< QPair<QString, QString> > categories = m_session->database()->getCategories();

    while (!categories.isEmpty()) {
        QAction *action = m_categoryMenu->addAction(categories.takeFirst().first);
        action->setActionGroup(group);
        action->setCheckable(true);
        this->connect(action, SIGNAL(triggered(bool)), this, SLOT(setTransferCategory()));
    }
}

void MainWindow::showContextMenu(const QPoint &pos) {
    QModelIndex index = m_view->currentIndex();

    if (!index.isValid()) {
        return;
    }

    this->setCategoryContextActions();

    Transfers::Status status = static_cast<Transfers::Status>(index.data(TransferModel::StatusRole).toInt());

    switch (status) {
    case Transfers::Paused:
        m_contextPauseResumeAction->setText(tr("Resume"));
        break;
    case Transfers::Failed:
        m_contextPauseResumeAction->setText(tr("Resume"));
        break;
    default:
        m_contextPauseResumeAction->setText(tr("Pause"));
    }

    Transfers::Priority priority = static_cast<Transfers::Priority>(index.data(TransferModel::PriorityRole).toInt());

    switch (priority) {
    case Transfers::HighPriority:
        m_highPriorityAction->setChecked(true);
        break;
    case Transfers::LowPriority:
        m_lowPriorityAction->setChecked(true);
        break;
    default:
        m_normalPriorityAction->setChecked(true);
    }

    m_saveAsAudioAction->setEnabled(index.data(TransferModel::ConvertibleToAudioRole).toBool());
    m_saveAsAudioAction->setChecked((m_saveAsAudioAction->isEnabled()) && (index.data(TransferModel::SaveAsAudioRole).toBool()));

    QString category = index.data(TransferModel::CategoryRole).toString();
    bool found = false;
    int i = 0;

    while ((!found) && (i < m_categoryMenu->actions().size())) {
        found = m_categoryMenu->actions().at(i)->text() == category;
        i++;
    }

    if (found) {
        m_categoryMenu->actions().at(i - 1)->setChecked(true);
    }

    m_contextMenu->popup(mapToGlobal(pos), m_contextPauseResumeAction);
}

void MainWindow::pauseResumeTransfer() {
    QModelIndex index = m_view->currentIndex();

    if (!index.isValid()) {
        return;
    }

    Transfers::Status status = static_cast<Transfers::Status>(index.data(TransferModel::StatusRole).toInt());

    switch (status) {
    case Transfers::Paused:
        m_session->transferManager()->setData(index, Transfers::Queued, TransferModel::StatusRole);
        break;
    case Transfers::Failed:
        m_session->transferManager()->setData(index, Transfers::Queued, TransferModel::StatusRole);
        break;
    default:
        m_session->transferManager()->setData(index, Transfers::Paused, TransferModel::StatusRole);
    }
}

void MainWindow::cancelTransfer() {    
    m_session->transferManager()->setData(m_view->currentIndex(), Transfers::Cancelled, TransferModel::StatusRole);
}

void MainWindow::setTransferHighPriority() {
    m_session->transferManager()->setData(m_view->currentIndex(), Transfers::HighPriority, TransferModel::PriorityRole);
}

void MainWindow::setTransferNormalPriority() {
    m_session->transferManager()->setData(m_view->currentIndex(), Transfers::NormalPriority, TransferModel::PriorityRole);
}

void MainWindow::setTransferLowPriority() {
    m_session->transferManager()->setData(m_view->currentIndex(), Transfers::LowPriority, TransferModel::PriorityRole);
}

void MainWindow::setTransferCategory() {
    if (QAction *action = qobject_cast<QAction*>(sender())) {
        m_session->transferManager()->setData(m_view->currentIndex(), action->text(), TransferModel::CategoryRole);
    }
}

void MainWindow::setSaveTransferAsAudioFile() {
    if (QAction *action = qobject_cast<QAction*>(sender())) {
        m_session->transferManager()->setData(m_view->currentIndex(), action->isChecked(), TransferModel::SaveAsAudioRole);
    }
}

void MainWindow::resumeSeletctedTransfers() {
    m_session->transferManager()->resumeSelectedTransfers();
    m_session->transferManager()->clearSelectedRows();
}

void MainWindow::pauseSelectedTransfers() {
    m_session->transferManager()->pauseSelectedTransfers();
    m_session->transferManager()->clearSelectedRows();
}

void MainWindow::cancelSelectedTransfers() {
    m_session->transferManager()->cancelSelectedTransfers();
    m_session->transferManager()->clearSelectedRows();
}

void MainWindow::addTransferFromUrlEdit() {
    if (!m_urlEdit->text().isEmpty()) {
        m_session->urlChecker()->addUrlToQueue(QUrl::fromUserInput(m_urlEdit->text()));
        m_urlEdit->clear();
    }
}

void MainWindow::showAddUrlsDialog() {
    AddUrlsDialog *dialog = new AddUrlsDialog(m_session, this);
    dialog->open();
}

void MainWindow::showTextFileDialog() {
    QString filePath = QFileDialog::getOpenFileName(this, tr("Choose text file"), QDesktopServices::storageLocation(QDesktopServices::HomeLocation), "*.txt");

    if (!filePath.isEmpty()) {
        m_session->urlChecker()->importUrlsFromTextFile(filePath);
    }
}

void MainWindow::showSettingsDialog() {
    SettingsDialog *dialog = new SettingsDialog(m_session, this);
    dialog->open();
}

void MainWindow::showAboutDialog() {
    AboutDialog *dialog = new AboutDialog(this);
    dialog->open();
}
