#include "transferitem.h"
#include "utils.h"
#include "captchadialog.h"
#include "qplatformdefs.h"
#include "session.h"
#ifdef MEEGO_EDITION_HARMATTAN
#include "../harmattan/qmlutils.h"
#include "../harmattan/captchaimageprovider.h"
#include <QDeclarativeView>
#include <QDeclarativeEngine>
#include <QDeclarativeContext>
#include <QGraphicsObject>
#endif
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QDir>
#include <QRegExp>
#include <QPixmap>
#include <QDebug>

#define ILLEGAL_CHARS "[\"@&~=\\/:?#!|<>*^]"

static const QString VideoSuffixes("mp4:flv:avi:divx:mpg:mpeg:mpeg2:mpeg4:ts:mkv:wmv:xvid:mov");
static const QString ArchiveSuffixes("rar:zip:tar:gz");

TransferItem::TransferItem(QObject *parent) :
    QObject(parent),
    m_session(0),
    m_servicePlugin(0),
    m_recaptchaPlugin(0),
    m_decaptchaPlugin(0),
    m_nam(0),
    m_reply(0),
    m_process(0),
    m_status(Transfers::Queued),
    m_priority(Transfers::NormalPriority),
    m_progress(0),
    m_speed(0.0),
    m_size(0),
    m_retries(0),
    m_row(0),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_saveAsAudio(false)
{
}

TransferItem::TransferItem(QString id, const QUrl &webUrl, const QString &service, QString fileName, Transfers::Status status, Transfers::Priority priority, QObject *parent) :
    QObject(parent),
    m_session(0),
    m_servicePlugin(0),
    m_recaptchaPlugin(0),
    m_decaptchaPlugin(0),
    m_nam(0),
    m_reply(0),
    m_process(0),
    m_id(id),
    m_webUrl(webUrl),
    m_service(service),
    m_filename(fileName.replace(QRegExp(ILLEGAL_CHARS), "_")),
    m_tempFileName(id.replace(QRegExp(ILLEGAL_CHARS), "_")),
    m_status(status),
    m_priority(priority),
    m_progress(0),
    m_speed(0.0),
    m_size(0),
    m_retries(0),
    m_row(0),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_saveAsAudio(false)
{
}

TransferItem::~TransferItem() {
    if (m_reply) {
        delete m_reply;
    }
}

void TransferItem::setCategory(const QString &category) {
    if (category != this->category()) {
        m_category = category;
        emit categoryChanged(category);
    }
}

void TransferItem::setFileName(const QString &name) {
    if (name != this->fileName()) {
        m_filename = name;

        if (this->status() != Transfers::Converting) {
            emit fileNameChanged(name);
            emit convertibleToAudioChanged(this->convertibleToAudio());
        }
    }
}

void TransferItem::setPriority(Transfers::Priority priority) {
    if (priority != this->priority()) {
        m_priority = priority;
        emit priorityChanged(priority);
    }
}

void TransferItem::setSize(qint64 size) {
    if (size != this->size()) {
        m_size = size;
        emit sizeChanged(size);
    }
}

void TransferItem::setRow(uint row) {
    if (row != this->row()) {
        m_row = row;
        emit rowChanged(row);
    }
}

bool TransferItem::convertibleToAudio() const {
    if (!m_checkedIfConvertible) {
        m_checkedIfConvertible = true;
        m_convertible = (VideoSuffixes.contains(this->fileName().section('.', -1), Qt::CaseInsensitive) && (QFile::exists("/usr/bin/ffmpeg"))); // Should probably check other locations
    }

    return m_convertible;
}

void TransferItem::setSaveAsAudio(bool saveAsAudio) {
    if (saveAsAudio != this->saveAsAudio()) {
        m_saveAsAudio = saveAsAudio;
        emit saveAsAudioChanged(saveAsAudio);
    }
}

void TransferItem::setServicePlugin(ServicePlugin *plugin) {
    if (this->servicePlugin()) {
        delete m_servicePlugin;
    }

    m_servicePlugin = plugin;

    if (this->servicePlugin()) {
        this->servicePlugin()->setParent(this);
        this->connect(this->servicePlugin(), SIGNAL(statusChanged(ServicePlugin::Status)), this, SLOT(onServicePluginStatusChanged(ServicePlugin::Status)));
        this->connect(this->servicePlugin(), SIGNAL(waiting(int)), this, SLOT(onServicePluginWaiting(int)));
        this->connect(this->servicePlugin(), SIGNAL(error(ServicePlugin::ErrorType)), this, SLOT(onServicePluginError(ServicePlugin::ErrorType)));
        this->connect(this->servicePlugin(), SIGNAL(downloadRequestReady(QNetworkRequest,QByteArray)), this, SLOT(performDownload(QNetworkRequest,QByteArray)));
    }
}

void TransferItem::setRecaptchaPlugin(RecaptchaPlugin *plugin) {
    if (this->recaptchaPlugin()) {
        delete m_recaptchaPlugin;
    }

    m_recaptchaPlugin = plugin;

    if (this->recaptchaPlugin()) {
        this->recaptchaPlugin()->setParent(this);
        this->connect(this->recaptchaPlugin(), SIGNAL(gotCaptcha(QByteArray)), this, SLOT(onCaptchaImageReceived(QByteArray)));
        this->connect(this->recaptchaPlugin(), SIGNAL(error(RecaptchaPlugin::ErrorType)), this, SLOT(onRecaptchaPluginError(RecaptchaPlugin::ErrorType)));
    }
}

void TransferItem::setDecaptchaPlugin(DecaptchaPlugin *plugin) {
    if (this->decaptchaPlugin()) {
        delete m_decaptchaPlugin;
    }

    m_decaptchaPlugin = plugin;

    if (this->decaptchaPlugin()) {
        this->decaptchaPlugin()->setParent(this);
        this->connect(this->decaptchaPlugin(), SIGNAL(gotCaptchaResponse(QString)), this, SLOT(onCaptchaResponseReceived(QString)));
        this->connect(this->decaptchaPlugin(), SIGNAL(error(DecaptchaPlugin::ErrorType)), this, SLOT(onDecaptchaPluginError(DecaptchaPlugin::ErrorType)));
    }
}

QString TransferItem::statusText() const {
    switch (status()) {
    case Transfers::Queued:
        return tr("Queued");
    case Transfers::Paused:
        return tr("Paused");
    case Transfers::Connecting:
        return tr("Connecting");
    case Transfers::ShortWait:
        return tr("Waiting");
    case Transfers::LongWait:
        return tr("Waiting");
    case Transfers::Downloading:
        return tr("Downloading");
    case Transfers::Cancelled:
        return tr("Cancelled");
    case Transfers::Failed:
        return tr("Failed");
    case Transfers::Completed:
        return tr("Completed");
    case Transfers::Converting:
        return tr("Converting");
    default:
        return tr("Unknown");
    }
}

QString TransferItem::priorityText() const {
    switch (priority()) {
    case Transfers::HighPriority:
        return tr("High");
    case Transfers::NormalPriority:
        return tr("Normal");
    case Transfers::LowPriority:
        return tr("Low");
    default:
        return QString();
    }
}

void TransferItem::setStatus(Transfers::Status status) {
    if (status != this->status()) {
        m_status = status;

        if (status == Transfers::Paused) {
            this->pauseDownload();
        }
        else if (status == Transfers::Cancelled) {
            this->cancelDownload();
        }

        emit statusChanged(status);
    }
}

void TransferItem::setStatusInfo(const QString &info) {
    if (info != this->statusInfo()) {
        m_statusInfo = info;
        emit statusInfoChanged(info);
    }
}

void TransferItem::startDownload() {
    m_retries = 0;

    if (this->servicePlugin()) {
        this->servicePlugin()->getDownloadUrl(this->webUrl());
    }
    else {
        this->performDownload(QNetworkRequest(this->webUrl()));
    }
}

void TransferItem::performDownload(QNetworkRequest request, const QByteArray &data) {
    this->setDownloadRequest(request);
    this->setDownloadPostData(data);
    this->setStatus(Transfers::Downloading);

    QDir dir;
    dir.mkpath(this->downloadPath());

    m_file.setFileName(this->downloadPath() + this->tempFileName());

    if (m_file.exists()) {
        if (!m_file.open(QIODevice::Append)) {
            this->setStatusInfo(tr("Cannot write to file"));
            this->setStatus(Transfers::Failed);
            return;
        }
    }
    else if (!m_file.open(QIODevice::WriteOnly)) {
        this->setStatusInfo(tr("Cannot write to file"));
        this->setStatus(Transfers::Failed);
        return;
    }

    if (m_file.size() > 0) {
        request.setRawHeader("Range", "bytes=" + QByteArray::number(m_file.size()) + "-"); // Set 'Range' header if resuming a download
    }

    if (!data.isEmpty()) {
        m_reply = this->networkAccessManager()->post(request, data);
    }
    else {
        m_reply = this->networkAccessManager()->get(request);
    }

    m_downloadTime.start();

    this->connect(m_reply, SIGNAL(metaDataChanged()), this, SLOT(onDownloadMetadataChanged()));
    this->connect(m_reply, SIGNAL(downloadProgress(qint64,qint64)), this, SLOT(onDownloadProgressChanged(qint64,qint64)));
    this->connect(m_reply, SIGNAL(readyRead()), this, SLOT(onDownloadReadyRead()));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onDownloadFinished()));
}

void TransferItem::pauseDownload() {
    if (m_reply) {
        m_reply->abort();
    }
}

void TransferItem::cancelDownload() {
    if (m_reply) {
        m_reply->abort();
    }

    if (m_file.exists()) {
        m_file.remove();
    }
}

void TransferItem::onDownloadMetadataChanged() {
    if (m_reply) {
        qint64 size = m_reply->rawHeader("Content-Length").toLongLong();
        QString fileName = QString(m_reply->rawHeader("Content-Disposition")).section("filename=", -1).remove(QRegExp("\'|\"")).replace(QRegExp(ILLEGAL_CHARS), "_");

        if (size > 0) {
            this->setSize(size + m_file.size());
        }

        if ((fileName.contains('.')) && (fileName != this->fileName())) {
            this->setFileName(fileName);
        }
    }
}

void TransferItem::onDownloadProgressChanged(qint64 received, qint64 total) {
    if (received) {
        qint64 size = m_file.size();
        this->setProgress((size + received) * 100 / (size + total));
        this->setSpeed(received * 1000 / m_downloadTime.elapsed());
    }
}

void TransferItem::onDownloadReadyRead() {
    if (m_reply) {
        m_file.write(m_reply->readAll());
    }
}

void TransferItem::onDownloadFinished() {
    if (!m_reply) {
        return;
    }

    m_file.close();
    QUrl redirect = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (!redirect.isEmpty()) {
        this->performDownload(QNetworkRequest(redirect));
    }
    else if (m_reply->error()) {
        if (m_reply->error() != QNetworkReply::OperationCanceledError) {
            if (m_retries < 3) {
                m_retries++;
                this->performDownload(this->downloadRequest(), this->downloadPostData());
            }
            else {
                QString statusText = m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                this->setStatusInfo(statusText);
                this->setStatus(Transfers::Failed);
            }
        }
    }
    else if ((this->convertibleToAudio()) && (this->saveAsAudio())) {
        this->convertVideoToAudio();
    }
    else {
        this->setStatus(Transfers::Completed);
    }
}

void TransferItem::convertVideoToAudio() {
    this->setStatus(Transfers::Converting);

    if (!m_process) {
        m_process = new QProcess(this);
        this->connect(m_process, SIGNAL(finished(int,QProcess::ExitStatus)), this, SLOT(onConvertToAudioFinished(int,QProcess::ExitStatus)));
    }

    m_process->setWorkingDirectory(this->downloadPath());
    m_process->start(QString("ffmpeg -i \"%1\" -acodec copy -y -vn \"%2\"").arg(this->tempFileName()).arg(this->tempFileName() + ".m4a"));
}

void TransferItem::onConvertToAudioFinished(int exitCode, QProcess::ExitStatus exitStatus) {
    if ((exitCode == 0) && (exitStatus == QProcess::NormalExit)) {
        QFile::remove(this->downloadPath() + this->tempFileName());
        this->setTempFileName(this->tempFileName() + ".m4a");
        this->setFileName(this->fileName().section('.', 0, -2) + ".m4a");
    }
    else {
        qWarning() << m_process->readAllStandardError();
    }

    this->setStatus(Transfers::Completed);
}

void TransferItem::onServicePluginError(ServicePlugin::ErrorType errorType) {
    switch (errorType) {
    case ServicePlugin::UrlError:
        this->setStatusInfo(tr("Cannot retreive download url"));
        break;
    case ServicePlugin::CaptchaError:
        this->onCaptchaIncorrect();
        return;
    case ServicePlugin::Unauthorised:
        this->setStatusInfo(tr("Service authorisation error"));
        break;
    case ServicePlugin::BadRequest:
        this->setStatusInfo(tr("Bad request"));
        break;
    case ServicePlugin::NotFound:
        this->setStatusInfo(tr("File not available for download"));
        break;
    case ServicePlugin::TrafficExceeded:
        this->setStatusInfo(tr("Maximum download traffic exceeded"));
        break;
    case ServicePlugin::ServiceUnavailable:
        this->setStatusInfo(tr("Service unavailable"));
        break;
    case ServicePlugin::NetworkError:
        this->setStatusInfo(tr("Network error"));
        break;
    default:
        this->setStatusInfo(this->servicePlugin()->errorString().isEmpty() ? tr("Cannot retreive download url") : this->servicePlugin()->errorString());
    }

    this->setStatus(Transfers::Failed);
}

void TransferItem::onServicePluginWaiting(int msecs) {
    this->setStatusInfo(Utils::durationFromMSecs(msecs));
}

void TransferItem::onServicePluginStatusChanged(ServicePlugin::Status status) {
    if (this->status() == Transfers::Paused) {
        return;
    }

    switch (status) {
    case ServicePlugin::Connecting:
        this->setStatus(Transfers::Connecting);
        break;
    case ServicePlugin::ShortWait:
        this->setStatus(Transfers::ShortWait);
        break;
    case ServicePlugin::LongWait:
        this->setStatus(Transfers::LongWait);
        break;
    case ServicePlugin::CaptchaRequired:
        this->onCaptchaRequested();
        break;
    case ServicePlugin::Ready:
        this->setStatus(Transfers::Queued);
        break;
    }
}

void TransferItem::onRecaptchaPluginError(RecaptchaPlugin::ErrorType errorType) {
    switch (errorType) {
    case RecaptchaPlugin::NetworkError:
        this->setStatusInfo(tr("NetworkError"));
        break;
    case RecaptchaPlugin::Unauthorised:
        this->setStatusInfo(tr("Invalid captcha key"));
        break;
    case RecaptchaPlugin::ServiceUnavailable:
        this->setStatusInfo(tr("Recaptcha service unavailable"));
        break;
    case RecaptchaPlugin::InternalError:
        this->setStatusInfo(tr("Internal server error at recaptcha service"));
        break;
    default:
        this->setStatusInfo(tr("Cannot retrieve captcha challenge"));
    }

    this->setStatus(Transfers::Failed);
}

void TransferItem::onDecaptchaPluginError(DecaptchaPlugin::ErrorType errorType) {
    switch (errorType) {
    case DecaptchaPlugin::CaptchaNotFound:
        this->setStatusInfo(tr("Invalid captcha ID"));
        break;
    case DecaptchaPlugin::CaptchaUnsolved:
        this->onCaptchaIncorrect();
        return;
    case DecaptchaPlugin::ServiceUnavailable:
        this->setStatusInfo(tr("Decaptcha service unavailable"));
        break;
    case DecaptchaPlugin::Unauthorised:
        this->setStatusInfo(tr("Decaptcha service authorisation error"));
        break;
    case DecaptchaPlugin::BadRequest:
        this->setStatusInfo(tr("Decaptcha bad request"));
        break;
    case DecaptchaPlugin::InternalError:
        this->setStatusInfo(tr("Internal server error at decaptcha service"));
        break;
    case DecaptchaPlugin::NetworkError:
        this->setStatusInfo(tr("Network error"));
        break;
    default:
        this->setStatusInfo(tr("Decaptcha service error"));
    }

    this->setStatus(Transfers::Failed);
}

void TransferItem::onCaptchaRequested() {
    if (this->status() == Transfers::Paused) {
        return;
    }

    if (this->recaptchaPlugin()) {
        if (!this->servicePlugin()->recaptchaKey().isEmpty()) {
            this->setStatusInfo(tr("Requesting captcha"));
            this->recaptchaPlugin()->getCaptcha(this->servicePlugin()->recaptchaKey());
        }
        else {
            this->setStatusInfo(tr("No recaptcha key available"));
            this->setStatus(Transfers::Failed);
        }
    }
    else {
        this->setStatusInfo(tr("No recaptcha plugin"));
        this->setStatus(Transfers::Failed);
    }
}

void TransferItem::onCaptchaImageReceived(const QByteArray &imageData) {
    if (!this->session()) {
        this->showCaptchaDialog(imageData);
    }
    else {
        QString decaptchaService = this->session()->settings()->decaptchaService();

        if ((this->decaptchaPlugin()) && (this->decaptchaPlugin()->serviceName() == decaptchaService)) {
            if (!this->decaptchaPlugin()->password().isEmpty()) {
                this->setStatusInfo(tr("Requesting captcha response"));
                this->decaptchaPlugin()->getCaptcha(imageData);
            }
        }
        else if (!decaptchaService.isEmpty()) {
            this->setDecaptchaPlugin(this->session()->pluginManager()->createDecaptchaPlugin(decaptchaService));

            if (!this->decaptchaPlugin()->password().isEmpty()) {
                this->setStatusInfo(tr("Requesting captcha response"));
                this->decaptchaPlugin()->getCaptcha(imageData);
            }
        }
        else {
            this->showCaptchaDialog(imageData);
        }
    }
}

void TransferItem::onCaptchaResponseReceived(const QString &response) {
    if ((this->servicePlugin()) && (this->recaptchaPlugin())) {
        this->setStatusInfo(tr("Submitting captcha response"));
        this->servicePlugin()->submitCaptchaResponse(this->recaptchaPlugin()->challenge(), response);
    }
}

void TransferItem::onCaptchaIncorrect() {
    if ((this->decaptchaPlugin()) && (!this->decaptchaPlugin()->captchaId().isEmpty())) {
        this->decaptchaPlugin()->reportIncorrectCaptcha(this->decaptchaPlugin()->captchaId());
    }

    this->onCaptchaRequested();
}

void TransferItem::onCaptchaRejectedByUser() {
    this->setStatusInfo(tr("No captcha response"));
    this->setStatus(Transfers::Failed);
}

void TransferItem::showCaptchaDialog(const QByteArray &imageData) {
#ifdef MEEGO_EDITION_HARMATTAN
    this->setStatusInfo(tr("Awaiting captcha response"));
    QDeclarativeView *view = new QDeclarativeView;
    view->engine()->addImageProvider(QString("captcha"), new CaptchaImageProvider);
    view->rootContext()->setContextProperty("Utils", new QmlUtils(view));
    view->setSource(QUrl::fromLocalFile("/opt/qdl/qml/harmattan/CaptchaDialog.qml"));
    QGraphicsObject *dialog = view->rootObject();
    dialog->setProperty("captchaImage", QString("image://captcha/" + imageData.toBase64()));
    dialog->setProperty("timeOut", 120);
    this->connect(dialog, SIGNAL(rejected()), view, SLOT(deleteLater()));
    this->connect(dialog, SIGNAL(rejected()), this, SLOT(onCaptchaRejectedByUser()));
    this->connect(dialog, SIGNAL(gotCaptchaResponse(QString)), view, SLOT(deleteLater()));
    this->connect(dialog, SIGNAL(gotCaptchaResponse(QString)), this, SLOT(onCaptchaResponseReceived(QString)));
    view->showFullScreen();
#else
    QPixmap image;

    if (image.loadFromData(imageData)) {
        this->setStatusInfo(tr("Awaiting captcha response"));
        CaptchaDialog *dialog = new CaptchaDialog;
        dialog->setCaptchaImage(image);
        dialog->setTimeout(120);
        dialog->open();
        this->connect(dialog, SIGNAL(gotCaptchaResponse(QString)), this, SLOT(onCaptchaResponseReceived(QString)));
        this->connect(dialog, SIGNAL(rejected()), this, SLOT(onCaptchaRejectedByUser()));
    }
    else {
        this->setStatusInfo(tr("Captcha image error"));
        this->setStatus(Transfers::Failed);
    }
#endif
}
