#include "youtube.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QStringList>
#include <QRegExp>
#include <QSettings>
#include <QDomDocument>
#include <QDomElement>

YouTube::YouTube(QObject *parent) :
    ServicePlugin(parent)
{
    m_formatList << 37 << 22 << 35 << 34 << 18;
}

QRegExp YouTube::urlPattern() const {
    return QRegExp("(http://(www.|m.|)youtube.com/(v/|.+)(v=|list=|)|http://youtu.be/)", Qt::CaseInsensitive);
}

bool YouTube::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void YouTube::checkUrl(const QUrl &webUrl) {
    QString urlString = webUrl.toString();
    QString id(urlString.section(QRegExp("v=|list=|/"), -1).section(QRegExp("&|\\?"), 0, 0));
    QUrl url;

    if (urlString.contains("list=")) {
        // QUrl::hasQueryItem() does not work :/
        url.setUrl("https://gdata.youtube.com/feeds/api/playlists/" + id);
        url.addQueryItem("fields", "openSearch:totalResults,openSearch:startIndex,entry(content,media:group(media:title))");
        url.addQueryItem("max-results", "50");
    }
    else {
        url.setUrl("https://gdata.youtube.com/feeds/api/videos/" + id);
        url.addQueryItem("fields", "content,media:group(media:title)");
    }

    url.addQueryItem("v", "2.1");
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void YouTube::checkPlaylistVideoUrls(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void YouTube::checkUrlIsValid() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomNodeList entries = doc.elementsByTagName("entry");

    if (entries.isEmpty()) {
        emit urlChecked(false);
    }
    else {
        for (int i = 0; i < entries.count(); i++) {
            QDomNode entry = entries.at(i);
            QUrl url(entry.firstChildElement("content").attribute("src"));
            QString title = entry.firstChildElement("media:group").firstChildElement("media:title").text().trimmed();
            emit urlChecked((url.isValid()) && (!title.isEmpty()), url, this->serviceName(), title + ".mp4", i == (entries.count() - 1));
        }

        QDomElement resultsElement = doc.namedItem("feed").firstChildElement("openSearch:totalResults");
        QDomElement startElement = doc.namedItem("feed").firstChildElement("openSearch:startIndex");

        if ((!resultsElement.isNull()) && (!startElement.isNull())) {
            int totalResults = resultsElement.text().toInt();
            int startIndex = startElement.text().toInt();

            if (totalResults > (startIndex + entries.count())) {
                QString urlString = reply->request().url().toString();
                QUrl playlistUrl(urlString.section("&start-index=", 0, 0));
                playlistUrl.addQueryItem("start-index", QString::number(startIndex + entries.count()));
                this->checkPlaylistVideoUrls(playlistUrl);
            }
        }
    }

    reply->deleteLater();
}

void YouTube::getDownloadUrl(const QUrl &webUrl) {
    emit statusChanged(Connecting);
    QString id(webUrl.toString().section(QRegExp("v=|/"), -1).section(QRegExp("&|\\?"), 0, 0));
    QUrl url("http://www.youtube.com/get_video_info?&video_id=" + id + "&el=detailpage&ps=default&eurl=&gl=US&hl=en");
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVideoPage()));
}

void YouTube::parseVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QMap<int, QString> formats;
    QString response(QByteArray::fromPercentEncoding(reply->readAll()));

    if (!response.contains("url_encoded_fmt_stream_map=")) {
        emit error(UrlError);
    }
    else {
        response = response.section("url_encoded_fmt_stream_map=", 1, 1);
        response = QByteArray::fromPercentEncoding(response.toUtf8()).replace("%2C", ",").replace("sig=", "signature=").replace(",signature=", "&signature=");
        QStringList urlStrings = response.split("url=", QString::SkipEmptyParts);
        QStringList sigs = response.split("signature=");
        QStringList signatures;

        for (int i = 1; i < sigs.size(); i++) {
            signatures << sigs.at(i).section(QRegExp("(&|,)"), 0, 0);
        }

        foreach (QString urlString, urlStrings) {
            if (urlString.startsWith("http")) {
                QUrl url(urlString.section(QRegExp("(,|&|)itag="), 0, 1));
                int key = url.queryItemValue("itag").toInt();

                if (key > 0) {
                    if (!signatures.isEmpty()) {
                        url.removeQueryItem("signature");
                        url.addQueryItem("signature", signatures.takeFirst());
                    }

                    formats[key] = url.toString();
                }
            }
        }

        QString videoUrl;
        int format = QSettings("QDL", "QDL").value("YouTube/videoFormat", 18).toInt();
        int index = m_formatList.indexOf(format);

        while ((videoUrl.isEmpty()) && (index < m_formatList.size())) {
            format = m_formatList.at(index);
            videoUrl = formats.value(format, "");
            index++;

        }

        if (!videoUrl.startsWith("http")) {
            emit error(UrlError);
        }
        else {
            QNetworkRequest request;
            request.setUrl(QUrl(videoUrl));
            emit downloadRequestReady(request);
        }
    }

    reply->deleteLater();
}

Q_EXPORT_PLUGIN2(youtube, YouTube)
