#include "uloz.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QRegExp>

Uloz::Uloz(QObject *parent) :
    ServicePlugin(parent)
{
}

QRegExp Uloz::urlPattern() const {
    return QRegExp("http(s|)://(www.|)uloz.to/\\w+/[-\\w]+", Qt::CaseInsensitive);
}

bool Uloz::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void Uloz::login(const QString &username, const QString &password) {
    QString data = QString("username=%1&password=%2").arg(username).arg(password);
    QUrl url("http://uloz.to/login");
    url.addQueryItem("do", "loginForm-submit");
    QNetworkRequest request(url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkLogin()));
}

void Uloz::checkLogin() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    emit loggedIn((statusCode == 200) || (statusCode == 302));

    reply->deleteLater();
}

void Uloz::checkUrl(const QUrl &webUrl) {
    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void Uloz::checkUrlIsValid() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();
    QRegExp re("http://\\w+.uloz.to/Ps;Hs;fid=\\d+[^'\"]+");

    if ((!redirect.isEmpty()) && (re.indexIn(redirect) == -1)) {
        this->checkUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());
        QString fileName = response.section("<title>", 1).section(" | Ulo", 0, 0);

        if (fileName.isEmpty()) {
            emit urlChecked(false);
        }
        else {
            emit urlChecked(true, reply->request().url(), this->serviceName(), fileName);
        }
    }

    reply->deleteLater();
}

void Uloz::getDownloadUrl(const QUrl &webUrl) {
    emit statusChanged(Connecting);
    m_url = webUrl;
    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(onWebPageDownloaded()));
}

void Uloz::onWebPageDownloaded() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QRegExp re("http://\\w+.uloz.to/Ps;Hs;fid=\\d+[^'\"]+");
    QString redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

    if (re.indexIn(redirect) == 0) {
        QNetworkRequest request;
        request.setUrl(QUrl(re.cap()));
        emit downloadRequestReady(request);
    }
    else if (!redirect.isEmpty()) {
        this->getDownloadUrl(QUrl(redirect));
    }
    else {
        QString response(reply->readAll());

        if (re.indexIn(response) >= 0) {
            QNetworkRequest request;
            request.setUrl(QUrl(re.cap()));
            emit downloadRequestReady(request);
        }
        else {
            m_captchaId = response.section("captcha_id\" value=\"", 1, 1).section('"', 0, 0);
            m_captchaKey = response.section("captcha_key\" value=\"", 1, 1).section('"', 0, 0);
            m_timestamp = response.section("frmfreeDownloadForm-ts\" value=\"", 1, 1).section('"', 0, 0);
            m_cid = response.section("frmfreeDownloadForm-cid\" value=\"", 1, 1).section('"', 0, 0);
            m_sign = response.section("frmfreeDownloadForm-sign\" value=\"", 1, 1).section('"', 0, 0);

            if ((m_captchaId.isEmpty()) || (m_captchaKey.isEmpty()) || (m_timestamp.isEmpty()) || (m_cid.isEmpty()) || (m_sign.isEmpty())) {
                emit error(UnknownError);
            }
            else {
                emit statusChanged(CaptchaRequired);
            }
        }
    }

    reply->deleteLater();
}

void Uloz::submitCaptchaResponse(const QString &challenge, const QString &response) {
    QString data = QString("captcha_value=%1&captcha_id=%2&captcha_key=%3&ts=%4&cid=%5&sign=%6").arg(response).arg(challenge).arg(m_captchaKey).arg(m_timestamp).arg(m_cid).arg(m_sign);
    m_url.addQueryItem("do", "downloadDialog-freeDownloadForm-submit");
    QNetworkRequest request(m_url);
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->post(request, data.toUtf8());
    this->connect(reply, SIGNAL(finished()), this, SLOT(onCaptchaSubmitted()));
}

void Uloz::onCaptchaSubmitted() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (redirect.isValid()) {
        emit downloadRequestReady(QNetworkRequest(redirect));
    }
    else {
        QString response(reply->readAll());

        if (response.contains("Text je ops")) {
            emit error(CaptchaError);
        }
        else {
            emit error(UnknownError);
        }
    }

    reply->deleteLater();
}

Q_EXPORT_PLUGIN2(uloz, Uloz)
