#include "rapidshare.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QStringList>
#include <QTimer>

RapidShare::RapidShare(QObject *parent) :
    ServicePlugin(parent),
    m_waitTimer(new QTimer(this))
{
    this->connect(m_waitTimer, SIGNAL(timeout()), this, SLOT(updateWaitTime()));
}

QRegExp RapidShare::urlPattern() const {
    return QRegExp("http(s|)://(www.|)rapidshare.com/files/\\d+", Qt::CaseInsensitive);
}

bool RapidShare::urlSupported(const QUrl &url) const {
    return this->urlPattern().indexIn(url.toString()) == 0;
}

void RapidShare::login(const QString &username, const QString &password) {
    QUrl url("https://api.rapidshare.com/cgi-bin/rsapi.cgi");
    url.addQueryItem("sub", "getaccountdetails");
    url.addQueryItem("login", username);
    url.addQueryItem("password", password);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkLogin()));
}

void RapidShare::checkLogin() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    emit loggedIn(statusCode == 200);

    reply->deleteLater();
}

void RapidShare::checkUrl(const QUrl &webUrl) {
    if (webUrl.toString().contains("#!download")) {
        QString fileName = QByteArray::fromPercentEncoding(webUrl.toString().section('|', 3, 3).toUtf8());
        emit urlChecked(true, webUrl, serviceName(), fileName);
        return;
    }

    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUrlIsValid()));
}

void RapidShare::checkUrlIsValid() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit urlChecked(false);
        return;
    }

    QString response(reply->readAll());
    QString errorString = response.section("ERROR:", 1, 1).section('(', 0, 0).trimmed();

    if ((!errorString.isEmpty()) && (!errorString.startsWith("File owner's public traffic exhausted"))) {
        emit urlChecked(false);
    }
    else {
        QString redirect = response.section("location=\"", -1).section('"', 0, 0);

        if (redirect.contains("#!download")) {
            QString fileName = QByteArray::fromPercentEncoding(redirect.section('|', 3, 3).toUtf8());
            emit urlChecked(true, reply->request().url(), this->serviceName(), fileName);
        }
        else {
            emit urlChecked(false);
        }
    }

    reply->deleteLater();
}

void RapidShare::getDownloadUrl(const QUrl &webUrl) {
    emit statusChanged(Connecting);
    if (webUrl.toString().contains("#!download")) {
        this->constructDownloadUrl(webUrl.toString());
        return;
    }

    QNetworkRequest request(webUrl);
    request.setRawHeader("Accept-Language", "en-GB,en-US;q=0.8,en;q=0.6");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkRedirect()));
}

void RapidShare::checkRedirect() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        emit error(NetworkError);
        return;
    }

    QString response(reply->readAll());

    if (response.startsWith("ERROR")) {
        QString errorString = response.section("ERROR:", 1, 1).section('(', 0, 0).trimmed();

        if (errorString.startsWith("File owner's public traffic exhausted")) {
            this->startWait(600000);
            this->connect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
        }
        else if (errorString.startsWith("File not found")) {
            emit error(NotFound);
        }
        else {
            emit error(UnknownError);
        }
    }
    else {
        QString redirect = response.section("location=\"", -1).section('"', 0, 0);

        if (redirect.contains("#!download")) {
            this->constructDownloadUrl(redirect);
        }
        else {
            emit error(UnknownError);
        }
    }

    reply->deleteLater();
}

void RapidShare::constructDownloadUrl(const QString &url) {
    QStringList split = url.split('|');

    if (split.size() > 3) {
        split.removeFirst();
        QString server = split.takeFirst();
        QString fileId = split.takeFirst();
        QString fileName = QByteArray::fromPercentEncoding(split.takeFirst().toUtf8());
        QNetworkRequest request;
        request.setUrl(QUrl(QString("http://rs%1.rapidshare.com/cgi-bin/rsapi.cgi?sub=download&fileid=%2&filename=%3").arg(server).arg(fileId).arg(fileName)));
        emit downloadRequestReady(request);
    }
    else {
        emit error(UrlError);
    }
}

void RapidShare::startWait(int msecs) {
    if (msecs > 30000) {
        emit statusChanged(LongWait);
    }
    else {
        emit statusChanged(ShortWait);
    }

    emit waiting(msecs);
    m_waitTime = msecs;
    m_waitTimer->start(1000);
}

void RapidShare::updateWaitTime() {
    m_waitTime -= m_waitTimer->interval();
    emit waiting(m_waitTime);

    if (m_waitTime <= 0) {
        m_waitTimer->stop();
        emit waitFinished();
    }
}

void RapidShare::onWaitFinished() {
    emit statusChanged(Ready);
    this->disconnect(this, SIGNAL(waitFinished()), this, SLOT(onWaitFinished()));
}

Q_EXPORT_PLUGIN2(rapidshare, RapidShare)
