/*
 * QCPUFreq - a simple cpufreq GUI
 * Copyright (C) 2010 Daniel Klaffenbach <danielklaffenbach@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "settings.h"
#include "ui_settings.h"

#include <QFile>
#include <QMessageBox>
#include <QStringList>
#include <QTextStream>
#if defined(Q_WS_MAEMO_5)
    #include <QMaemo5InformationBox>
#endif

#define DEFAULT_FREQUENCY 600000

Settings::Settings(QWidget *parent) :
    QDialog(parent),
    settings("qcpufreq"),
    ui(new Ui::Settings)
{
    ui->setupUi(this);

    //read values from config file
    autoRotate = settings.value("main/autorotate", true).toBool();
    overclocking = settings.value("main/overclocking", false).toBool();
    advancedTemperature = settings.value("main/advanced_temperature", false).toBool();

    /* The next few lines of code check if QCPUFreq is running on a power kernel and if
     * the kernel-config script is installed.
     *
     * Basically we get the maximum available frequency - if it is greater than
     * DEFAULT_FREQUENCY we can be sure that the current kernel is a power kernel.
     */
    QFile file( "/sys/devices/system/cpu/cpu0/cpufreq/scaling_available_frequencies" );

    powerKernel = false;
    kernelConfigInstalled = false;

    //open the file
    if ( !file.exists() || !file.open( QIODevice::ReadOnly ) ) {
        QMessageBox::critical(this, tr("QCPUFreq"), tr("Could not get information from /sys!"));
    } else {
        //read the file
        QTextStream in( &file );
        QStringList freqList = in.readLine().split(" ");
        int maxFreq = freqList.at(0).toInt();

        //close the file
        file.close();

        if (maxFreq > DEFAULT_FREQUENCY) {
            powerKernel = true;
        }

        //Check if kernel-config is installed
        file.setFileName("/usr/sbin/kernel-config");
        if (file.exists()) {
            kernelConfigInstalled = true;
        }
    }

    //on non-power-kernels we can disable the overclocking and temperature options
    if (!powerKernel) {
        ui->checkBox_oc->setDisabled(true);
        ui->checkBox_temperature->setDisabled(true);
    }

    //reset GUI
    reset();

    connect(ui->buttonBox, SIGNAL(accepted()), this, SLOT(save()));
    connect(ui->checkBox_oc, SIGNAL(clicked()), this, SLOT(showOverclockingWarning()));
}

Settings::~Settings()
{
    delete ui;
}


/**
  * Returns True if the kernel-config script is installed.
  */
bool Settings::isKernelConfigInstalled()
{
    return kernelConfigInstalled;
}


/**
  * Returns true if we are on a Maemo 5 OS.
  */
bool Settings::platformMaemo()
{
    #if defined(Q_WS_MAEMO_5)
        return true;
    #else
        return false;
    #endif
}


/**
  * Resets the UI to the values stored by QSettings.
  */
void Settings::reset()
{
    ui->checkBox_rotate->setChecked(autoRotate);
    ui->checkBox_oc->setChecked(overclocking);
    ui->checkBox_temperature->setChecked(advancedTemperature);
}


/**
  * Saves the changes and hides the SettingsWidget.
  *
  * @emits: settingsChanged()
  */
void Settings::save()
{
    autoRotate = ui->checkBox_rotate->isChecked();
    overclocking = ui->checkBox_oc->isChecked();
    advancedTemperature = ui->checkBox_temperature->isChecked();

    settings.setValue("main/autorotate", autoRotate);
    settings.setValue("main/overclocking", overclocking);
    settings.setValue("main/advanced_temperature", advancedTemperature);

    //save settings to file
    settings.sync();

    hide();

    emit settingsChanged();
}


/**
  * Displays a warning when overclocking is being enabled.
  */
void Settings::showOverclockingWarning()
{
    if (ui->checkBox_oc->isChecked()) {
#if defined(Q_WS_MAEMO_5)
        QMaemo5InformationBox::information(this, tr( "Please note that overclocking voids your warranty and may break your device! Be careful!"), 0);
#else
        QMessageBox::warning(this, tr("Warning"), tr("Please note that overclocking voids your warranty and may break your device! Be careful!"));
#endif
    }
}


/**
  * Returns true if the user wants to display a more
  * accurate temperature.
  */
bool Settings::useAdvancedTemperature()
{
    return advancedTemperature;
}


/**
  * Returns true if auto-Rotate is enabled.
  */
bool Settings::useAutoRotate()
{
    return autoRotate;
}


/**
  * Returns true if the user enabled overclocking.
  */
bool Settings::useOverclocking()
{
    return overclocking;
}


/**
  * Returns true if QCPUFreq is running on a power kernel.
  */
bool Settings::usePowerKernel()
{
    return powerKernel;
}
