/*
 * QCPUFreq - a simple cpufreq GUI
 * Copyright (C) 2010 Daniel Klaffenbach <daniel.klaffenbach@cs.tu-chemnitz.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QFile>
#include <QMessageBox>
#include <QTextStream>
#include <QDesktopWidget>
#if defined(Q_WS_MAEMO_5)
    #include <QMaemo5InformationBox>
#endif

#define APPNAME "QCPUFreq"
#define APPVERSION "0.3.3"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    //do not allow overclocking per default
    allowOverclocking(false),
    //create helper process
    helperProcess( this ),
    //create a new, stackable help window
    helpWindow( this ),
    //set minFreq to 0
    minFreq(0),
    //create UI refresh timer
    refreshTimer( this ),
    //create a QGraphicsScene for the little chip icon
    scene( this ),
    //show errors about the sudo setup only once
    showSudoError( true )
{
    //this is a stacked window on Maemo 5
    #if defined(Q_WS_MAEMO_5)
        setAttribute(Qt::WA_Maemo5StackedWindow);
    #endif

    ui->setupUi(this);

    refresh();

    // enable auto rotation
    setAutoRotation();

    //initialize orientation
    orientationChanged();

    //refresh UI every 10 seconds
    refreshTimer.start( 10000 );

    // initialize stackable help window
    #if defined(Q_WS_MAEMO_5)
        helpWindow.setAttribute(Qt::WA_Maemo5StackedWindow);
    #endif
    helpWindow.setWindowFlags( windowFlags() | Qt::Window );

    //connect signals and slots
    connect(ui->actionHelp, SIGNAL(triggered()), this, SLOT(showHelp()));
    connect( ui->actionAbout, SIGNAL(triggered()), this, SLOT(about()) );
    connect( ui->freq_adjust, SIGNAL(valueChanged(int)), this, SLOT(adjustFreq()) );
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    connect(ui->sr_box, SIGNAL(clicked()), this, SLOT(setSmartReflex()));
    connect(&refreshTimer, SIGNAL(timeout()), this, SLOT(refresh()));
    connect(ui->actionOverclocking, SIGNAL(toggled(bool)), this, SLOT(setOverclocking()));

    //disable overclocking button on vanilla kernels
    if ( getScalingFreq(0) <= 600000 ) {
        ui->actionOverclocking->setDisabled(true);
    }

}

MainWindow::~MainWindow()
{
    delete ui;
}


/**
  * SLOT: Displays an about box
  */
void MainWindow::about()
{
    QMessageBox::about(this, APPNAME " " APPVERSION, "<p style=\"align:center;\">&copy; 2010 Daniel Klaffenbach</p>" );
    refresh();
}


/**
  * SLOT: Adjusts the maximum CPU frequency according to the scaler
  */
void MainWindow::adjustFreq()
{
    int newmax = getScalingFreq( ui->freq_adjust->sliderPosition() );
    QString max;

    //maxfreq should not be smaller than minfreq, because we do not want to decrease minfreq
    if (newmax < getMinFreq())
        newmax = getMinFreq();

    max.setNum( newmax );

    //check for overclocking
    #if defined(Q_WS_MAEMO_5)
    if (this->allowOverclocking == false && newmax > 600000) {
        QMaemo5InformationBox::information(this, tr( "You need to enable overclocking in QCPUFreq's menu for setting frequencies above 600MHz!"), 0);
        refresh();
        return;
    }
    #endif

    callHelper( "set_maxfreq", max );

    refresh();
}


/**
  * Calls the QCPUFreq helper script with "sudo action param"
  *
  * @param  action : the action of the helper script
  * @param  param : the parameter for the action
  * @return exit code
  */
int MainWindow::callHelper(QString action, QString param)
{
    QStringList arguments;

    #if defined(Q_WS_MAEMO_5)
    //On Maemo 5 the helper script resides in /opt/usr/bin, which is usually not in $PATH
    arguments.append( "/opt/usr/bin/QCPUFreq.helper" );
    #else
    arguments.append( "QCPUFreq.helper" );
    #endif

    arguments.append( action );
    arguments.append( param );

    helperProcess.start( "sudo", arguments, QIODevice::NotOpen );

    if ( showSudoError && !helperProcess.waitForFinished( 400 )) {
        //do not show this error again
        showSudoError = false;
        QMessageBox::critical(this, tr("QCPUFreq"), tr("There seems to be a problem with your sudo setup!"));
    }

    return helperProcess.exitCode();
}


/**
  * Returns the current CPU temperature
  */
QString MainWindow::getCPUTemp()
{
#if defined(Q_WS_MAEMO_5)
    QFile file( "/sys/class/power_supply/bq27200-0/temp" );

    //check if we can read a more accurate temperature (only for power kernel)
    if (file.exists())
        return QString( readSysFile( "class/power_supply/bq27200-0/temp" ) + " " + QString::fromUtf8("\302\260") + "C" );
    else {
        /*
          We actually only need to read the raw temperature, but it appears that by also reading temp1_input
          the raw temperature (temp1_input_raw) is being updated more frequently.
        */
        readSysFile( "devices/platform/omap34xx_temp/temp1_input" );

        //read the current system temperature
        QString tstring = readSysFile( "devices/platform/omap34xx_temp/temp1_input_raw" );
        if (tstring == "0")
            return tr( "Unknown" );

        //convert it to an integer and calculate the approx. temperature from the raw value
        int tint = tstring.toInt();
        tint = ( 0.65 * tint );
        tstring.setNum(tint);
        return QString( tstring + " " + QString::fromUtf8("\302\260") + "C" );
    }
#endif
    return tr( "Unknown" );
}


/**
  * Returns the maximum CPU frequency
  */
int MainWindow::getMaxFreq()
{
    QString tmp = readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_max_freq" );
    return tmp.toInt();
}


/**
  * Returns the minimum CPU frequency
  */
int MainWindow::getMinFreq()
{
    if (this->minFreq == 0) {
        QString min = readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_min_freq" );
        //check if avoid file exists (only on power kernel)
        QFile file( "/sys/devices/system/cpu/cpu0/cpufreq/ondemand/avoid_frequencies" );
        if (file.exists()) {
            QString avoid = readSysFile( "devices/system/cpu/cpu0/cpufreq/ondemand/avoid_frequencies" );
            QStringList avoidList = avoid.split( " " );

            //check if min is in avoid_frequencies
            for (int i = getScalingStep( min.toInt() ); i>0; --i) {
                if (!avoidList.contains(min.setNum( getScalingFreq(i) ))) {
                    this->minFreq = min.toInt();
                    return this->minFreq;
                }
            }

            //should not happen at all
            this->minFreq = 125000;
            return this->minFreq;
        } else {
            this->minFreq = min.toInt();
            return this->minFreq;
        }
    } else {
        return this->minFreq;
    }
}


/**
  * Returns the CPU frequency for the specified scaling step
  */
int MainWindow::getScalingFreq(int step)
{
    QString tmp = readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_available_frequencies" );
    QStringList freqs = tmp.split( " " );
    step = step - 1;
    if ( step < 0 )
         step = 0;
    if ( step > getScalingSteps() )
        step = getScalingSteps();

    tmp = freqs[ step ];
    return tmp.toInt();
}


/**
  * Returns the name of the current CPU frequency scaling governor
  *
  * \return	name of governor
  */
QString MainWindow::getScalingGovernor()
{
    return readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_governor" );
}


/**
  * Returns the amount of available scaling steps.
  */
int MainWindow::getScalingSteps()
{
    QString tmp = readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_available_frequencies" );
    QStringList freqs = tmp.split( " " );
    return (freqs.size() - 1);
}


/**
  * Returns the scaling step for the specified frequency.
  */
int MainWindow::getScalingStep( int freq )
{
    for( int i = 1; i <= getScalingSteps(); ++i ) {
           if ( getScalingFreq(i) == freq )
                return i;
    }

    return 1;
}


/**
  * Returns the SmartReflex(tm) state
  *
  * \return	0|1
  */
int MainWindow::getSmartReflexState()
{
//SmartReflex is only supprted on Maemo5
#if defined(Q_WS_MAEMO_5)
    QString tmp = readSysFile( "power/sr_vdd1_autocomp" );

    if ( tmp == "1" )
	return 1;
    else
	return 0;
#else
    //disable UI checkbox
    ui->sr_box->setDisabled( true );

    return 0;
#endif
}


/**
  * Reads any file in /sys/
  *
  * \param	sys_file : full path to sys file - omit "/sys/"
  * \return	content of sys file
  */
QString MainWindow::readSysFile(QString sys_file)
{
    QFile file( "/sys/"+sys_file );

    //open the file
    if ( !file.exists() || !file.open( QIODevice::ReadOnly ) ) {
        QMessageBox::critical(this, tr("QCPUFreq"), tr("Could not get information from /sys!"));
        return "";
    }

    //read the file
    QTextStream in( &file );
    QString txt = in.readLine();

    //close the file
    file.close();

    return txt;
}


/**
  * Refreshes all of the values to display
  */
void MainWindow::refresh()
{
    //get the current frequency and calculate the MHz value
    int freq = ( getMinFreq() / 1000 );
    QString display;
    display.setNum( freq );
    display.append( " MHz" );
    ui->freq_min->setText( display );

    //do the same thing for the maximum frequency
    freq = ( getMaxFreq() / 1000 );
    display.setNum( freq );
    display.append( " MHz" );
    ui->freq_max->setText( display );

    //display the current governor
    ui->freq_governor->setText( getScalingGovernor() );

    //display current temperature
    ui->cpu_temp->setText( getCPUTemp() );

    //smart reflex button
    if ( getSmartReflexState() == 1 )
	ui->sr_box->setCheckState( Qt::Checked );
    else
	ui->sr_box->setCheckState( Qt::Unchecked );


    //display frequency slider
    ui->freq_adjust->setMinimum( 1 );
    ui->freq_adjust->setMaximum( getScalingSteps() );
    ui->freq_adjust->setInvertedAppearance( true );
    ui->freq_adjust->setSliderPosition( getScalingStep(getMaxFreq()) );

    //ui->retranslateUi(this);
}


/**
  * Repaints part of the GUI after the device was rotated
  */
void MainWindow::orientationChanged()
{
    QPixmap image;

    //check whether we are using portrait or landscape mode
    if ( usePortrait() ) {
	//in portrait mode we want to display the large image
	image.load( ":/img/chip256" );
	scene.clear();
	scene.addPixmap(  image  );

	ui->graphicsPortrait->setScene( &scene );
	ui->graphicsPortrait->setMaximumSize( 256, 256 );
	ui->graphicsLandscape->setMaximumSize( 0, 0 );
    } else {
	image.load( ":/img/chip128" );
	scene.clear();
	scene.addPixmap(  image  );

	ui->graphicsLandscape->setScene( &scene );
	ui->graphicsLandscape->setMaximumSize( 128, 128 );
	ui->graphicsPortrait->setMaximumSize( 0, 0 );
    }
}


/**
  * Enables the auto-rotation feature of Maemo5 devices
  */
void MainWindow::setAutoRotation()
{
#if defined(Q_WS_MAEMO_5)
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif
}


/**
  * SLOT: enable/disable overclocking.
  */
void MainWindow::setOverclocking()
{
    if (ui->actionOverclocking->isChecked()) {
        #if defined(Q_WS_MAEMO_5)
        QMaemo5InformationBox::information(this, tr( "Please note that overclocking voids your warranty and may break your device! Be careful!"), 0);
        #endif
        this->allowOverclocking = true;
    } else {
        this->allowOverclocking = false;
    }
}


/**
  * SLOT: Enables or disables Smart Reflex(tm) after pressing sr_btn
  */
void MainWindow::setSmartReflex()
{
//SmartReflex is only supported on Maemo5
#if defined(Q_WS_MAEMO_5)
    if ( getSmartReflexState() == 1 )
        callHelper( "set_sr", "off");
    else {
        QMaemo5InformationBox::information(this, tr( "SmartReflex support is known to be unstable on some devices and may cause random reboots." ), QMaemo5InformationBox::DefaultTimeout);
        callHelper( "set_sr", "on");
    }

#endif
    //refresh the UI
    refresh();
}


/**
  * SLOT: display the help window
  */
void MainWindow::showHelp()
{
    helpWindow.show();
}


/**
  * Returns true when the device is in portrait mode
  */
bool MainWindow::usePortrait()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height())
        return false;
    else
        return true;
}
