/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtcore_python.h"

#include "qvariant_wrapper.h"

// Extra includes
#include <QList>
#include <QMap>
#include <pysideconversions.h>
#include <qbitarray.h>
#include <qbytearray.h>
#include <qchar.h>
#include <qdatastream.h>
#include <qdatetime.h>
#include <qline.h>
#include <qlocale.h>
#include <qpoint.h>
#include <qrect.h>
#include <qregexp.h>
#include <qsize.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qurl.h>
#include <qvariant.h>

using namespace Shiboken;

static int
SbkQVariant_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QVariant* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0};

    if (!PyArg_UnpackTuple(args, "QVariant", 0, 1, &(pyargs[0])))
        return -1;

    if (numArgs == 0) {
        // QVariant()
        cptr = new QVariant();
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<Qt::GlobalColor >::isConvertible(pyargs[0])) {
        Qt::GlobalColor cpp_arg0 = Shiboken::Converter<Qt::GlobalColor >::toCpp(pyargs[0]);
        // QVariant(Qt::GlobalColor)
        cptr = new QVariant(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QVariant::Type >::isConvertible(pyargs[0])) {
        QVariant::Type cpp_arg0 = Shiboken::Converter<QVariant::Type >::toCpp(pyargs[0]);
        // QVariant(QVariant::Type)
        cptr = new QVariant(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<const char * >::isConvertible(pyargs[0])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        // QVariant(const char*)
        cptr = new QVariant(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && PyObject_Check(pyargs[0])) {
        // QVariant(PyObject*)
        // Begin code injection
        
        cptr = new QVariant(Shiboken::Converter<QVariant >::toCpp(pyargs[0]));
        
        // End of code injection

        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariant_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QVariant >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQVariant_Init_TypeError;

    sbkSelf->validCppObject = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQVariant_Init_TypeError:
        const char* overloads[] = {"", "PyObject", "PySide.QtCore.QVariant.Type", "PySide.QtCore.Qt.GlobalColor", "str", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QVariant", overloads);
        return -1;
}

static PyObject* SbkQVariantFunc_clear(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // clear()
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    cppSelf->QVariant::clear();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject* SbkQVariantFunc_cmp(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QVariant & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QVariant > cpp_arg0_auto_ptr;
        QVariant* cpp_arg0 = Shiboken::Converter<QVariant* >::toCpp(arg);
        if (!SbkQVariant_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QVariant >(cpp_arg0);
        // cmp(QVariant)const
        QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QVariant::cmp(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariantFunc_cmp_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQVariantFunc_cmp_TypeError:
        const char* overloads[] = {"PySide.QtCore.QVariant", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QVariant.cmp", overloads);
        return 0;
}

static PyObject* SbkQVariantFunc_convert(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QVariant::Type >::isConvertible(arg)) {
        QVariant::Type cpp_arg0 = Shiboken::Converter<QVariant::Type >::toCpp(arg);
        // convert(QVariant::Type)
        QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QVariant::convert(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariantFunc_convert_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQVariantFunc_convert_TypeError:
        const char* overloads[] = {"PySide.QtCore.QVariant.Type", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QVariant.convert", overloads);
        return 0;
}

static PyObject* SbkQVariantFunc_isNull(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isNull()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QVariant::isNull();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_isValid(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // isValid()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QVariant::isValid();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_load(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QDataStream & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QDataStream* cpp_arg0 = Shiboken::Converter<QDataStream* >::toCpp(arg);
        // load(QDataStream&)
        QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
        cppSelf->QVariant::load(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariantFunc_load_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQVariantFunc_load_TypeError:
        const char* overloads[] = {"PySide.QtCore.QDataStream", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QVariant.load", overloads);
        return 0;
}

static PyObject* SbkQVariantFunc_nameToType(PyObject* self, PyObject* arg)
{
    PyObject* py_result = 0;

    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // nameToType(const char*)
        QVariant::Type cpp_result = QVariant::nameToType(cpp_arg0);
        py_result = Shiboken::Converter<QVariant::Type >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariantFunc_nameToType_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQVariantFunc_nameToType_TypeError:
        const char* overloads[] = {"str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QVariant.nameToType", overloads);
        return 0;
}

static PyObject* SbkQVariantFunc_save(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QDataStream & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QDataStream* cpp_arg0 = Shiboken::Converter<QDataStream* >::toCpp(arg);
        // save(QDataStream&)const
        QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
        cppSelf->QVariant::save(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariantFunc_save_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQVariantFunc_save_TypeError:
        const char* overloads[] = {"PySide.QtCore.QDataStream", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QVariant.save", overloads);
        return 0;
}

static PyObject* SbkQVariantFunc_toBitArray(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toBitArray()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QBitArray cpp_result = cppSelf->QVariant::toBitArray();
    py_result = Shiboken::Converter<QBitArray >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toBool(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toBool()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QVariant::toBool();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toByteArray(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toByteArray()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QByteArray cpp_result = cppSelf->QVariant::toByteArray();
    py_result = Shiboken::Converter<QByteArray >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toChar(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toChar()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QChar cpp_result = cppSelf->QVariant::toChar();
    py_result = Shiboken::Converter<QChar >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toDate(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toDate()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QDate cpp_result = cppSelf->QVariant::toDate();
    py_result = Shiboken::Converter<QDate >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toDateTime(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toDateTime()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QDateTime cpp_result = cppSelf->QVariant::toDateTime();
    py_result = Shiboken::Converter<QDateTime >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toDouble(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toDouble(bool*)const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    // Begin code injection
    
    // TEMPLATE - fix_bool* - START
    bool ok_;
    double retval_ = cppSelf->toDouble(&ok_);
    py_result = Shiboken::makeTuple(retval_, ok_);
    // TEMPLATE - fix_bool* - END
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toFloat(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toFloat(bool*)const
        QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
        float cpp_result = cppSelf->QVariant::toFloat();
        py_result = Shiboken::Converter<float >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (PyBool_Check(arg)) {
        bool * cpp_arg0 = Shiboken::Converter<bool * >::toCpp(arg);
        // toFloat(bool*)const
        QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
        float cpp_result = cppSelf->QVariant::toFloat(cpp_arg0);
        py_result = Shiboken::Converter<float >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariantFunc_toFloat_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQVariantFunc_toFloat_TypeError:
        const char* overloads[] = {"bool = 0", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QVariant.toFloat", overloads);
        return 0;
}

static PyObject* SbkQVariantFunc_toHash(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toHash()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QHash<QString, QVariant > cpp_result = cppSelf->QVariant::toHash();
    py_result = Shiboken::Converter<QHash<QString, QVariant > >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toInt(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toInt(bool*)const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    // Begin code injection
    
    // TEMPLATE - fix_bool* - START
    bool ok_;
    int retval_ = cppSelf->toInt(&ok_);
    py_result = Shiboken::makeTuple(retval_, ok_);
    // TEMPLATE - fix_bool* - END
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toLine(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toLine()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QLine cpp_result = cppSelf->QVariant::toLine();
    py_result = Shiboken::Converter<QLine >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toLineF(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toLineF()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QLineF cpp_result = cppSelf->QVariant::toLineF();
    py_result = Shiboken::Converter<QLineF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toList(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toList()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    // Begin code injection
    
    if (cppSelf->userType() == qMetaTypeId<PyObjectHolder>()) {
        PyObject* val = cppSelf->value<PyObjectHolder>();
        if (PySequence_Check(val)) {
            py_result = val;
            Py_INCREF(val);
        } else {
            py_result = Py_None;
            Py_INCREF(Py_None);
        }
    } else {
        py_result = Shiboken::Converter<QList<QVariant > >::toPython(cppSelf->toList());
    }
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toLocale(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toLocale()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QLocale cpp_result = cppSelf->QVariant::toLocale();
    py_result = Shiboken::Converter<QLocale >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toLongLong(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toLongLong(bool*)const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    // Begin code injection
    
    // TEMPLATE - fix_bool* - START
    bool ok_;
    qlonglong retval_ = cppSelf->toLongLong(&ok_);
    py_result = Shiboken::makeTuple(retval_, ok_);
    // TEMPLATE - fix_bool* - END
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toMap(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toMap()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QMap<QString, QVariant > cpp_result = cppSelf->QVariant::toMap();
    py_result = Shiboken::Converter<QMap<QString, QVariant > >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toPoint(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toPoint()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QPoint cpp_result = cppSelf->QVariant::toPoint();
    py_result = Shiboken::Converter<QPoint >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toPointF(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toPointF()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QPointF cpp_result = cppSelf->QVariant::toPointF();
    py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toPyObject(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toPyObject()
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    (void)cppSelf; // avoid warnings about unused variables
    // Begin code injection
    
    uint typeId = cppSelf->userType();
    if (typeId == qMetaTypeId<PyObjectHolder>()) {
        py_result = cppSelf->value<PyObjectHolder>();
    } else if (typeId == QVariant::String) {
        py_result = Shiboken::Converter<QString >::toPython(cppSelf->toString());
    } else {
        py_result = Py_None;
        Py_INCREF(Py_None);
    }
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toReal(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = (arg == 0 ? 0 : 1);
    if (numArgs == 0) {
        // toReal(bool*)const
        QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
        qreal cpp_result = cppSelf->QVariant::toReal();
        py_result = Shiboken::Converter<double >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else if (PyBool_Check(arg)) {
        bool * cpp_arg0 = Shiboken::Converter<bool * >::toCpp(arg);
        // toReal(bool*)const
        QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
        qreal cpp_result = cppSelf->QVariant::toReal(cpp_arg0);
        py_result = Shiboken::Converter<double >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariantFunc_toReal_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQVariantFunc_toReal_TypeError:
        const char* overloads[] = {"bool = 0", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QVariant.toReal", overloads);
        return 0;
}

static PyObject* SbkQVariantFunc_toRect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toRect()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QRect cpp_result = cppSelf->QVariant::toRect();
    py_result = Shiboken::Converter<QRect >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toRectF(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toRectF()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QRectF cpp_result = cppSelf->QVariant::toRectF();
    py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toRegExp(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toRegExp()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QRegExp cpp_result = cppSelf->QVariant::toRegExp();
    py_result = Shiboken::Converter<QRegExp >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toSize(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toSize()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QSize cpp_result = cppSelf->QVariant::toSize();
    py_result = Shiboken::Converter<QSize >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toSizeF(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toSizeF()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QSizeF cpp_result = cppSelf->QVariant::toSizeF();
    py_result = Shiboken::Converter<QSizeF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toString(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toString()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QVariant::toString();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toStringList(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toStringList()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QStringList cpp_result = cppSelf->QVariant::toStringList();
    py_result = Shiboken::Converter<QStringList >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toTime(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toTime()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QTime cpp_result = cppSelf->QVariant::toTime();
    py_result = Shiboken::Converter<QTime >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toUInt(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toUInt(bool*)const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    // Begin code injection
    
    // TEMPLATE - fix_bool* - START
    bool ok_;
    uint retval_ = cppSelf->toUInt(&ok_);
    py_result = Shiboken::makeTuple(retval_, ok_);
    // TEMPLATE - fix_bool* - END
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toULongLong(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toULongLong(bool*)const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    // Begin code injection
    
    // TEMPLATE - fix_bool* - START
    bool ok_;
    qulonglong retval_ = cppSelf->toULongLong(&ok_);
    py_result = Shiboken::makeTuple(retval_, ok_);
    // TEMPLATE - fix_bool* - END
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_toUrl(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // toUrl()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QUrl cpp_result = cppSelf->QVariant::toUrl();
    py_result = Shiboken::Converter<QUrl >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_type(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // type()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    QVariant::Type cpp_result = cppSelf->QVariant::type();
    py_result = Shiboken::Converter<QVariant::Type >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_typeName(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // typeName()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    // Begin code injection
    
    if (cppSelf->isNull()) {
        py_result = Py_None;
        Py_INCREF(Py_None);
    } else if (cppSelf->userType() == qMetaTypeId<PyObjectHolder>()) {
        // emulate PyQt4 behaviour
        PyObject* obj = cppSelf->value<PyObjectHolder>();
        if (PySequence_Check(obj))
            py_result = Shiboken::Converter<const char * >::toPython("QVariantList");
    }
    if (!py_result)
        py_result = Shiboken::Converter<const char * >::toPython(cppSelf->typeName());
    
    // End of code injection


    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQVariantFunc_typeToName(PyObject* self, PyObject* arg)
{
    PyObject* py_result = 0;

    if (Shiboken::Converter<QVariant::Type >::isConvertible(arg)) {
        QVariant::Type cpp_arg0 = Shiboken::Converter<QVariant::Type >::toCpp(arg);
        // typeToName(QVariant::Type)
        const char * cpp_result = QVariant::typeToName(cpp_arg0);
        py_result = Shiboken::Converter<const char * >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQVariantFunc_typeToName_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQVariantFunc_typeToName_TypeError:
        const char* overloads[] = {"PySide.QtCore.QVariant.Type", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QVariant.typeToName", overloads);
        return 0;
}

static PyObject* SbkQVariantFunc_userType(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // userType()const
    QVariant* cppSelf = Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QVariant::userType();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyMethodDef SbkQVariant_methods[] = {
    {"clear", (PyCFunction)SbkQVariantFunc_clear, METH_NOARGS},
    {"cmp", (PyCFunction)SbkQVariantFunc_cmp, METH_O},
    {"convert", (PyCFunction)SbkQVariantFunc_convert, METH_O},
    {"isNull", (PyCFunction)SbkQVariantFunc_isNull, METH_NOARGS},
    {"isValid", (PyCFunction)SbkQVariantFunc_isValid, METH_NOARGS},
    {"load", (PyCFunction)SbkQVariantFunc_load, METH_O},
    {"nameToType", (PyCFunction)SbkQVariantFunc_nameToType, METH_O|METH_STATIC},
    {"save", (PyCFunction)SbkQVariantFunc_save, METH_O},
    {"toBitArray", (PyCFunction)SbkQVariantFunc_toBitArray, METH_NOARGS},
    {"toBool", (PyCFunction)SbkQVariantFunc_toBool, METH_NOARGS},
    {"toByteArray", (PyCFunction)SbkQVariantFunc_toByteArray, METH_NOARGS},
    {"toChar", (PyCFunction)SbkQVariantFunc_toChar, METH_NOARGS},
    {"toDate", (PyCFunction)SbkQVariantFunc_toDate, METH_NOARGS},
    {"toDateTime", (PyCFunction)SbkQVariantFunc_toDateTime, METH_NOARGS},
    {"toDouble", (PyCFunction)SbkQVariantFunc_toDouble, METH_NOARGS},
    {"toFloat", (PyCFunction)SbkQVariantFunc_toFloat, METH_NOARGS|METH_O},
    {"toHash", (PyCFunction)SbkQVariantFunc_toHash, METH_NOARGS},
    {"toInt", (PyCFunction)SbkQVariantFunc_toInt, METH_NOARGS},
    {"toLine", (PyCFunction)SbkQVariantFunc_toLine, METH_NOARGS},
    {"toLineF", (PyCFunction)SbkQVariantFunc_toLineF, METH_NOARGS},
    {"toList", (PyCFunction)SbkQVariantFunc_toList, METH_NOARGS},
    {"toLocale", (PyCFunction)SbkQVariantFunc_toLocale, METH_NOARGS},
    {"toLongLong", (PyCFunction)SbkQVariantFunc_toLongLong, METH_NOARGS},
    {"toMap", (PyCFunction)SbkQVariantFunc_toMap, METH_NOARGS},
    {"toPoint", (PyCFunction)SbkQVariantFunc_toPoint, METH_NOARGS},
    {"toPointF", (PyCFunction)SbkQVariantFunc_toPointF, METH_NOARGS},
    {"toPyObject", (PyCFunction)SbkQVariantFunc_toPyObject, METH_NOARGS},
    {"toReal", (PyCFunction)SbkQVariantFunc_toReal, METH_NOARGS|METH_O},
    {"toRect", (PyCFunction)SbkQVariantFunc_toRect, METH_NOARGS},
    {"toRectF", (PyCFunction)SbkQVariantFunc_toRectF, METH_NOARGS},
    {"toRegExp", (PyCFunction)SbkQVariantFunc_toRegExp, METH_NOARGS},
    {"toSize", (PyCFunction)SbkQVariantFunc_toSize, METH_NOARGS},
    {"toSizeF", (PyCFunction)SbkQVariantFunc_toSizeF, METH_NOARGS},
    {"toString", (PyCFunction)SbkQVariantFunc_toString, METH_NOARGS},
    {"toStringList", (PyCFunction)SbkQVariantFunc_toStringList, METH_NOARGS},
    {"toTime", (PyCFunction)SbkQVariantFunc_toTime, METH_NOARGS},
    {"toUInt", (PyCFunction)SbkQVariantFunc_toUInt, METH_NOARGS},
    {"toULongLong", (PyCFunction)SbkQVariantFunc_toULongLong, METH_NOARGS},
    {"toUrl", (PyCFunction)SbkQVariantFunc_toUrl, METH_NOARGS},
    {"type", (PyCFunction)SbkQVariantFunc_type, METH_NOARGS},
    {"typeName", (PyCFunction)SbkQVariantFunc_typeName, METH_NOARGS},
    {"typeToName", (PyCFunction)SbkQVariantFunc_typeToName, METH_O|METH_STATIC},
    {"userType", (PyCFunction)SbkQVariantFunc_userType, METH_NOARGS},
    {0} // Sentinel
};

// Rich comparison
static PyObject* SbkQVariant_richcompare(PyObject* self, PyObject* other, int op)
{
    bool result = false;
    QVariant& cpp_self = *Shiboken::Converter<QVariant* >::toCpp((PyObject*)self);

    switch (op) {
        case Py_NE:
            if (Shiboken::Converter<QVariant & >::isConvertible(other)) {
                // operator!=(const QVariant & v) const
                QVariant* cpp_other = Shiboken::Converter<QVariant* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QVariant  >::toCpp(other) : (cpp_self != (*cpp_other));
            } else if (Shiboken::Converter<QVariant & >::isConvertible(other)) {
                // operator!=(const QVariant & v2)
                QVariant* cpp_other = Shiboken::Converter<QVariant* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QVariant  >::toCpp(other) : (cpp_self != (*cpp_other));
            } else if (Shiboken::Converter<QVariant >::isConvertible(other)) {
                if (Shiboken::cppObjectIsInvalid(other))
                    return 0;
                std::auto_ptr<QVariant > cpp_other_auto_ptr;
                QVariant* cpp_other = Shiboken::Converter<QVariant* >::toCpp(other);
                if (!SbkQVariant_Check(other))
                    cpp_other_auto_ptr = std::auto_ptr<QVariant >(cpp_other);
                result = (cpp_self != (*cpp_other));
            } else goto SbkQVariant_RichComparison_TypeError;

            break;
        case Py_EQ:
            if (Shiboken::Converter<QVariant & >::isConvertible(other)) {
                // operator==(const QVariant & v) const
                QVariant* cpp_other = Shiboken::Converter<QVariant* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QVariant  >::toCpp(other) : (cpp_self == (*cpp_other));
            } else if (Shiboken::Converter<QVariant & >::isConvertible(other)) {
                // operator==(const QVariant & v2)
                QVariant* cpp_other = Shiboken::Converter<QVariant* >::toCpp((PyObject*)other);
                result = !cpp_other ? cpp_self == Shiboken::Converter<QVariant  >::toCpp(other) : (cpp_self == (*cpp_other));
            } else if (Shiboken::Converter<QVariant >::isConvertible(other)) {
                if (Shiboken::cppObjectIsInvalid(other))
                    return 0;
                std::auto_ptr<QVariant > cpp_other_auto_ptr;
                QVariant* cpp_other = Shiboken::Converter<QVariant* >::toCpp(other);
                if (!SbkQVariant_Check(other))
                    cpp_other_auto_ptr = std::auto_ptr<QVariant >(cpp_other);
                result = (cpp_self == (*cpp_other));
            } else goto SbkQVariant_RichComparison_TypeError;

            break;
        default:
            PyErr_SetString(PyExc_NotImplementedError, "operator not implemented.");
            return 0;
    }

    if (result)
        Py_RETURN_TRUE;
    SbkQVariant_RichComparison_TypeError:
    Py_RETURN_FALSE;

}

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQVariant_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtCore.QVariant",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      SbkQVariant_richcompare,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQVariant_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQVariant_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QVariant >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static PyGetSetDef SbkPySide_QtCore_QVariant_Type_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtCore_QVariant_Type_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtCore_QVariant_Type_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "Type",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtCore_QVariant_Type_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtCore_QVariant_Type_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtCore_QVariant_Type_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


PyAPI_FUNC(void) init_QVariant(PyObject* module)
{
    SbkPySide_QtCoreTypes[SBK_QVARIANT_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQVariant_Type);

    if (PyType_Ready((PyTypeObject*)&SbkQVariant_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQVariant_Type));
    PyModule_AddObject(module, "QVariant",
        ((PyObject*)&SbkQVariant_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: Type
    SbkPySide_QtCoreTypes[SBK_QVARIANT_TYPE_IDX] = &SbkPySide_QtCore_QVariant_Type_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtCore_QVariant_Type_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtCore_QVariant_Type_Type);
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
            "Type",((PyObject*)&SbkPySide_QtCore_QVariant_Type_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Invalid, "Invalid");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Invalid", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Invalid", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Bool, "Bool");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Bool", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Bool", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Int, "Int");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Int", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Int", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::UInt, "UInt");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "UInt", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "UInt", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::LongLong, "LongLong");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "LongLong", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "LongLong", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::ULongLong, "ULongLong");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "ULongLong", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "ULongLong", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Double, "Double");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Double", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Double", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Char, "Char");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Char", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Char", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Map, "Map");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Map", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Map", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::List, "List");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "List", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "List", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::String, "String");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "String", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "String", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::StringList, "StringList");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "StringList", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "StringList", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::ByteArray, "ByteArray");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "ByteArray", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "ByteArray", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::BitArray, "BitArray");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "BitArray", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "BitArray", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Date, "Date");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Date", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Date", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Time, "Time");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Time", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Time", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::DateTime, "DateTime");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "DateTime", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "DateTime", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Url, "Url");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Url", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Url", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Locale, "Locale");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Locale", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Locale", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Rect, "Rect");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Rect", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Rect", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::RectF, "RectF");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "RectF", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "RectF", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Size, "Size");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Size", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Size", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::SizeF, "SizeF");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "SizeF", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "SizeF", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Line, "Line");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Line", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Line", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::LineF, "LineF");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "LineF", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "LineF", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Point, "Point");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Point", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Point", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::PointF, "PointF");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "PointF", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "PointF", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::RegExp, "RegExp");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "RegExp", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "RegExp", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Hash, "Hash");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Hash", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Hash", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::LastCoreType, "LastCoreType");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "LastCoreType", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "LastCoreType", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Font, "Font");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Font", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Font", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Pixmap, "Pixmap");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Pixmap", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Pixmap", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Brush, "Brush");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Brush", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Brush", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Color, "Color");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Color", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Color", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Palette, "Palette");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Palette", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Palette", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Icon, "Icon");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Icon", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Icon", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Image, "Image");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Image", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Image", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Polygon, "Polygon");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Polygon", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Polygon", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Region, "Region");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Region", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Region", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Bitmap, "Bitmap");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Bitmap", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Bitmap", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Cursor, "Cursor");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Cursor", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Cursor", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::SizePolicy, "SizePolicy");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "SizePolicy", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "SizePolicy", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::KeySequence, "KeySequence");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "KeySequence", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "KeySequence", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Pen, "Pen");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Pen", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Pen", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::TextLength, "TextLength");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "TextLength", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "TextLength", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::TextFormat, "TextFormat");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "TextFormat", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "TextFormat", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Matrix, "Matrix");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Matrix", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Matrix", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Transform, "Transform");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Transform", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Transform", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Matrix4x4, "Matrix4x4");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Matrix4x4", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Matrix4x4", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Vector2D, "Vector2D");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Vector2D", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Vector2D", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Vector3D, "Vector3D");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Vector3D", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Vector3D", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Vector4D, "Vector4D");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Vector4D", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Vector4D", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::Quaternion, "Quaternion");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "Quaternion", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "Quaternion", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::LastGuiType, "LastGuiType");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "LastGuiType", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "LastGuiType", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::UserType, "UserType");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "UserType", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "UserType", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtCore_QVariant_Type_Type,
        (long) QVariant::LastType, "LastType");
    PyDict_SetItemString(SbkQVariant_Type.super.ht_type.tp_dict,
        "LastType", enum_item);
    PyDict_SetItemString(SbkPySide_QtCore_QVariant_Type_Type.tp_dict,
        "LastType", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QVariant::Type");

    Shiboken::TypeResolver::createValueTypeResolver<QVariant >("QVariant");
    Shiboken::TypeResolver::createValueTypeResolver<QVariant >(typeid(QVariant).name());
}


} // extern "C"

