/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include <signalmanager.h>
#include <dynamicqmetaobject.h>
#include "pyside_qtcore_python.h"

#include "qbuffer_wrapper.h"

// Extra includes
#include <QList>
#include <qbuffer.h>
#include <qbytearray.h>
#include <qcoreevent.h>
#include <qobject.h>
#include <qobjectdefs.h>
#include <qstring.h>
#include <qthread.h>
#include <qvariant.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QBufferWrapper::QBufferWrapper(QByteArray * buf, QObject * parent) : QBuffer(buf, parent), m_metaObject(0) {
    // ... middle
}

QBufferWrapper::QBufferWrapper(QObject * parent) : QBuffer(parent), m_metaObject(0) {
    // ... middle
}

bool QBufferWrapper::atEnd() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "atEnd"));
    if (py_override.isNull()) {
        return this->QBuffer::atEnd();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.atEnd", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

qint64 QBufferWrapper::bytesAvailable() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "bytesAvailable"));
    if (py_override.isNull()) {
        return this->QIODevice::bytesAvailable();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.bytesAvailable", SbkType<qint64 >()->tp_name, py_result->ob_type->tp_name);
        return qint64(0);
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

qint64 QBufferWrapper::bytesToWrite() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "bytesToWrite"));
    if (py_override.isNull()) {
        return this->QIODevice::bytesToWrite();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.bytesToWrite", SbkType<qint64 >()->tp_name, py_result->ob_type->tp_name);
        return qint64(0);
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

bool QBufferWrapper::canReadLine() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "canReadLine"));
    if (py_override.isNull()) {
        return this->QBuffer::canReadLine();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.canReadLine", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

void QBufferWrapper::childEvent(QChildEvent * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "childEvent"));
    if (py_override.isNull()) {
        return this->QObject::childEvent(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QChildEvent * >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QBufferWrapper::close()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "close"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QBuffer::close();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QBufferWrapper::connectNotify(const char * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "connectNotify"));
    if (py_override.isNull()) {
        return this->QBuffer::connectNotify(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(z)",
        arg__1
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QBufferWrapper::customEvent(QEvent * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "customEvent"));
    if (py_override.isNull()) {
        return this->QObject::customEvent(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QEvent * >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QBufferWrapper::disconnectNotify(const char * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "disconnectNotify"));
    if (py_override.isNull()) {
        return this->QBuffer::disconnectNotify(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(z)",
        arg__1
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

bool QBufferWrapper::event(QEvent * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "event"));
    if (py_override.isNull()) {
        return this->QObject::event(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QEvent * >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.event", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

bool QBufferWrapper::eventFilter(QObject * arg__1, QEvent * arg__2)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "eventFilter"));
    if (py_override.isNull()) {
        return this->QObject::eventFilter(arg__1, arg__2);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QObject * >::toPython(arg__1),
        Shiboken::Converter<QEvent * >::toPython(arg__2)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.eventFilter", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 1));
    return cpp_result;
}

bool QBufferWrapper::isSequential() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "isSequential"));
    if (py_override.isNull()) {
        return this->QIODevice::isSequential();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.isSequential", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QBufferWrapper::open(QFlags<QIODevice::OpenModeFlag> openMode)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "open"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QBuffer::open(openMode);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QFlags<QIODevice::OpenModeFlag> >::toPython(openMode)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.open", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

qint64 QBufferWrapper::pos() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "pos"));
    if (py_override.isNull()) {
        return this->QBuffer::pos();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.pos", SbkType<qint64 >()->tp_name, py_result->ob_type->tp_name);
        return qint64(0);
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

qint64 QBufferWrapper::readData(char * data, qint64 maxlen)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "readData"));
    if (py_override.isNull()) {
        return this->QBuffer::readData(data, maxlen);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(zL)",
        data,
        Shiboken::Converter<long long >::toPython(maxlen)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.readData", SbkType<qint64 >()->tp_name, py_result->ob_type->tp_name);
        return qint64(0);
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

qint64 QBufferWrapper::readLineData(char * data, qint64 maxlen)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "readLineData"));
    if (py_override.isNull()) {
        return this->QIODevice::readLineData(data, maxlen);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(zL)",
        data,
        Shiboken::Converter<long long >::toPython(maxlen)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.readLineData", SbkType<qint64 >()->tp_name, py_result->ob_type->tp_name);
        return qint64(0);
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

bool QBufferWrapper::reset()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "reset"));
    if (py_override.isNull()) {
        return this->QIODevice::reset();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.reset", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QBufferWrapper::seek(qint64 off)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "seek"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QBuffer::seek(off);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(L)",
        Shiboken::Converter<long long >::toPython(off)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.seek", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

qint64 QBufferWrapper::size() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "size"));
    if (py_override.isNull()) {
        return this->QBuffer::size();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.size", SbkType<qint64 >()->tp_name, py_result->ob_type->tp_name);
        return qint64(0);
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

void QBufferWrapper::timerEvent(QTimerEvent * arg__1)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "timerEvent"));
    if (py_override.isNull()) {
        return this->QObject::timerEvent(arg__1);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QTimerEvent * >::toPython(arg__1)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

bool QBufferWrapper::waitForBytesWritten(int msecs)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "waitForBytesWritten"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QIODevice::waitForBytesWritten(msecs);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(i)",
        msecs
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.waitForBytesWritten", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QBufferWrapper::waitForReadyRead(int msecs)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "waitForReadyRead"));
    if (py_override.isNull()) {
        Shiboken::ThreadStateSaver thread_state_saver;
        thread_state_saver.save();
        return this->QIODevice::waitForReadyRead(msecs);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(i)",
        msecs
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.waitForReadyRead", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

qint64 QBufferWrapper::writeData(const char * data, qint64 len)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "writeData"));
    if (py_override.isNull()) {
        return this->QBuffer::writeData(data, len);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(zL)",
        data,
        Shiboken::Converter<long long >::toPython(len)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return qint64(0);
    }
    // Check return type
    bool typeIsValid = PyLong_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QBuffer.writeData", SbkType<qint64 >()->tp_name, py_result->ob_type->tp_name);
        return qint64(0);
    }
    qint64 cpp_result(Shiboken::Converter<long long >::toCpp(py_result));
    return cpp_result;
}

const QMetaObject* QBufferWrapper::metaObject() const
{
    if (!m_metaObject) {
        PyObject *pySelf = BindingManager::instance().retrieveWrapper(this);
        QString className(pySelf->ob_type->tp_name);
        className = className.mid(className.lastIndexOf(".")+1);
        m_metaObject = new PySide::DynamicQMetaObject(className.toAscii(), &QBuffer::staticMetaObject);
}
    return m_metaObject;
}

int QBufferWrapper::qt_metacall(QMetaObject::Call call, int id, void** args)
{
    int result = QBuffer::qt_metacall(call, id, args);
    return result < 0 ? result : PySide::SignalManager::qt_metacall(this, call, id, args);
}

QBufferWrapper::~QBufferWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
    delete m_metaObject;
}

// Target ---------------------------------------------------------

static int
SbkQBuffer_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QBufferWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "QBuffer", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return -1;

    if (numArgs == 0) {
        // QBuffer(QObject*)
        cptr = new QBufferWrapper();
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(pyargs[0], self);
    } else if (Shiboken::Converter<QObject * >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QObject* cpp_arg0 = Shiboken::Converter<QObject* >::toCpp(pyargs[0]);
        // QBuffer(QObject*)
        cptr = new QBufferWrapper(cpp_arg0);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(pyargs[0], self);
    } else if (Shiboken::Converter<QByteArray * >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(pyargs[0]);
        if (!SbkQByteArray_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        if (numArgs == 1) {
            // QBuffer(QByteArray*,QObject*)
            cptr = new QBufferWrapper(cpp_arg0);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[1], self);
        } else if (Shiboken::Converter<QObject * >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return -1;
            QObject* cpp_arg1 = Shiboken::Converter<QObject* >::toCpp(pyargs[1]);
            // QBuffer(QByteArray*,QObject*)
            cptr = new QBufferWrapper(cpp_arg0, cpp_arg1);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[1], self);
        } else goto SbkQBuffer_Init_TypeError;
    } else goto SbkQBuffer_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QBuffer >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQBuffer_Init_TypeError;

    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQBuffer_Init_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray, PySide.QtCore.QObject = None", "PySide.QtCore.QObject = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QBuffer", overloads);
        return -1;
}

static PyObject* SbkQBufferFunc_atEnd(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // atEnd()const
    QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QBuffer::atEnd();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQBufferFunc_buffer(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // buffer()
    QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
    QByteArray & cpp_result = cppSelf->QBuffer::buffer();
    py_result = Shiboken::Converter<QByteArray & >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQBufferFunc_canReadLine(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // canReadLine()const
    QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
    bool cpp_result = cppSelf->QBuffer::canReadLine();
    py_result = Shiboken::Converter<bool >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQBufferFunc_close(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    Shiboken::ThreadStateSaver thread_state_saver;

    // close()
    QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
    thread_state_saver.save();
    cppSelf->QBuffer::close();
    thread_state_saver.restore();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject* SbkQBufferFunc_connectNotify(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // connectNotify(const char*)
        QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
        cppSelf->QBuffer::connectNotify(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQBufferFunc_connectNotify_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQBufferFunc_connectNotify_TypeError:
        const char* overloads[] = {"str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QBuffer.connectNotify", overloads);
        return 0;
}

static PyObject* SbkQBufferFunc_disconnectNotify(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<const char * >::isConvertible(arg)) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(arg);
        // disconnectNotify(const char*)
        QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
        cppSelf->QBuffer::disconnectNotify(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQBufferFunc_disconnectNotify_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQBufferFunc_disconnectNotify_TypeError:
        const char* overloads[] = {"str", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QBuffer.disconnectNotify", overloads);
        return 0;
}

static PyObject* SbkQBufferFunc_open(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    if (SbkPySide_QtCore_QIODevice_OpenModeFlag_Check(arg)) {
        QFlags<QIODevice::OpenModeFlag> cpp_arg0 = Shiboken::Converter<QFlags<QIODevice::OpenModeFlag> >::toCpp(arg);
        // open(QFlags<QIODevice::OpenModeFlag>)
        QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QBuffer::open(cpp_arg0);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQBufferFunc_open_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQBufferFunc_open_TypeError:
        const char* overloads[] = {"PySide.QtCore.QIODevice.OpenMode", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QBuffer.open", overloads);
        return 0;
}

static PyObject* SbkQBufferFunc_pos(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // pos()const
    QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
    qint64 cpp_result = cppSelf->QBuffer::pos();
    py_result = Shiboken::Converter<long long >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQBufferFunc_readData(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "readData", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<char * >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        char * cpp_arg0 = Shiboken::Converter<char * >::toCpp(pyargs[0]);
        qint64 cpp_arg1 = Shiboken::Converter<qint64 >::toCpp(pyargs[1]);
        // readData(char*,qint64)
        QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
        qint64 cpp_result = cppSelf->QBuffer::readData(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<long long >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQBufferFunc_readData_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQBufferFunc_readData_TypeError:
        const char* overloads[] = {"str, long long", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QBuffer.readData", overloads);
        return 0;
}

static PyObject* SbkQBufferFunc_seek(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;
    Shiboken::ThreadStateSaver thread_state_saver;

    if (PyNumber_Check(arg)) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(arg);
        // seek(qint64)
        QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
        thread_state_saver.save();
        bool cpp_result = cppSelf->QBuffer::seek(cpp_arg0);
        thread_state_saver.restore();
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQBufferFunc_seek_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQBufferFunc_seek_TypeError:
        const char* overloads[] = {"long long", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QBuffer.seek", overloads);
        return 0;
}

static PyObject* SbkQBufferFunc_setBuffer(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QByteArray * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // setBuffer(QByteArray*)
        QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
        cppSelf->QBuffer::setBuffer(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQBufferFunc_setBuffer_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQBufferFunc_setBuffer_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QBuffer.setBuffer", overloads);
        return 0;
}

static PyObject* SbkQBufferFunc_setData(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QByteArray & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QByteArray > cpp_arg0_auto_ptr;
        QByteArray* cpp_arg0 = Shiboken::Converter<QByteArray* >::toCpp(arg);
        if (!SbkQByteArray_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QByteArray >(cpp_arg0);
        // setData(QByteArray)
        QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
        cppSelf->QBuffer::setData(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQBufferFunc_setData_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQBufferFunc_setData_TypeError:
        const char* overloads[] = {"PySide.QtCore.QByteArray", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtCore.QBuffer.setData", overloads);
        return 0;
}

static PyObject* SbkQBufferFunc_size(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // size()const
    QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
    qint64 cpp_result = cppSelf->QBuffer::size();
    py_result = Shiboken::Converter<long long >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQBufferFunc_writeData(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "writeData", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<const char * >::isConvertible(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        const char * cpp_arg0 = Shiboken::Converter<const char * >::toCpp(pyargs[0]);
        qint64 cpp_arg1 = Shiboken::Converter<qint64 >::toCpp(pyargs[1]);
        // writeData(const char*,qint64)
        QBuffer* cppSelf = Shiboken::Converter<QBuffer* >::toCpp((PyObject*)self);
        qint64 cpp_result = cppSelf->QBuffer::writeData(cpp_arg0, cpp_arg1);
        py_result = Shiboken::Converter<long long >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQBufferFunc_writeData_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQBufferFunc_writeData_TypeError:
        const char* overloads[] = {"str, long long", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtCore.QBuffer.writeData", overloads);
        return 0;
}

static PyMethodDef SbkQBuffer_methods[] = {
    {"atEnd", (PyCFunction)SbkQBufferFunc_atEnd, METH_NOARGS},
    {"buffer", (PyCFunction)SbkQBufferFunc_buffer, METH_NOARGS},
    {"canReadLine", (PyCFunction)SbkQBufferFunc_canReadLine, METH_NOARGS},
    {"close", (PyCFunction)SbkQBufferFunc_close, METH_NOARGS},
    {"connectNotify", (PyCFunction)SbkQBufferFunc_connectNotify, METH_O},
    {"disconnectNotify", (PyCFunction)SbkQBufferFunc_disconnectNotify, METH_O},
    {"open", (PyCFunction)SbkQBufferFunc_open, METH_O},
    {"pos", (PyCFunction)SbkQBufferFunc_pos, METH_NOARGS},
    {"readData", (PyCFunction)SbkQBufferFunc_readData, METH_VARARGS},
    {"seek", (PyCFunction)SbkQBufferFunc_seek, METH_O},
    {"setBuffer", (PyCFunction)SbkQBufferFunc_setBuffer, METH_O},
    {"setData", (PyCFunction)SbkQBufferFunc_setData, METH_O},
    {"size", (PyCFunction)SbkQBufferFunc_size, METH_NOARGS},
    {"writeData", (PyCFunction)SbkQBufferFunc_writeData, METH_VARARGS},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQBuffer_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtCore.QBuffer",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQBuffer_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             0,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQBuffer_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QBuffer >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};


PyAPI_FUNC(void) init_QBuffer(PyObject* module)
{
    SbkPySide_QtCoreTypes[SBK_QBUFFER_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQBuffer_Type);

    SbkQBuffer_Type.super.ht_type.tp_base = SbkPySide_QtCoreTypes[SBK_QIODEVICE_IDX];
    // Fill type discovery information
    SbkQBuffer_Type.type_discovery = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_QtCoreTypes[SBK_QOBJECT_IDX])->type_discovery;

    if (PyType_Ready((PyTypeObject*)&SbkQBuffer_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQBuffer_Type));
    PyModule_AddObject(module, "QBuffer",
        ((PyObject*)&SbkQBuffer_Type));

    Shiboken::TypeResolver::createObjectTypeResolver<QBuffer >("QBuffer*");
    Shiboken::TypeResolver::createObjectTypeResolver<QBuffer >(typeid(QBuffer).name());
}


} // extern "C"

