/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtgui_python.h"

#include "qundocommand_wrapper.h"

// Extra includes
#include <qstring.h>
#include <qundostack.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QUndoCommandWrapper::QUndoCommandWrapper(QUndoCommand * parent) : QUndoCommand(parent) {
    // ... middle
}

QUndoCommandWrapper::QUndoCommandWrapper(const QString & text, QUndoCommand * parent) : QUndoCommand(text, parent) {
    // ... middle
}

int QUndoCommandWrapper::id() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "id"));
    if (py_override.isNull()) {
        return this->QUndoCommand::id();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QUndoCommand.id", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

bool QUndoCommandWrapper::mergeWith(const QUndoCommand * other)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mergeWith"));
    if (py_override.isNull()) {
        return this->QUndoCommand::mergeWith(other);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QUndoCommand * >::toPython(other)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QUndoCommand.mergeWith", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

void QUndoCommandWrapper::redo()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "redo"));
    if (py_override.isNull()) {
        return this->QUndoCommand::redo();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QUndoCommandWrapper::undo()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "undo"));
    if (py_override.isNull()) {
        return this->QUndoCommand::undo();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

QUndoCommandWrapper::~QUndoCommandWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQUndoCommand_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QUndoCommandWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "QUndoCommand", 0, 2, &(pyargs[0]), &(pyargs[1])))
        return -1;

    if (numArgs == 0) {
        // QUndoCommand(QUndoCommand*)
        cptr = new QUndoCommandWrapper();
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(pyargs[0], self);
    } else if (Shiboken::Converter<QString & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(pyargs[0]);
        if (!SbkQString_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        if (numArgs == 1) {
            // QUndoCommand(QString,QUndoCommand*)
            cptr = new QUndoCommandWrapper(*cpp_arg0);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[1], self);
        } else if (Shiboken::Converter<QUndoCommand * >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return -1;
            QUndoCommand* cpp_arg1 = Shiboken::Converter<QUndoCommand* >::toCpp(pyargs[1]);
            // QUndoCommand(QString,QUndoCommand*)
            cptr = new QUndoCommandWrapper(*cpp_arg0, cpp_arg1);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[1], self);
        } else goto SbkQUndoCommand_Init_TypeError;
    } else if (Shiboken::Converter<QUndoCommand * >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QUndoCommand* cpp_arg0 = Shiboken::Converter<QUndoCommand* >::toCpp(pyargs[0]);
        // QUndoCommand(QUndoCommand*)
        cptr = new QUndoCommandWrapper(cpp_arg0);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(pyargs[0], self);
    } else goto SbkQUndoCommand_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QUndoCommand >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQUndoCommand_Init_TypeError;

    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQUndoCommand_Init_TypeError:
        const char* overloads[] = {"PySide.QtGui.QUndoCommand = None", "PySide.QtCore.QString, PySide.QtGui.QUndoCommand = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QUndoCommand", overloads);
        return -1;
}

static PyObject* SbkQUndoCommandFunc_child(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (PyNumber_Check(arg)) {
        int cpp_arg0 = Shiboken::Converter<int >::toCpp(arg);
        // child(int)const
        QUndoCommand* cppSelf = Shiboken::Converter<QUndoCommand* >::toCpp((PyObject*)self);
        const QUndoCommand * cpp_result = cppSelf->QUndoCommand::child(cpp_arg0);
        py_result = Shiboken::Converter<QUndoCommand * >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(self, py_result);
    } else goto SbkQUndoCommandFunc_child_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQUndoCommandFunc_child_TypeError:
        const char* overloads[] = {"int", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QUndoCommand.child", overloads);
        return 0;
}

static PyObject* SbkQUndoCommandFunc_childCount(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // childCount()const
    QUndoCommand* cppSelf = Shiboken::Converter<QUndoCommand* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QUndoCommand::childCount();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQUndoCommandFunc_id(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // id()const
    QUndoCommand* cppSelf = Shiboken::Converter<QUndoCommand* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QUndoCommand::id();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQUndoCommandFunc_mergeWith(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QUndoCommand * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QUndoCommand* cpp_arg0 = Shiboken::Converter<QUndoCommand* >::toCpp(arg);
        // mergeWith(const QUndoCommand*)
        QUndoCommand* cppSelf = Shiboken::Converter<QUndoCommand* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QUndoCommand::mergeWith(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQUndoCommandFunc_mergeWith_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQUndoCommandFunc_mergeWith_TypeError:
        const char* overloads[] = {"PySide.QtGui.QUndoCommand", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QUndoCommand.mergeWith", overloads);
        return 0;
}

static PyObject* SbkQUndoCommandFunc_redo(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // redo()
    QUndoCommand* cppSelf = Shiboken::Converter<QUndoCommand* >::toCpp((PyObject*)self);
    cppSelf->QUndoCommand::redo();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject* SbkQUndoCommandFunc_setText(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QString & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QString > cpp_arg0_auto_ptr;
        QString* cpp_arg0 = Shiboken::Converter<QString* >::toCpp(arg);
        if (!SbkQString_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QString >(cpp_arg0);
        // setText(QString)
        QUndoCommand* cppSelf = Shiboken::Converter<QUndoCommand* >::toCpp((PyObject*)self);
        cppSelf->QUndoCommand::setText(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQUndoCommandFunc_setText_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQUndoCommandFunc_setText_TypeError:
        const char* overloads[] = {"PySide.QtCore.QString", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QUndoCommand.setText", overloads);
        return 0;
}

static PyObject* SbkQUndoCommandFunc_text(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // text()const
    QUndoCommand* cppSelf = Shiboken::Converter<QUndoCommand* >::toCpp((PyObject*)self);
    QString cpp_result = cppSelf->QUndoCommand::text();
    py_result = Shiboken::Converter<QString >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQUndoCommandFunc_undo(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // undo()
    QUndoCommand* cppSelf = Shiboken::Converter<QUndoCommand* >::toCpp((PyObject*)self);
    cppSelf->QUndoCommand::undo();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyMethodDef SbkQUndoCommand_methods[] = {
    {"child", (PyCFunction)SbkQUndoCommandFunc_child, METH_O},
    {"childCount", (PyCFunction)SbkQUndoCommandFunc_childCount, METH_NOARGS},
    {"id", (PyCFunction)SbkQUndoCommandFunc_id, METH_NOARGS},
    {"mergeWith", (PyCFunction)SbkQUndoCommandFunc_mergeWith, METH_O},
    {"redo", (PyCFunction)SbkQUndoCommandFunc_redo, METH_NOARGS},
    {"setText", (PyCFunction)SbkQUndoCommandFunc_setText, METH_O},
    {"text", (PyCFunction)SbkQUndoCommandFunc_text, METH_NOARGS},
    {"undo", (PyCFunction)SbkQUndoCommandFunc_undo, METH_NOARGS},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQUndoCommand_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtGui.QUndoCommand",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQUndoCommand_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQUndoCommand_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QUndoCommand >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static SbkBaseWrapperType* SbkQUndoCommand_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQUndoCommand_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QUndoCommand*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}


PyAPI_FUNC(void) init_QUndoCommand(PyObject* module)
{
    SbkPySide_QtGuiTypes[SBK_QUNDOCOMMAND_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQUndoCommand_Type);

    // Fill type discovery information
    SbkQUndoCommand_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQUndoCommand_Type.type_discovery->addTypeDiscoveryFunction(&SbkQUndoCommand_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQUndoCommand_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQUndoCommand_Type));
    PyModule_AddObject(module, "QUndoCommand",
        ((PyObject*)&SbkQUndoCommand_Type));

    Shiboken::TypeResolver::createObjectTypeResolver<QUndoCommand >("QUndoCommand*");
    Shiboken::TypeResolver::createObjectTypeResolver<QUndoCommand >(typeid(QUndoCommand).name());
}


} // extern "C"

