/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtgui_python.h"

#include "qgraphicspixmapitem_wrapper.h"

// Extra includes
#include <QList>
#include <qcoreevent.h>
#include <qcursor.h>
#include <qevent.h>
#include <qgraphicseffect.h>
#include <qgraphicsitem.h>
#include <qgraphicsscene.h>
#include <qgraphicssceneevent.h>
#include <qgraphicstransform.h>
#include <qgraphicswidget.h>
#include <qpainter.h>
#include <qpainterpath.h>
#include <qpixmap.h>
#include <qpoint.h>
#include <qpolygon.h>
#include <qrect.h>
#include <qregion.h>
#include <qsize.h>
#include <qstring.h>
#include <qstyleoption.h>
#include <qtransform.h>
#include <qvariant.h>
#include <qwidget.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QGraphicsPixmapItemWrapper::QGraphicsPixmapItemWrapper(QGraphicsItem * parent, QGraphicsScene * scene) : QGraphicsPixmapItem(parent, scene) {
    // ... middle
}

QGraphicsPixmapItemWrapper::QGraphicsPixmapItemWrapper(const QPixmap & pixmap, QGraphicsItem * parent, QGraphicsScene * scene) : QGraphicsPixmapItem(pixmap, parent, scene) {
    // ... middle
}

void QGraphicsPixmapItemWrapper::advance(int phase)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "advance"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::advance(phase);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(i)",
        phase
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

QRectF QGraphicsPixmapItemWrapper::boundingRect() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "boundingRect"));
    if (py_override.isNull()) {
        return this->QGraphicsPixmapItem::boundingRect();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QRectF();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QRectF >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.boundingRect", SbkType<QRectF >()->tp_name, py_result->ob_type->tp_name);
        return QRectF();
    }
    QRectF cpp_result(Shiboken::Converter<QRectF >::toCpp(py_result));
    return cpp_result;
}

bool QGraphicsPixmapItemWrapper::collidesWithItem(const QGraphicsItem * other, Qt::ItemSelectionMode mode) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "collidesWithItem"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::collidesWithItem(other, mode);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QGraphicsItem * >::toPython(other),
        Shiboken::Converter<Qt::ItemSelectionMode >::toPython(mode)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.collidesWithItem", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

bool QGraphicsPixmapItemWrapper::collidesWithPath(const QPainterPath & path, Qt::ItemSelectionMode mode) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "collidesWithPath"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::collidesWithPath(path, mode);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QPainterPath & >::toPython(path),
        Shiboken::Converter<Qt::ItemSelectionMode >::toPython(mode)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.collidesWithPath", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

bool QGraphicsPixmapItemWrapper::contains(const QPointF & point) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "contains"));
    if (py_override.isNull()) {
        return this->QGraphicsPixmapItem::contains(point);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QPointF & >::toPython(point)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.contains", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsPixmapItemWrapper::contextMenuEvent(QGraphicsSceneContextMenuEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "contextMenuEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::contextMenuEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneContextMenuEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::dragEnterEvent(QGraphicsSceneDragDropEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "dragEnterEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::dragEnterEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneDragDropEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::dragLeaveEvent(QGraphicsSceneDragDropEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "dragLeaveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::dragLeaveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneDragDropEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::dragMoveEvent(QGraphicsSceneDragDropEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "dragMoveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::dragMoveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneDragDropEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::dropEvent(QGraphicsSceneDragDropEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "dropEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::dropEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneDragDropEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QVariant QGraphicsPixmapItemWrapper::extension(const QVariant & variant) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "extension"));
    if (py_override.isNull()) {
        return this->QGraphicsPixmapItem::extension(variant);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QVariant & >::toPython(variant)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QVariant();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QVariant >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.extension", SbkType<QVariant >()->tp_name, py_result->ob_type->tp_name);
        return QVariant();
    }
    QVariant cpp_result(Shiboken::Converter<QVariant >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsPixmapItemWrapper::focusInEvent(QFocusEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "focusInEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::focusInEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QFocusEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::focusOutEvent(QFocusEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "focusOutEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::focusOutEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QFocusEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::hoverEnterEvent(QGraphicsSceneHoverEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "hoverEnterEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::hoverEnterEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneHoverEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::hoverLeaveEvent(QGraphicsSceneHoverEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "hoverLeaveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::hoverLeaveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneHoverEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::hoverMoveEvent(QGraphicsSceneHoverEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "hoverMoveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::hoverMoveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneHoverEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::inputMethodEvent(QInputMethodEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "inputMethodEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::inputMethodEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QInputMethodEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QVariant QGraphicsPixmapItemWrapper::inputMethodQuery(Qt::InputMethodQuery query) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "inputMethodQuery"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::inputMethodQuery(query);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<Qt::InputMethodQuery >::toPython(query)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QVariant();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QVariant >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.inputMethodQuery", SbkType<QVariant >()->tp_name, py_result->ob_type->tp_name);
        return QVariant();
    }
    QVariant cpp_result(Shiboken::Converter<QVariant >::toCpp(py_result));
    return cpp_result;
}

bool QGraphicsPixmapItemWrapper::isObscuredBy(const QGraphicsItem * item) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "isObscuredBy"));
    if (py_override.isNull()) {
        return this->QGraphicsPixmapItem::isObscuredBy(item);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsItem * >::toPython(item)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.isObscuredBy", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

QVariant QGraphicsPixmapItemWrapper::itemChange(QGraphicsItem::GraphicsItemChange change, const QVariant & value)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "itemChange"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::itemChange(change, value);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QGraphicsItem::GraphicsItemChange >::toPython(change),
        Shiboken::Converter<QVariant & >::toPython(value)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QVariant();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QVariant >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.itemChange", SbkType<QVariant >()->tp_name, py_result->ob_type->tp_name);
        return QVariant();
    }
    QVariant cpp_result(Shiboken::Converter<QVariant >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsPixmapItemWrapper::keyPressEvent(QKeyEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "keyPressEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::keyPressEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QKeyEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::keyReleaseEvent(QKeyEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "keyReleaseEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::keyReleaseEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QKeyEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::mouseDoubleClickEvent(QGraphicsSceneMouseEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mouseDoubleClickEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::mouseDoubleClickEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneMouseEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::mouseMoveEvent(QGraphicsSceneMouseEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mouseMoveEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::mouseMoveEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneMouseEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::mousePressEvent(QGraphicsSceneMouseEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mousePressEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::mousePressEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneMouseEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

void QGraphicsPixmapItemWrapper::mouseReleaseEvent(QGraphicsSceneMouseEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "mouseReleaseEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::mouseReleaseEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneMouseEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QPainterPath QGraphicsPixmapItemWrapper::opaqueArea() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "opaqueArea"));
    if (py_override.isNull()) {
        return this->QGraphicsPixmapItem::opaqueArea();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QPainterPath();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QPainterPath >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.opaqueArea", SbkType<QPainterPath >()->tp_name, py_result->ob_type->tp_name);
        return QPainterPath();
    }
    QPainterPath cpp_result(Shiboken::Converter<QPainterPath >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsPixmapItemWrapper::paint(QPainter * painter, const QStyleOptionGraphicsItem * option, QWidget * widget)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "paint"));
    if (py_override.isNull()) {
        return this->QGraphicsPixmapItem::paint(painter, option, widget);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SSS)",
        Shiboken::Converter<QPainter * >::toPython(painter),
        Shiboken::Converter<QStyleOptionGraphicsItem * >::toPython(option),
        Shiboken::Converter<QWidget * >::toPython(widget)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

bool QGraphicsPixmapItemWrapper::sceneEvent(QEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "sceneEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::sceneEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.sceneEvent", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    return cpp_result;
}

bool QGraphicsPixmapItemWrapper::sceneEventFilter(QGraphicsItem * watched, QEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "sceneEventFilter"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::sceneEventFilter(watched, event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QGraphicsItem * >::toPython(watched),
        Shiboken::Converter<QEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.sceneEventFilter", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 1));
    return cpp_result;
}

QPainterPath QGraphicsPixmapItemWrapper::shape() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "shape"));
    if (py_override.isNull()) {
        return this->QGraphicsPixmapItem::shape();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QPainterPath();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QPainterPath >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.shape", SbkType<QPainterPath >()->tp_name, py_result->ob_type->tp_name);
        return QPainterPath();
    }
    QPainterPath cpp_result(Shiboken::Converter<QPainterPath >::toCpp(py_result));
    return cpp_result;
}

int QGraphicsPixmapItemWrapper::type() const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "type"));
    if (py_override.isNull()) {
        return this->QGraphicsPixmapItem::type();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return int(0);
    }
    // Check return type
    bool typeIsValid = PyInt_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QGraphicsPixmapItem.type", SbkType<int >()->tp_name, py_result->ob_type->tp_name);
        return int(0);
    }
    int cpp_result(Shiboken::Converter<int >::toCpp(py_result));
    return cpp_result;
}

void QGraphicsPixmapItemWrapper::wheelEvent(QGraphicsSceneWheelEvent * event)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "wheelEvent"));
    if (py_override.isNull()) {
        return this->QGraphicsItem::wheelEvent(event);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QGraphicsSceneWheelEvent * >::toPython(event)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    BindingManager::instance().invalidateWrapper(PyTuple_GET_ITEM(pyargs, 0));
}

QGraphicsPixmapItemWrapper::~QGraphicsPixmapItemWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQGraphicsPixmapItem_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QGraphicsPixmapItemWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "QGraphicsPixmapItem", 0, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return -1;

    if (numArgs == 0) {
        // QGraphicsPixmapItem(QGraphicsItem*,QGraphicsScene*)
        cptr = new QGraphicsPixmapItemWrapper();
        //CppGenerator::writeParentChildManagement
        Shiboken::setParent(pyargs[0], self);
    } else if (Shiboken::Converter<QPixmap & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        std::auto_ptr<QPixmap > cpp_arg0_auto_ptr;
        QPixmap* cpp_arg0 = Shiboken::Converter<QPixmap* >::toCpp(pyargs[0]);
        if (!SbkQPixmap_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPixmap >(cpp_arg0);
        if (numArgs == 1) {
            // QGraphicsPixmapItem(QPixmap,QGraphicsItem*,QGraphicsScene*)
            cptr = new QGraphicsPixmapItemWrapper(*cpp_arg0);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[1], self);
        } else if (Shiboken::Converter<QGraphicsItem * >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return -1;
            QGraphicsItem* cpp_arg1 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[1]);
            if (numArgs == 2) {
                // QGraphicsPixmapItem(QPixmap,QGraphicsItem*,QGraphicsScene*)
                cptr = new QGraphicsPixmapItemWrapper(*cpp_arg0, cpp_arg1);
                //CppGenerator::writeParentChildManagement
                Shiboken::setParent(pyargs[1], self);
            } else if (Shiboken::Converter<QGraphicsScene * >::isConvertible(pyargs[2])) {
                if (Shiboken::cppObjectIsInvalid(pyargs[2]))
                    return -1;
                QGraphicsScene* cpp_arg2 = Shiboken::Converter<QGraphicsScene* >::toCpp(pyargs[2]);
                // QGraphicsPixmapItem(QPixmap,QGraphicsItem*,QGraphicsScene*)
                cptr = new QGraphicsPixmapItemWrapper(*cpp_arg0, cpp_arg1, cpp_arg2);
                //CppGenerator::writeParentChildManagement
                Shiboken::setParent(pyargs[1], self);
            } else goto SbkQGraphicsPixmapItem_Init_TypeError;
        } else goto SbkQGraphicsPixmapItem_Init_TypeError;
    } else if (Shiboken::Converter<QGraphicsItem * >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // QGraphicsPixmapItem(QGraphicsItem*,QGraphicsScene*)
            cptr = new QGraphicsPixmapItemWrapper(cpp_arg0);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[0], self);
        } else if (Shiboken::Converter<QGraphicsScene * >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return -1;
            QGraphicsScene* cpp_arg1 = Shiboken::Converter<QGraphicsScene* >::toCpp(pyargs[1]);
            // QGraphicsPixmapItem(QGraphicsItem*,QGraphicsScene*)
            cptr = new QGraphicsPixmapItemWrapper(cpp_arg0, cpp_arg1);
            //CppGenerator::writeParentChildManagement
            Shiboken::setParent(pyargs[0], self);
        } else goto SbkQGraphicsPixmapItem_Init_TypeError;
    } else goto SbkQGraphicsPixmapItem_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QGraphicsPixmapItem >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQGraphicsPixmapItem_Init_TypeError;

    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQGraphicsPixmapItem_Init_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem = None, PySide.QtGui.QGraphicsScene = None", "PySide.QtGui.QPixmap, PySide.QtGui.QGraphicsItem = None, PySide.QtGui.QGraphicsScene = None", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsPixmapItem", overloads);
        return -1;
}

static PyObject* SbkQGraphicsPixmapItemFunc_boundingRect(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // boundingRect()const
    QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
    QRectF cpp_result = cppSelf->QGraphicsPixmapItem::boundingRect();
    py_result = Shiboken::Converter<QRectF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsPixmapItemFunc_contains(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QPointF & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(arg);
        if (!SbkQPointF_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // contains(QPointF)const
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsPixmapItem::contains(*cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_contains_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQGraphicsPixmapItemFunc_contains_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPointF", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsPixmapItem.contains", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_extension(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QVariant & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QVariant > cpp_arg0_auto_ptr;
        QVariant* cpp_arg0 = Shiboken::Converter<QVariant* >::toCpp(arg);
        if (!SbkQVariant_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QVariant >(cpp_arg0);
        // extension(QVariant)const
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        QVariant cpp_result = cppSelf->QGraphicsPixmapItem::extension(*cpp_arg0);
        py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_extension_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQGraphicsPixmapItemFunc_extension_TypeError:
        const char* overloads[] = {"PySide.QtCore.QVariant", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsPixmapItem.extension", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_isObscuredBy(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QGraphicsItem * >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        QGraphicsItem* cpp_arg0 = Shiboken::Converter<QGraphicsItem* >::toCpp(arg);
        // isObscuredBy(const QGraphicsItem*)const
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsPixmapItem::isObscuredBy(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_isObscuredBy_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQGraphicsPixmapItemFunc_isObscuredBy_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsPixmapItem.isObscuredBy", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_offset(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // offset()const
    QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
    QPointF cpp_result = cppSelf->QGraphicsPixmapItem::offset();
    py_result = Shiboken::Converter<QPointF >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsPixmapItemFunc_opaqueArea(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // opaqueArea()const
    QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
    QPainterPath cpp_result = cppSelf->QGraphicsPixmapItem::opaqueArea();
    py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsPixmapItemFunc_paint(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "paint", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && Shiboken::Converter<QPainter * >::isConvertible(pyargs[0]) && Shiboken::Converter<QStyleOptionGraphicsItem * >::isConvertible(pyargs[1]) && Shiboken::Converter<QWidget * >::isConvertible(pyargs[2])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        QPainter* cpp_arg0 = Shiboken::Converter<QPainter* >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        QStyleOptionGraphicsItem* cpp_arg1 = Shiboken::Converter<QStyleOptionGraphicsItem* >::toCpp(pyargs[1]);
        if (Shiboken::cppObjectIsInvalid(pyargs[2]))
            return 0;
        QWidget* cpp_arg2 = Shiboken::Converter<QWidget* >::toCpp(pyargs[2]);
        // paint(QPainter*,const QStyleOptionGraphicsItem*,QWidget*)
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsPixmapItem::paint(cpp_arg0, cpp_arg1, cpp_arg2);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_paint_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsPixmapItemFunc_paint_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPainter, PySide.QtGui.QStyleOptionGraphicsItem, PySide.QtGui.QWidget", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsPixmapItem.paint", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_pixmap(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // pixmap()const
    QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
    QPixmap cpp_result = cppSelf->QGraphicsPixmapItem::pixmap();
    py_result = Shiboken::Converter<QPixmap >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsPixmapItemFunc_setExtension(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setExtension", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && Shiboken::Converter<QGraphicsItem::Extension >::isConvertible(pyargs[0]) && Shiboken::Converter<QVariant & >::isConvertible(pyargs[1])) {
        QGraphicsItem::Extension cpp_arg0 = Shiboken::Converter<QGraphicsItem::Extension >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QVariant > cpp_arg1_auto_ptr;
        QVariant* cpp_arg1 = Shiboken::Converter<QVariant* >::toCpp(pyargs[1]);
        if (!SbkQVariant_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QVariant >(cpp_arg1);
        // setExtension(QGraphicsItem::Extension,QVariant)
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsPixmapItem::setExtension(cpp_arg0, *cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_setExtension_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsPixmapItemFunc_setExtension_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem.Extension, PySide.QtCore.QVariant", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsPixmapItem.setExtension", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_setOffset(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "setOffset", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1])) {
        qreal cpp_arg0 = Shiboken::Converter<qreal >::toCpp(pyargs[0]);
        qreal cpp_arg1 = Shiboken::Converter<qreal >::toCpp(pyargs[1]);
        // setOffset(qreal,qreal)
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsPixmapItem::setOffset(cpp_arg0, cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else if (numArgs == 1 && Shiboken::Converter<QPointF & >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return 0;
        std::auto_ptr<QPointF > cpp_arg0_auto_ptr;
        QPointF* cpp_arg0 = Shiboken::Converter<QPointF* >::toCpp(pyargs[0]);
        if (!SbkQPointF_Check(pyargs[0]))
            cpp_arg0_auto_ptr = std::auto_ptr<QPointF >(cpp_arg0);
        // setOffset(QPointF)
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsPixmapItem::setOffset(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_setOffset_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsPixmapItemFunc_setOffset_TypeError:
        const char* overloads[] = {"PySide.QtCore.QPointF", "float, float", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtGui.QGraphicsPixmapItem.setOffset", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_setPixmap(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QPixmap & >::isConvertible(arg)) {
        if (Shiboken::cppObjectIsInvalid(arg))
            return 0;
        std::auto_ptr<QPixmap > cpp_arg0_auto_ptr;
        QPixmap* cpp_arg0 = Shiboken::Converter<QPixmap* >::toCpp(arg);
        if (!SbkQPixmap_Check(arg))
            cpp_arg0_auto_ptr = std::auto_ptr<QPixmap >(cpp_arg0);
        // setPixmap(QPixmap)
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsPixmapItem::setPixmap(*cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_setPixmap_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsPixmapItemFunc_setPixmap_TypeError:
        const char* overloads[] = {"PySide.QtGui.QPixmap", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsPixmapItem.setPixmap", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_setShapeMode(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<QGraphicsPixmapItem::ShapeMode >::isConvertible(arg)) {
        QGraphicsPixmapItem::ShapeMode cpp_arg0 = Shiboken::Converter<QGraphicsPixmapItem::ShapeMode >::toCpp(arg);
        // setShapeMode(QGraphicsPixmapItem::ShapeMode)
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsPixmapItem::setShapeMode(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_setShapeMode_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsPixmapItemFunc_setShapeMode_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsPixmapItem.ShapeMode", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsPixmapItem.setShapeMode", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_setTransformationMode(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (Shiboken::Converter<Qt::TransformationMode >::isConvertible(arg)) {
        Qt::TransformationMode cpp_arg0 = Shiboken::Converter<Qt::TransformationMode >::toCpp(arg);
        // setTransformationMode(Qt::TransformationMode)
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        cppSelf->QGraphicsPixmapItem::setTransformationMode(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_setTransformationMode_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQGraphicsPixmapItemFunc_setTransformationMode_TypeError:
        const char* overloads[] = {"PySide.QtCore.Qt.TransformationMode", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsPixmapItem.setTransformationMode", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_shape(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // shape()const
    QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
    QPainterPath cpp_result = cppSelf->QGraphicsPixmapItem::shape();
    py_result = Shiboken::Converter<QPainterPath >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsPixmapItemFunc_shapeMode(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // shapeMode()const
    QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
    QGraphicsPixmapItem::ShapeMode cpp_result = cppSelf->QGraphicsPixmapItem::shapeMode();
    py_result = Shiboken::Converter<QGraphicsPixmapItem::ShapeMode >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsPixmapItemFunc_supportsExtension(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QGraphicsItem::Extension >::isConvertible(arg)) {
        QGraphicsItem::Extension cpp_arg0 = Shiboken::Converter<QGraphicsItem::Extension >::toCpp(arg);
        // supportsExtension(QGraphicsItem::Extension)const
        QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QGraphicsPixmapItem::supportsExtension(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQGraphicsPixmapItemFunc_supportsExtension_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQGraphicsPixmapItemFunc_supportsExtension_TypeError:
        const char* overloads[] = {"PySide.QtGui.QGraphicsItem.Extension", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtGui.QGraphicsPixmapItem.supportsExtension", overloads);
        return 0;
}

static PyObject* SbkQGraphicsPixmapItemFunc_transformationMode(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // transformationMode()const
    QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
    Qt::TransformationMode cpp_result = cppSelf->QGraphicsPixmapItem::transformationMode();
    py_result = Shiboken::Converter<Qt::TransformationMode >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQGraphicsPixmapItemFunc_type(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // type()const
    QGraphicsPixmapItem* cppSelf = Shiboken::Converter<QGraphicsPixmapItem* >::toCpp((PyObject*)self);
    int cpp_result = cppSelf->QGraphicsPixmapItem::type();
    py_result = Shiboken::Converter<int >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyMethodDef SbkQGraphicsPixmapItem_methods[] = {
    {"boundingRect", (PyCFunction)SbkQGraphicsPixmapItemFunc_boundingRect, METH_NOARGS},
    {"contains", (PyCFunction)SbkQGraphicsPixmapItemFunc_contains, METH_O},
    {"extension", (PyCFunction)SbkQGraphicsPixmapItemFunc_extension, METH_O},
    {"isObscuredBy", (PyCFunction)SbkQGraphicsPixmapItemFunc_isObscuredBy, METH_O},
    {"offset", (PyCFunction)SbkQGraphicsPixmapItemFunc_offset, METH_NOARGS},
    {"opaqueArea", (PyCFunction)SbkQGraphicsPixmapItemFunc_opaqueArea, METH_NOARGS},
    {"paint", (PyCFunction)SbkQGraphicsPixmapItemFunc_paint, METH_VARARGS},
    {"pixmap", (PyCFunction)SbkQGraphicsPixmapItemFunc_pixmap, METH_NOARGS},
    {"setExtension", (PyCFunction)SbkQGraphicsPixmapItemFunc_setExtension, METH_VARARGS},
    {"setOffset", (PyCFunction)SbkQGraphicsPixmapItemFunc_setOffset, METH_VARARGS},
    {"setPixmap", (PyCFunction)SbkQGraphicsPixmapItemFunc_setPixmap, METH_O},
    {"setShapeMode", (PyCFunction)SbkQGraphicsPixmapItemFunc_setShapeMode, METH_O},
    {"setTransformationMode", (PyCFunction)SbkQGraphicsPixmapItemFunc_setTransformationMode, METH_O},
    {"shape", (PyCFunction)SbkQGraphicsPixmapItemFunc_shape, METH_NOARGS},
    {"shapeMode", (PyCFunction)SbkQGraphicsPixmapItemFunc_shapeMode, METH_NOARGS},
    {"supportsExtension", (PyCFunction)SbkQGraphicsPixmapItemFunc_supportsExtension, METH_O},
    {"transformationMode", (PyCFunction)SbkQGraphicsPixmapItemFunc_transformationMode, METH_NOARGS},
    {"type", (PyCFunction)SbkQGraphicsPixmapItemFunc_type, METH_NOARGS},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQGraphicsPixmapItem_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtGui.QGraphicsPixmapItem",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQGraphicsPixmapItem_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             0,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQGraphicsPixmapItem_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QGraphicsPixmapItem >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static PyGetSetDef SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "ShapeMode",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


PyAPI_FUNC(void) init_QGraphicsPixmapItem(PyObject* module)
{
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSPIXMAPITEM_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQGraphicsPixmapItem_Type);

    SbkQGraphicsPixmapItem_Type.super.ht_type.tp_base = SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_IDX];
    // Fill type discovery information
    SbkQGraphicsPixmapItem_Type.type_discovery = reinterpret_cast<SbkBaseWrapperType*>(SbkPySide_QtGuiTypes[SBK_QGRAPHICSITEM_IDX])->type_discovery;

    if (PyType_Ready((PyTypeObject*)&SbkQGraphicsPixmapItem_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQGraphicsPixmapItem_Type));
    PyModule_AddObject(module, "QGraphicsPixmapItem",
        ((PyObject*)&SbkQGraphicsPixmapItem_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: ShapeMode
    SbkPySide_QtGuiTypes[SBK_QGRAPHICSPIXMAPITEM_SHAPEMODE_IDX] = &SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type);
    PyDict_SetItemString(SbkQGraphicsPixmapItem_Type.super.ht_type.tp_dict,
            "ShapeMode",((PyObject*)&SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type,
        (long) QGraphicsPixmapItem::MaskShape, "MaskShape");
    PyDict_SetItemString(SbkQGraphicsPixmapItem_Type.super.ht_type.tp_dict,
        "MaskShape", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type.tp_dict,
        "MaskShape", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type,
        (long) QGraphicsPixmapItem::BoundingRectShape, "BoundingRectShape");
    PyDict_SetItemString(SbkQGraphicsPixmapItem_Type.super.ht_type.tp_dict,
        "BoundingRectShape", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type.tp_dict,
        "BoundingRectShape", enum_item);
    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type,
        (long) QGraphicsPixmapItem::HeuristicMaskShape, "HeuristicMaskShape");
    PyDict_SetItemString(SbkQGraphicsPixmapItem_Type.super.ht_type.tp_dict,
        "HeuristicMaskShape", enum_item);
    PyDict_SetItemString(SbkPySide_QtGui_QGraphicsPixmapItem_ShapeMode_Type.tp_dict,
        "HeuristicMaskShape", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QGraphicsPixmapItem::ShapeMode");

    Shiboken::TypeResolver::createObjectTypeResolver<QGraphicsPixmapItem >("QGraphicsPixmapItem*");
    Shiboken::TypeResolver::createObjectTypeResolver<QGraphicsPixmapItem >(typeid(QGraphicsPixmapItem).name());
}


} // extern "C"

