/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#include <QtCore/QString>
#include <QtCore/QBool>

#include "boost_headers.hpp"
#include "type_converter.hpp"
#include "type_manager.hpp"

using namespace boost;

namespace PySide
{

/************************* QString ****************************/
struct Py2QString
{
    static void*
    convertible(PyObject* py_obj)
    {
        if (!(PyString_Check(py_obj) || PyUnicode_Check(py_obj)) )
            return 0;
        return py_obj;
    }

    static void
    construct(PyObject* py_obj,
              python::converter::rvalue_from_python_stage1_data* data)
    {
        typedef python::converter::rvalue_from_python_storage<QString> storage_t;
        storage_t* the_storage = reinterpret_cast<storage_t*>(data);
        void* memory_chunk = the_storage->storage.bytes;

        if (PyString_Check(py_obj)) {
            const char* value = PyString_AsString(py_obj);
            new (memory_chunk) QString(value);
        } else {
            Py_ssize_t size = PyUnicode_GetSize(py_obj);
            wchar_t* ws = new wchar_t[size];
            PyUnicode_AsWideChar((PyUnicodeObject*)py_obj, ws, size);
            new (memory_chunk) QString(QString::fromWCharArray(ws, size));
            delete ws;
        }

        data->convertible = memory_chunk;
    }
};

void
register_qstring_converter()
{
    python::converter::registry::push_back(&Py2QString::convertible,
                                           &Py2QString::construct,
                                           python::type_id<QString>());
}

/************************* QBool ****************************/

struct PYSIDE_LOCAL QBool2Py
{
    static PyObject*
    convert(const QBool& qbool)
    {
        return python::incref(boost::python::object(qbool==true).ptr());
    }
};


void
register_qbool_converter()
{
    python::to_python_converter<QBool, QBool2Py>();
}

/************************* QByteArray ****************************/
struct Py2QByteArray
{
    static void*
    convertible(PyObject* py_obj)
    {
        if (!(PyString_Check(py_obj)) )
            return 0;
        return py_obj;
    }

    static void
    construct(PyObject* py_obj,
              python::converter::rvalue_from_python_stage1_data* data)
    {
        typedef python::converter::rvalue_from_python_storage<QByteArray> storage_t;
        storage_t* the_storage = reinterpret_cast<storage_t*>(data);
        void* memory_chunk = the_storage->storage.bytes;

        const char* value = 0;
        std::size_t value_size;
        if (PyString_Check(py_obj)) {
            value = PyString_AsString(py_obj);
            value_size = PyString_Size(py_obj);
        }

        if (value) {
            new (memory_chunk) QByteArray(value, value_size);
            data->convertible = memory_chunk;
        }
    }
};

void
register_qbytearray_converter()
{
    python::converter::registry::push_back (&Py2QByteArray::convertible,
                                            &Py2QByteArray::construct,
                                            python::type_id<QByteArray>());
}

} // namespace PySide

