/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __SIGNALMANAGER_H__
#define __SIGNALMANAGER_H__

#include <QtCore/QHash>
#include <QtCore/QList>
#include <QtCore/QByteArray>
#include "pyside_global.hpp"
#include "boost_headers.hpp"
#include <QMap>

class QObject;

namespace PySide
{

class signal_holder;
class pyqt_signal;
class pyqt_slot;
class trigger;

struct signal_manager_data;

/**
 *   This singleton class manage all signal/slot connections made by the python
 *   code or by C++ code involving a Python object.
 *   It also take care of dynamic signals (pure python signals).
 *
 *   Every object that has a signal/slot connection managed by this class has
 *   an associated trigger, the trigger implements the internal stuff to call
 *   the right methods at the right time. So what this class do when connecting
 *   signals to slots/callables is:
 *   <ul>
 *       <li>If the emitter and the receiver are C++ objects,
 *       connect then using QMetaObject.</li>
 *       <li>Else, find the object associated trigger or create
 *       one if we can't find one and delegate the job to the trigger.</li>
 *   </ul>
 *   \see trigger
 */
class PYSIDE_API signal_manager
{
public:
    /// Returns the signal_manager instance.
    static signal_manager& instance();

    /// connects a signal to a python callable object.
    bool connect(QObject* src, const pyqt_signal& signal,
                 boost::python::object& callable);
    /// connects a signal to a qobject slot.
    bool connect(QObject* sender, const pyqt_signal& signal,
                 QObject* receiver, const pyqt_slot& slot);
    /// disconnects a python callable from a signal.
    bool disconnect(QObject* src, const pyqt_signal& signal,
                    boost::python::object& callable);
    /// disconnects a signal from a object slot.
    bool disconnect(QObject* sender, const pyqt_signal& signal,
                    QObject* receiver,
                    const pyqt_slot& slot);

    int dynamicsignal_receivers(const QObject *sender,
                                const pyqt_signal &signal,
                                int offset) const;
    QList<QByteArray> signals_of(const QObject* obj);

    /**
     *   Register a dynamic signal.
     *   \param src the signal emitter.
     *   \param signal the dynamic signal.
     *   \param trigger the trigger of the object that this signal
     *   is connected on.
     *   \param slot_id the slot id to identify the slot
     */
    void register_dynamic_signal(QObject *src, const pyqt_signal& signal,
                                 trigger* trigger, int slot_id);

    /**
     *   Create and associate a callback to a qobject and slot name.
     *   \param parent the parent object to control slot life-time.
     *   \param callback the python::object to be called when the slot is activated.
     *   \param slot_nome a dynamic name created associated to callback
     */
    QObject* register_dynamic_slot(QObject *parent,
                                     const boost::python::object &callback,
                                     QString &slot_name);
    const QMetaObject* get_dynamic_metaobject(QObject* obj);
    /// Unregister the object, removing all connections
    /// where it is a emitter or a receiver.
    void unregister_qobj(QObject* obj);
    /// Emit the \p signal from \p src with \p args arguments,
    /// where \p args is a boost::python list.
    void emit_(QObject *src, const pyqt_signal& signal,
               const boost::python::object& args);

    ~signal_manager();

    void register_sender(QObject* sender);
    void unregister_sender();
    QObject* sender();

private:
    signal_manager_data *m_data;

    trigger* find_trigger(QObject* src);
    trigger* get_trigger(const QObject *src) const;
    void emit_dynamic_signal(QObject *source, const pyqt_signal& signal,
                             const boost::python::object& args);
    void parse_objects_to_qarguments(const QStringList &args_type,
                                     const boost::python::object &argv,
                                     QList<QGenericArgument> &q_args);

    // disable copy
    signal_manager();
    signal_manager(const signal_manager&);
    signal_manager& operator=(const signal_manager&);
};

class PYSIDE_LOCAL set_sender
{
    signal_manager& m_sm;

public:
    set_sender(QObject* sender) : m_sm(signal_manager::instance())
    {
        m_sm.register_sender(sender);
    }

    ~set_sender()
    {
        m_sm.unregister_sender();
    }
};

} // namespace PySide

#endif

