/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __QPTR_HPP__
#define __QPTR_HPP__

#include "boost_headers.hpp"
#include "pyside_global.hpp"
#include "qptr_cleanup.hpp"

#include <QHash>
#include <QDebug>

namespace PySide
{

/**
 *   This class is used to acess qptr without templates
 */
class PYSIDE_API qptr_base
{
public:
    enum construction_mode {
        check_cache,
        avoid_cache
    };

    qptr_base(void* cpp_obj, void (*deleter)(void*), construction_mode mode);
    qptr_base(const qptr_base& other);
    qptr_base() : m_data(0) {}
    virtual ~qptr_base();

    /**
     *   Clear all reference control
     *   Used when C++ object was deleted, this avoid python to
     *   delete the same C++ object twice.
     */
    static void invalidate(void *cpp_obj, bool cleanup = false);
    static void cleanup(void *cpp_obj);
    static bool exists(void *cpp_obj);

    bool is_null() const;

    qptr_base& operator=(const qptr_base& other);

    bool
    operator==(const qptr_base& other)
    {
        return m_data == other.m_data;
    }

    // Python VM will acquire the C++ object ownership
    void acquire_ownership();
    void release_ownership();
    bool has_ownership();

    //c++ related ref
    void add_cpp_ref();
    void remove_cpp_ref();
    bool has_cpp_ref();

    void add_child(qptr_base &child);
    void remove_child(qptr_base &child);
    void remove_parent();

    PyObject* get_pyobject();
    void set_pyobject(PyObject *py_obj);

    void release();

    // debug methods
    bool has_parent();
    int children_count();
    int refcount();
    static inline int list_size() { return m_cpp_to_qptrs.size(); }

    void* raw_ptr() const;

protected:
    bool ptr_assert() const;
    void destroy();
    void invalidate();
    void invalidate(bool invalidate_ptr);

    class qptr_base_private;
    mutable qptr_base_private* m_data;
    static QHash<void*, qptr_base_private*> m_cpp_to_qptrs;
    explicit qptr_base(qptr_base::qptr_base_private* data);
    qptr_base_private* create_copy() const;
};

//override this function to implement generic deleter
template<typename T>
inline PYSIDE_LOCAL
void delete_pointer(T *p)
{
    delete p;
}

template<typename T>
inline PYSIDE_LOCAL
void delete_pointer_helper(void* p)
{
    T *ptr = reinterpret_cast<T*>(p);
    delete_pointer(ptr);
}

/**
 * A smart pointer used by boost.python to create a compatibility between Qt and python objects
 */
template<class T,
         qptr_base::construction_mode c_mode  = qptr_base::check_cache>
class PYSIDE_LOCAL qptr : public qptr_base
{
public:
    typedef T element_type;
    typedef T value_type;
    typedef T * pointer;

public:

    qptr(T* ptr, qptr_base::construction_mode mode = c_mode)
        : qptr_base(ptr, &delete_pointer_helper<T>, mode)
    {
    }

    template<typename Y>
    qptr(const qptr<Y>& ptr) : qptr_base(ptr)
    {
    }

    qptr(T* ptr, PyObject* py_obj)
        : qptr_base(ptr, &delete_pointer_helper<T>, avoid_cache)
    {
        set_pyobject(py_obj);
    }

    qptr(PyObject* py_obj)
        : qptr_base(boost::python::extract<T*>(py_obj),
                    &delete_pointer_helper<T>, c_mode)
    {
        set_pyobject(py_obj);
    }

    template<typename Y>
    qptr&
    operator=(const qptr<Y>& ptr)
    {
        if ( m_data != ptr.m_data ) {
            release();
            m_data = ptr.create_copy();
        }
        return *this;
    }

    inline T&
    operator*() const
    {
        return *get();
    }

    inline T*
    operator->() const
    {
        return get();
    }

    inline T*
    get() const
    {
        ptr_assert();
        return reinterpret_cast<T*>(raw_ptr());
    }

    bool is_wrapper()
    {
        return is_wrapper_impl(raw_ptr(), boost::is_polymorphic<T>());
    }

    ~qptr()
    {
        if (!is_null() && (refcount() == 1) && has_ownership())
        {
            T* pointer = reinterpret_cast<T*>(raw_ptr());
            ::cleanup_pointer(pointer);
        }
    }

private:
    bool is_wrapper_impl(void *, boost::mpl::false_)
    {
        return false;
    }

    bool is_wrapper_impl(void *pointer, boost::mpl::true_)
    {
        if (pointer) {
            T* data = reinterpret_cast<T*>(pointer);
            return dynamic_cast<boost::python::detail::wrapper_base*>(data);
        } else {
            return false;
        }
    }
};

// "get_pointer" function returns pointer to the object managed by smart pointer
// class instance

template<class T, qptr_base::construction_mode mode>
inline PYSIDE_LOCAL T*
get_pointer(qptr<T, mode> const& p){
    return p.get();
}

} // namespace PySide

#endif
