import sys
import imp
import types
import ctypes

'''
Import hooks for PySide bindings.

Most of the finder/loader code was based on [1].

[1] http://code.activestate.com/recipes/473888/
'''

class Finder(object):
    '''PySide Finder class

    Filters the imports, using our loader only for our own modules'''

    def find_module(self, fullname, path=None):
        origName = fullname
        if not path: # Is toplevel module?
            mod = sys.modules.get(fullname, False)
            if mod is None or mod and isinstance(mod, types.ModuleType):
                return mod

        if '.' in fullname:
            head, fullname = fullname.rsplit('.', 1)

            mod = sys.modules.get(head, None)
            if mod is None:
                return

            if hasattr(mod, '__path__'):
                path = mod.__path__

        if not ('PySide' in fullname or (path and 'PySide' in path[0])):
            return

        try:
            filehandle, pathname, desc = imp.find_module(fullname, path)
            return Loader(origName, filehandle, pathname, desc)
        except ImportError:
            return None

class Loader(object):
    '''Loader class for PySide modules.'''
    def __init__(self, name, filehandle, pathname, desc):
        self.name = name
        self.filehandle = filehandle
        self.pathname = pathname
        self.desc = desc

    def load_module(self, fullname):
        # Preload with RTLD_GLOBAL so the dependencies can be searched
        # for definitions.
        ctypes.PyDLL(self.pathname, ctypes.RTLD_GLOBAL)
        mod = imp.load_module(self.name, self.filehandle,
                              self.pathname, self.desc)

        if self.filehandle:
            self.filehandle.close()
        sys.modules[fullname] = mod

        return mod

if not filter(lambda x: isinstance(x, Finder), sys.meta_path):
    sys.meta_path.append(Finder())


__all__ = ['QtCore', 'QtGui', 'QtHelp', 'QtNetwork', 'QtOpenGL', 'QtScript', 'QtScriptTools', 'QtSql', 'QtSvg', 'QtUiTools', 'QtWebKit', 'QtXml', 'QtXmlPatterns']
