/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "test_ownership_control.h"
#include "pyside.hpp"

#include <QtTest/QTest>
#include <QMetaMethod>
#include <QDebug>

using namespace boost::python;

void test_ownership_control::initTestCase()
{
    Py_Initialize();

    _main_module = import("__main__");
    _qt_module = import("PySide.QtCore");
    _qt_namespace = _qt_module.attr("__dict__");
    _main_namespace = _main_module.attr("__dict__");
}

void test_ownership_control::cleanupTestCase()
{
    Py_Finalize();
}

void test_ownership_control::test_parent_destroy()
{
    try {
        type_manager::instance(); // register type converters
        ownership_control *oc = ownership_control::instance();

        object child = eval("QObject()", _qt_namespace, _qt_namespace);

        QObject *child_data = extract<QObject*>(child);
        QObject *parent_data;
        PyObject *parent_py;

        qDebug() << "child: " << child_data;

        int initial_parent_size = oc->parent_count();

        {
            object parent = eval("QObject()", _qt_namespace, _qt_namespace);
            parent_py = parent.ptr();
            parent_data = extract<QObject*>(parent);

            oc->add<QObject*>(parent.ptr(), child.ptr());
            child_data->setParent(parent_data);

            QCOMPARE(oc->is_python_owner(parent_data), true);
            QCOMPARE(oc->is_cpp_owner(child_data), true);
        }

        qDebug() << "child deve morrer";

        QCOMPARE(oc->parent_count(), initial_parent_size);

        //child <-> parent
        QVERIFY(oc->find_parent(child.ptr()) == 0);

        //cpp is live
        QCOMPARE(oc->is_cpp_live(parent_data), false);
        QCOMPARE(oc->is_cpp_live(child_data), false);

        //ownership
        QCOMPARE(oc->is_cpp_owner(parent_data), true);
        QCOMPARE(oc->is_cpp_owner(child_data), true);

        QCOMPARE(oc->is_python_owner(parent_data), false);
        QCOMPARE(oc->is_python_owner(child_data), false);
    } catch(error_already_set&) {
        PyErr_Print();
        QFAIL("Python raised an error!");
    }
}


void test_ownership_control::test_add_remove()
{
    try {
        type_manager::instance(); // register type converters
        ownership_control *oc = ownership_control::instance();

        object parent = eval("QObject()", _qt_namespace, _qt_namespace);
        object child = eval("QObject()", _qt_namespace, _qt_namespace);

        QObject *parent_data = extract<QObject*>(parent);
        QObject *child_data = extract<QObject*>(child);

        int initial_parent_size = oc->parent_count();
        //object refcount;

        //child <-> parent
        QCOMPARE(oc->child_count(parent.ptr()), 0);
        QVERIFY(oc->find_parent(child.ptr()) == 0);

        //refcount
        QCOMPARE(parent.ptr()->ob_refcnt, 1);
        QCOMPARE(child.ptr()->ob_refcnt, 1);

        //ownership
        QCOMPARE(oc->is_python_owner(parent_data), true);
        QCOMPARE(oc->is_python_owner(child_data), true);
        QCOMPARE(oc->is_cpp_owner(parent_data), false);
        QCOMPARE(oc->is_cpp_owner(child_data), false);

        QCOMPARE(oc->is_cpp_live(parent_data), true);
        QCOMPARE(oc->is_cpp_live(child_data), true);

        oc->add<QObject*>(parent.ptr(), child.ptr());

        QCOMPARE(oc->parent_count(), initial_parent_size + 1);

        QCOMPARE(oc->child_count(parent.ptr()), 1);
        QCOMPARE(oc->find_parent(child.ptr()), parent.ptr());

        //refcount
        QCOMPARE(parent.ptr()->ob_refcnt, 1);
        QCOMPARE(child.ptr()->ob_refcnt, 2);

        //ownership
        QCOMPARE(oc->is_python_owner(parent_data), true);
        QCOMPARE(oc->is_python_owner(child_data), false);
        QCOMPARE(oc->is_cpp_owner(parent_data), false);
        QCOMPARE(oc->is_cpp_owner(child_data), true);

        oc->remove<QObject*>(parent.ptr(), child.ptr());

        QCOMPARE(oc->parent_count(), initial_parent_size);

        //child <-> parent
        QCOMPARE(oc->child_count(parent.ptr()), 0);
        QVERIFY(oc->find_parent(child.ptr()) == 0);

        //refcount
        QCOMPARE(parent.ptr()->ob_refcnt, 1);
        QCOMPARE(child.ptr()->ob_refcnt, 1);

        //ownership
        QCOMPARE(oc->is_python_owner(parent_data), true);
        QCOMPARE(oc->is_python_owner(child_data), true);
        QCOMPARE(oc->is_cpp_owner(parent_data), false);
        QCOMPARE(oc->is_cpp_owner(child_data), false);

        QCOMPARE(oc->is_cpp_live(parent_data), true);
        QCOMPARE(oc->is_cpp_live(child_data), true);
    } catch(error_already_set&) {
        PyErr_Print();
        QFAIL("Python raised an error!");
    }
}

QTEST_APPLESS_MAIN( test_ownership_control )
#include "test_ownership_control.moc"
