/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#ifndef TYPE_DETAILS_H
#define TYPE_DETAILS_H

#include <QtCore/QString>
#include <QtCore/QGenericArgument>

#include <boost_headers.hpp>
#include <pyside_global.hpp>
#include <parent_policy.hpp>

namespace PySide
{

class PYSIDE_LOCAL type_details
{
    typedef boost::python::object(*func_cpp_to_python_type)(void*);
    typedef QGenericArgument(*func_python_to_cpp_type)(const boost::python::object&, const char*);
    typedef void(*func_delete_type)(void*);

    // disable object copy
    type_details() {}
    type_details(const type_details&);
    type_details& operator=(const type_details&);

public:
    template<typename T>
    static type_details*
    create_object_type_details(const char* type_name) {
        type_details* self = new type_details();
        self->m_type_name = type_name;
        self->m_type_object = boost::python::converter::registry::query(boost::python::type_id<T>())->get_class_object();
        self->m_func_cpp_to_python = &objecttype_to_python<T>;
        self->m_func_python_to_cpp = &python_to_objecttype<T*>;
        self->m_func_delete = 0;
        return self;
    }

    template<typename T>
    static type_details*
    create_value_type_details(const char* type_name) {
        type_details* self = new type_details();
        self->m_type_name = type_name;
        self->m_type_object = boost::python::converter::registry::query(boost::python::type_id<T>())->get_class_object();
        self->m_func_cpp_to_python = &valuetype_to_python<T>;
        self->m_func_python_to_cpp = &python_to_value_type<T>;
        self->m_func_delete = &func_delete_data<T>;
        return self;
    }
    template<typename T>
    static type_details*
    create_native_type_details(const char* type_name) {
        type_details* self = new type_details();
        self->m_type_name = type_name;
        self->m_type_object = 0;
        self->m_func_cpp_to_python = &native_to_python<T>;
        self->m_func_python_to_cpp = &python_to_value_type<T>;
        self->m_func_delete = &func_delete_data<T>;
        return self;
    }

    template<typename T>
    static type_details*
    create_container_type_details(const char* type_name) {
        type_details* self = new type_details();
        self->m_type_name = type_name;
        self->m_type_object = 0;
        self->m_func_cpp_to_python = &container_to_python<T>;
        self->m_func_python_to_cpp = &python_to_container<T>;
        self->m_func_delete = &func_delete_data<T>;
        return self;
    }

    boost::python::object
    to_python(void *data) const
    {
        return m_func_cpp_to_python(data);
    }

    QGenericArgument
    to_cpp(const boost::python::object &obj) const
    {
        return m_func_python_to_cpp(obj, m_type_name);
    }

    const PyTypeObject*
    get_python_type_object() const
    {
        return m_type_object;
    }

    void
    delete_data(QGenericArgument &arg) const
    {
        if (m_func_delete)
            m_func_delete(arg.data());
    }

private:
    const char* m_type_name;
    const PyTypeObject *m_type_object;

    func_cpp_to_python_type m_func_cpp_to_python;
    func_python_to_cpp_type m_func_python_to_cpp;
    func_delete_type m_func_delete;

    /* Object-Type Convertion functions */
    template <class T>
    static boost::python::object
    objecttype_to_python(void* p)
    {
        T* obj = *(reinterpret_cast< T*(*)>(p));
        boost::python::object py_obj(PySide::ptr(obj));
        if (!py_obj.ptr())
        {
            fprintf(stderr, "Fail to create python object from object in adress: %p\n", obj);
            py_obj = boost::python::object();
        }
        else
        {
            boost::python::incref(py_obj.ptr());
        }

        return py_obj;
    }

    template <class T>
    static QGenericArgument
    python_to_objecttype(const boost::python::object &obj, const char *type_name)
    {
        T val = boost::python::extract<T>(obj);
        return QGenericArgument(type_name, val);
    }

    /* value type convertion functions */
    template <class T>
    static boost::python::object
    valuetype_to_python(void* p)
    {
        T* val = reinterpret_cast<T*>(p);
        return boost::python::object(val);
    }

    template <class T>
    static QGenericArgument
    python_to_value_type(const boost::python::object& obj, const char *type_name)
    {
        T* val = new T(boost::python::extract<T>(obj));
        return QGenericArgument(type_name, static_cast<const void*>(val));
    }

    template <typename T>
    static QGenericArgument
    python_to_container(const boost::python::object& obj, const char* type_name)
    {
        T* val = new T(boost::python::extract<T>(obj));
        return QGenericArgument(type_name, static_cast<const void*>(val));
    }

    template <typename T>
    static boost::python::object
    container_to_python(void* p)
    {
        return boost::python::object(reinterpret_cast<T*>(p));
    }

    template<typename T>
    static boost::python::object
    container_value_to_python(T t)
    {
        return valuetype_to_python<T>(&t);
    }

    template<typename T>
    static boost::python::object
    container_value_to_python(T* t)
    {
        return objecttype_to_python<T>(&t);
    }

    template <class T>
    static void
    func_delete_data(void *data)
    {
        delete reinterpret_cast<T*>(data);
    }

    template<typename T>
    static boost::python::object
    native_to_python(void* p)
    {
        T* val = new T(*reinterpret_cast<T*>(p));
        return boost::python::object(*val);
    }
};

} // namespace PySide

#endif

