/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef __QPTR_HPP__
#define __QPTR_HPP__

#include "boost_headers.hpp"
#include "pyside_global.hpp"
#include "wrapper.hpp"

#include <QHash>
#include <QDebug>
#include <typeinfo>

namespace PySide
{

/**
 *   This class is used to acess qptr without templates
 */
class PYSIDE_API qptr_base
{
public:
    enum construction_mode {
        none                    = 0x00000000,
        check_cache             = 0x00000001,
        no_check_cache          = 0x00000002,
        wrapper_pointer         = 0x00000004,
        no_wrapper_pointer      = 0x00000008
    };

    qptr_base(void* cpp_obj, PyObject *py_obj, void (*deleter)(void*), int mode);
    qptr_base(const qptr_base& other);
    qptr_base() : m_data(0) {}
    virtual ~qptr_base();

    /**
     *   Clear all reference control
     *   Used when C++ object was deleted, this avoid python to
     *   delete the same C++ object twice.
     */
    static void invalidate(void *cpp_obj, bool cleanup = false);
    static bool exists(void *cpp_obj);

    bool is_null() const;

    qptr_base& operator=(const qptr_base& other);

    bool
    operator==(const qptr_base& other)
    {
        return m_data == other.m_data;
    }

    // Python VM will acquire the C++ object ownership
    void acquire_ownership();
    void release_ownership();
    bool has_ownership();

    //c++ related ref
    bool is_wrapper() const;
    void add_cpp_ref();
    void remove_cpp_ref();
    bool has_cpp_ref();

    void add_child(qptr_base &child);
    void remove_child(qptr_base &child);
    void remove_parent();

    PyObject* get_pyobject();
    void set_pyobject(PyObject *py_obj);

    void release();

    // debug methods
    bool has_parent();
    int children_count();
    int refcount();
    static inline int list_size() { return m_cpp_to_qptrs.size(); }

    void* raw_ptr() const;

protected:
    bool ptr_assert() const;
    void destroy();
    void invalidate();
    void invalidate(bool invalidate_ptr);

    class qptr_base_private;
    mutable qptr_base_private* m_data;
    static QHash<const volatile void*, qptr_base_private*> m_cpp_to_qptrs;
    explicit qptr_base(qptr_base::qptr_base_private* data);
    qptr_base_private* create_copy() const;
};

//override this function to implement generic deleter
template<typename T>
inline PYSIDE_LOCAL
void delete_pointer(T *p)
{
    delete p;
}

template<typename T>
inline PYSIDE_LOCAL
void delete_pointer_helper(void* p)
{
    T *ptr = reinterpret_cast<T*>(p);
    delete_pointer(ptr);
}

/**
 * A smart pointer used by boost.python to create a compatibility between Qt and python objects
 */
template<class T,
         int c_mode  = (int) qptr_base::check_cache>
class PYSIDE_LOCAL qptr : public qptr_base
{
public:
    typedef T element_type;
    typedef T value_type;
    typedef T * pointer;

public:
    qptr(T* ptr, int mode = c_mode)
        : qptr_base(get_void_pointer(ptr),
                    (mode & wrapper_pointer ? extract_pyobject(ptr, boost::is_polymorphic<T>()) : 0 ),
                    &delete_pointer_helper<T>, mode)
    {
    }

    template<typename Y>
    qptr(const qptr<Y>& ptr) : qptr_base(ptr)
    {
    }

    qptr(T* ptr, PyObject* py_obj)
        : qptr_base(get_void_pointer(ptr), py_obj, &delete_pointer_helper<T>, no_check_cache | no_wrapper_pointer)
    {
    }

    qptr(PyObject* py_obj)
        : qptr_base(get_void_pointer(boost::python::extract<T*>(py_obj)), py_obj,
                    &delete_pointer_helper<T>, c_mode)
    {
    }

    template<typename Y>
    qptr&
    operator=(const qptr<Y>& ptr)
    {
        if ( m_data != ptr.m_data ) {
            release();
            m_data = ptr.create_copy();
        }
        return *this;
    }

    inline T&
    operator*() const
    {
        return *get();
    }

    inline T*
    operator->() const
    {
        return get();
    }

    inline T*
    get() const
    {
        ptr_assert();
        return reinterpret_cast<T*>(raw_ptr());
    }

    ~qptr()
    {
        if (!is_null() && refcount() == 1 && has_cpp_ref()) {
            notify_ptr_del(reinterpret_cast<T*>(raw_ptr()), boost::is_polymorphic<T>());
        }
    }

    static inline void* get_void_pointer(T* pointer)
    {
        return get_void_pointer(pointer, boost::is_polymorphic<T>());
    }

private:
    static inline PyObject*
    extract_pyobject(T* ptr,  boost::mpl::false_)
    {
        return 0;
    }

    static inline PyObject*
    extract_pyobject(T* ptr,  boost::mpl::true_)
    {
        PySide::wrapper *w = dynamic_cast<PySide::wrapper*>(ptr);
        if (w)
            return w->py_object();
        else
            return 0;
    }

    static inline void
    notify_ptr_del(T* ptr,  boost::mpl::false_)
    {
    }

    static inline void
    notify_ptr_del(T* ptr,  boost::mpl::true_)
    {
        PySide::wrapper *w = dynamic_cast<PySide::wrapper*>(ptr);
        if (w)
            w->keep_cpp_ref();
    }

    static inline void*
    get_void_pointer(T* ptr, boost::mpl::false_)
    {
        return static_cast<void*>(ptr);
    }

    static inline void*
    get_void_pointer(T* ptr, boost::mpl::true_)
    {
        boost::python::objects::dynamic_id_t base_id =
            boost::python::objects::polymorphic_id_generator<T>::execute(ptr);
        return base_id.first;
    }

};

// "get_pointer" function returns pointer to the object managed by smart pointer
// class instance

template<class T, int mode>
inline PYSIDE_LOCAL T*
get_pointer(qptr<T, mode> const& p){
    return p.get();
}

} // namespace PySide

#endif
