# -*- coding: UTF8 -*-
import gtk
try:
    import hildon
    HILDON = True
except:
    HILDON = False
import gobject
import os
from xml.dom import minidom
import datetime
import pyrecipe_utils

#See if is running ins scratchbox, and get the directories path in consequence
#FIXME: if run in the desktop this exists too
if os.path.exists('/scratchbox'):
    scratchbox = True
else:
    scratchbox = False
    
if scratchbox:
    imagesdir = icondir = 'pixmaps/'
    glade_file = "shopping_list.glade" 
else:
    imagesdir = '/usr/share/pyrecipe/pixmaps/'
    glade_file = "/usr/share/pyrecipe/shopping_list.glade"  

(ITEM, AMOUNT) = range(2)

class ShopGui():
    def __init__(self):
        glade=gtk.glade.XML(glade_file)
        if HILDON:
            self.window = hildon.Window()
            vbox1 = glade.get_widget("vbox1")
            vbox1.reparent(self.window)            
        else:
            self.window = glade.get_widget("window1")
            
        self.window.set_default_size(800, 480)
        self.window.set_title('Pyrecipe - Shopping List')
        self.window.connect("key-press-event", self.on_key_press)
        self.window.connect("window-state-event", self.on_window_state_change)
        self.window.connect("delete-event", self.hide_window)
        
        self.window_in_fullscreen = False

        ## Create menu  ##
        #get menu from glade and reparent as common menu in hildon program
        if HILDON:
            mainMenu = glade.get_widget("menubar1")
            menu = gtk.Menu()
            for child in mainMenu.get_children():
                child.reparent(menu)
            self.window.set_menu(menu)
            mainMenu.destroy()

        shop_list_sw = glade.get_widget("shop_list_sw")
        if HILDON:
            hildon.hildon_helper_set_thumb_scrollbar(shop_list_sw, True)
        
        # create tree model
        shop_list_model = self.create_shop_list_model()
        shop_list_tv = glade.get_widget("shop_list_tv")        
        shop_list_tv.set_model(shop_list_model)
        shop_list_tv.set_rules_hint(True)
        shop_list_tv.set_headers_visible(True)
       
        # add columns to the tree view
        self.add_columns_to_shop_list(shop_list_tv)
        
        self.amount_entry = glade.get_widget("amount_entry")
        self.unit_entry = glade.get_widget("unit_entry")
        self.item_entry = glade.get_widget("item_entry")

        add_items_toolbar = glade.get_widget("add_items_toolbar")
        
        add_item_btn = glade.get_widget("add_item_btn")
        add_item_btn.connect("toggled", self.show_add_toolbar, add_items_toolbar)

        add_button = glade.get_widget("add_button")
        add_button.connect("clicked", self.add_item, shop_list_model)
        cancel_button = glade.get_widget("cancel_button")
        
        remove_item_btn = glade.get_widget("remove_item_btn")
        remove_item_btn.connect("clicked", self.remove_item, shop_list_tv)

        clear_shop_list_btn = glade.get_widget("clear_shop_list_btn")
        clear_shop_list_btn.connect("clicked", self.clear_shop_list, shop_list_model)

        mark_btn = glade.get_widget("mark_btn")
        mark_btn.connect("clicked", self.mark_item_as_bought, shop_list_tv, shop_list_model)

        open_menu = glade.get_widget("open_menu")
        open_menu.connect("activate", self.open_shopping_list, shop_list_model)
        save_menu = glade.get_widget("save_menu")    
        save_menu.connect("activate", self.save_shopping_list, shop_list_model)

        # Set the images for the buttons/menu
        remove_item_img = glade.get_widget("remove_item_img")
        remove_item_img.set_from_file(imagesdir + 'remove.png')
        clear_shop_list_img = glade.get_widget("clear_shop_list_img")
        clear_shop_list_img.set_from_file(imagesdir + 'clear.png')
        add_item_toolbar_img = glade.get_widget("add_item_toolbar_img")
        add_item_toolbar_img.set_from_file(imagesdir + 'add.png')
        mark_bought_img = glade.get_widget("mark_bought_img")
        mark_bought_img.set_from_file(imagesdir + 'mark.png')
        add_item_img = glade.get_widget("add_item_img")
        add_item_img.set_from_file(imagesdir + 'add.png')        
        open_menu_img = glade.get_widget("open_menu_img")
        open_menu_img.set_from_file(imagesdir + 'import.png')        
        save_menu_img = glade.get_widget("save_menu_img")
        save_menu_img.set_from_file(imagesdir + 'save.png')

        self.window.show_all()

    #Show the window if it's hidden.
    def show(self):
        self.window.show()
        self.window.present()
        return True

    #Hide the window, when closing it
    def hide_window(self, widget, event=None):
        self.window.hide()
        return True

    #Functions for fullscreen
    def on_window_state_change(self, widget, event, *args):           
        if event.new_window_state & gtk.gdk.WINDOW_STATE_FULLSCREEN:
            self.window_in_fullscreen = True
        else:
            self.window_in_fullscreen = False

    #Key press function for the F6 key(fullscreen)
    def on_key_press(self, widget, event, *args):
        if HILDON:         
            if event.keyval == gtk.keysyms.F6:
                if self.window_in_fullscreen:
                    self.window.unfullscreen ()
                else:
                    self.window.fullscreen ()
                
    def create_shop_list_model(self):
        lstore = gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_STRING,\
                               'gboolean')
        return lstore 
           
    def edit_treeview(self, renderer, row, text, model, col):
        model[int(row)][col]=text 
       
    def add_columns_to_shop_list(self, treeview):
        model = treeview.get_model()

        # column for shopping items 
        renderer = gtk.CellRendererText()
        renderer.set_property('editable', True)
        renderer.connect('edited', self.edit_treeview, model, 0)
        renderer.set_property('strikethrough', False) 
        renderer.set_property('scale', 1.3)
        renderer.set_property('wrap-mode',gtk.WRAP_WORD)
        renderer.set_property('wrap-width',350)
        column = gtk.TreeViewColumn('Item', renderer, text=ITEM, \
                                    strikethrough=2)
        column.set_property("expand", True)       
        treeview.append_column(column)

        # column for amount
        renderer = gtk.CellRendererText()
        renderer.set_property('editable', True)
        renderer.connect('edited', self.edit_treeview, model, 1)        
        renderer.set_property('strikethrough', False)        
        renderer.set_property('scale', 1.3)
        column = gtk.TreeViewColumn('Amount', renderer, text=AMOUNT, \
                                   strikethrough =2)
        column.set_property("expand", True)
        treeview.append_column(column)        

    #Add a new item to the shopping list                
    def add_item(self, widget, shop_list_model):
        item = self.item_entry.get_text()                           

        if item == '':
            if HILDON:
                hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                               'Must add an title for the item')
            else:
                print 'Must add an title for the item.'
            return 

        unit = self.unit_entry.get_text()
        amount = self.amount_entry.get_text()
        
        newitem = [item, amount+' '+unit, False]
        
        newiter = shop_list_model.append()
        shop_list_model.set(newiter, ITEM, newitem[ITEM], AMOUNT, \
                            newitem[AMOUNT], 2, newitem[2])

        self.clear_text_entries()

    #Show the toolbar to add new items        
    def show_add_toolbar(self, widget, add_items_toolbar):
        if widget.get_active() == True:
           add_items_toolbar.show()
        else:
           add_items_toolbar.hide()                    
   
    def on_cancel_button_clicked(self, widget):
        self.clear_text_entries()      

    #Clear entries if it's not added a new item            
    def clear_text_entries(self):
        self.item_entry.set_text('')                   
        self.unit_entry.set_text('')
        self.amount_entry.set_text('')           

    #Remove item from the shopping list
    def remove_item(self, widget, shop_list_tv):
        selection = shop_list_tv.get_selection()
        select_model, select_iter = selection.get_selected()

        if select_iter:
            select_model.remove(select_iter)
        else: 
            if HILDON:
                hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                                 'No selected item') 
            else:
                print 'No selected item'
            return   

    #Clear completely the shopping list                        
    def clear_shop_list(self, widget, shop_list_model):
        shop_list_model.clear()

    #Mark or unmark a shopped item.
    def mark_item_as_bought(self, widget, shop_list_tv, shop_list_model):
        selection = shop_list_tv.get_selection()
        select_model, select_iter = selection.get_selected()

        if select_iter:
            item = shop_list_model.get_value(select_iter, ITEM)           
            amount = shop_list_model.get_value(select_iter, AMOUNT)
            marked = shop_list_model.get_value(select_iter, 2)

            if marked == True:
                shop_list_model.set(select_iter, 0, item, 1, amount, 2, \
                                    False)
            else:
                shop_list_model.set(select_iter, 0, item, 1, amount, 2, \
                                    True)
                        
        else:
            if HILDON:       
                hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                                 'No selected item') 
            else:
                print 'No selected item'            
            return          
        
    #Open a saved shopping list
    def open_shopping_list(self, widget, shop_list_model):
        filename = self.show_filechooser_dialog(gtk.FILE_CHOOSER_ACTION_OPEN)
        if filename == None:
            return

        shop_list = self.xml_load_from_file(filename)
        
        if shop_list == False:
            if HILDON:
                hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'Error opening the shopping list')        
            else:
                print 'Error opening the shopping list'

        shop_list_model.clear()
        for item in shop_list:
            model_iter = shop_list_model.append()
            if item[2] == 'False':
                shop_list_model.set(model_iter, 0, item[0], 1, item[1], 2, False)            
            else:
                shop_list_model.set(model_iter, 0, item[0], 1, item[1], 2, True)

    #Save a shopping list in a file        
    def save_shopping_list(self, widget, shop_list_model):
        filename = self.show_filechooser_dialog(gtk.FILE_CHOOSER_ACTION_SAVE)
        if filename == None:
            return
        
        ing_list = []
        for i in range(len(shop_list_model)):
            item = shop_list_model[i][0]
            amount = shop_list_model[i][1]
            marked  = shop_list_model[i][2]
            lista = (amount, item, str(marked))
            ing_list.append(lista)

        xml_document = self.create_xml(ing_list)
        
        success = pyrecipe_utils.save_xml_file(filename, xml_document)
        
        if success == True:
            if HILDON:
                hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'Shopping list saved')
            else:
                print 'Shopping list saved'
        else:
            if HILDON:        
                hildon.hildon_banner_show_information(widget, 'qgn_note_infoprint',\
                                             'Error ocurred while saving the file') 
            else:
                print 'Error ocurred while saving the file'

    #Create the xml document for the shopping list saved
    def create_xml(self, ing_list):
        impl = minidom.getDOMImplementation()
        xml_document = impl.createDocument(None, 'pyrecipe', None)

        list_element = xml_document.createElement('shopping-list')

        for i in range(len(ing_list)):
            item_element = xml_document.createElement('item')
            
            amount_element = xml_document.createElement('amount')
            amount_element.appendChild(xml_document.createTextNode(ing_list[i][0]))

            title_element = xml_document.createElement('title')
            title_element.appendChild(xml_document.createTextNode(ing_list[i][1])) 

            marked_element = xml_document.createElement('marked')
            marked_element.appendChild(xml_document.createTextNode(ing_list[i][2])) 
            
            item_element.appendChild(amount_element)           
            item_element.appendChild(title_element)
            item_element.appendChild(marked_element)
                                   
            list_element.appendChild(item_element)

        xml_document.documentElement.appendChild(list_element)

        return xml_document    

    #Load the xml for the saved shopping list
    #<shopping-list>
    #<item>
    #<amount>1 l</amount>
    #<title>milk</title>
    #<marked>False</marked>
    #</item>
    #</shopping-list>
    def xml_load(self, xml_document):
        shop_list = []
        for node in xml_document.documentElement.childNodes:
            if (node.nodeName == 'shopping-list'):
                for item_node in node.childNodes:
                    if (item_node.nodeName == 'item'):
                        for i in item_node.childNodes:
                            if (i.nodeName == 'amount'):
                                amount = i.firstChild.nodeValue
                            elif (i.nodeName == 'title'):
                                title = i.firstChild.nodeValue
                            elif (i.nodeName == 'marked'):
                                marked = i.firstChild.nodeValue
                                shop_list.append([title, amount, marked])
        return shop_list

    #Show the hildon.filechooser dialog to open/save a file.
    def show_filechooser_dialog(self, action):
        if (action==gtk.FILE_CHOOSER_ACTION_OPEN):
                    dlg_title = "Open Shopping List"
        else:
            dlg_title = "Save Shopping List"

        if HILDON:
            file_dialog = hildon.FileChooserDialog(self.window, action=action)
        else:
            file_dialog = gtk.FileChooserDialog(parent=self.window, action=action, \
                            buttons=(gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                            gtk.STOCK_OK, gtk.RESPONSE_OK))
        file_dialog.set_title(dlg_title)
        file_dialog.set_default_response(gtk.RESPONSE_CANCEL)

        if (action==gtk.FILE_CHOOSER_ACTION_SAVE):
            date_today = datetime.date.today()
            file_dialog.set_current_name('shop-list-' + date_today.isoformat())
            if os.path.exists(os.path.expanduser("~")+'/MyDocs/.documents'):
                file_dialog.set_current_folder(os.path.expanduser("~")+'/MyDocs/.documents')
            else:
                file_dialog.set_current_folder(os.path.expanduser("~"))

        result = file_dialog.run()
        if result == gtk.RESPONSE_OK:
            namefile = file_dialog.get_filename()
            if (action==gtk.FILE_CHOOSER_ACTION_SAVE):
                namefile, extension = os.path.splitext(namefile)
                namefile = namefile + "." + 'shpl'

        else:
            namefile = None                
        file_dialog.destroy()

        return namefile

    #Get the xml document from a file.        
    def xml_load_from_file(self, namefile):
        success = False
        try:
            xml_document = minidom.parse(namefile)
            if (xml_document):
                success = ((self.xml_load(xml_document)))
        except IOError, (errno, strerror):
            print "Error loading post file(%s): %s" % (errno, strerror)
        except:
            print "Error loading post file."
        return success 
