#!/usr/bin/env python2.5

#
# PyGTKEditor
#
# Copyright (c) 2007 Khertan (Benoit HERVIER)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#        
# Khertan (Benoit HERVIER) khertan@khertan.net
#

import sys
import gtk
import gtk.glade
import osso
import pango
import threading
import re
import os.path
import commands
import cPickle
from subprocess import *
import gtksourceview2
from preferences import *

###############################################################################
# @brief Code editor widget
###############################################################################
class CodeEditor (gtksourceview2.View):

  #############################################################################
  # @brief Initialize CodeEditor
  #############################################################################
  def __init__ (self):

    self.language_manager=gtksourceview2.LanguageManager()
    self.buffer=gtksourceview2.Buffer()
    gtksourceview2.View.__init__ (self,self.buffer)

    self.set_editable (True)
    self.set_indent (2)
    self.font_size=11
    self.modify_font (pango.FontDescription ("Monospace "+str(self.font_size)))
    self.connect ('populate-popup', self.on_populate_popup)
    self.__changing = False
    self.__syntax = {}
    self.__language = "pygtk"
    self.saved=True
    self.searched_text = None
    self.user_action = 0

    # connect to TextBuffer 'changed' event
    self.buffer.connect ('changed', self.on_modified)
    self.set_language("python")

    self.buffer.set_highlight_syntax(True)

#    self.buffer.connect("insert-text",self.undo_text_inserted)
#    self.buffer.connect("delete-range",self.undo_text_deleted)
#    self.buffer.connect("begin-user-action",self.undo_begin_user_action)
#    self.buffer.connect("end-user-action",self.undo_end_user_action)

    tag = self.buffer.create_tag('search_hilight')
    tag.set_property('background','yellow')

  #############################################################################
  # @brief Increase font size
  #############################################################################
  def increase_font_size(self):
    self.font_size=self.font_size+1
    self.modify_font (pango.FontDescription ("Monospace "+str(self.font_size)))

  #############################################################################
  # @brief Undo Inserted Text
  #############################################################################
#  def undo_text_inserted(self,textbuffer, iter, text, length):
#    if (self.user_action>0):
#      pos = iter.get_offset()
#      self.undo_add("inserted",pos,text,length)
#
  #############################################################################
  # @brief Undo deleted text
  #############################################################################
#  def undo_text_deleted(self,textbuffer, start, end):
#    if (self.user_action>0):
#      text = textbuffer.get_text(start,end)
#      length = len(text)
#      pos = start.get_offset()
#      self.undo_add("deleted",pos,text,length)

  #############################################################################
  # @brief Undo end user action
  #############################################################################
#  def undo_end_user_action(self,textbuffer):
#    self.user_action = self.user_action - 1

  #############################################################################
  # @brief Undo Begin user action
  #############################################################################
#  def undo_begin_user_action(self,textbuffer):
#    self.user_action = self.user_action + 1

  #############################################################################
  # @brief Undo add
  #############################################################################
#  def undo_add(self,type_action, offset, text, len):
#    self.undo_type_action.append(type_action)
#    self.undo_offset.append(offset)
#    self.undo_text.append(text)
#    self.undo_len.append(len)

  #############################################################################
  # @brief Redo add
  #############################################################################
#  def redo_add(self,type_action, offset, text, len):
#    self.redo_type_action.append(type_action)
#    self.redo_offset.append(offset)
#    self.redo_text.append(text)
#    self.redo_len.append(len)

  #############################################################################
  # @brief Undo/Redo Insert
  #############################################################################
#  def undoredoInsert(self, textbuffer ,undo_type_action,undo_offset,undo_text,undo_len):    
#    iter = textbuffer.get_iter_at_offset(undo_offset)
#    textbuffer.insert(iter,undo_text)
#
  #############################################################################
  # @brief Undo/Redo Delete
  #############################################################################
#  def undoredoDelete(self, textbuffer ,undo_type_action,undo_offset,undo_text,undo_len):    
#    start = textbuffer.get_iter_at_offset(undo_offset)
#    end = textbuffer.get_iter_at_offset(undo_offset+undo_len)
#    textbuffer.delete(start,end)

  #############################################################################
  # @brief Undo
  #############################################################################
  def undo(self):
    self.buffer.undo()
#    undo_type_action = self.undo_type_action.pop()
#    undo_offset = self.undo_offset.pop()
#    undo_text = self.undo_text.pop()
#    undo_len = self.undo_len.pop()
#
#    textbuffer = self.get_buffer()
#    if (undo_type_action=="inserted"):
#      self.undoredoDelete(textbuffer,undo_type_action,undo_offset,undo_text,undo_len)
#    else:
#      self.undoredoInsert(textbuffer,undo_type_action,undo_offset,undo_text,undo_len)
#
#    self.redo_add(undo_type_action,undo_offset,undo_text,undo_len)

  #############################################################################
  # @brief Redo
  #############################################################################
  def redo(self):
    self.buffer.redo()
#    undo_type_action = self.redo_type_action.pop()
#    undo_offset = self.redo_offset.pop()
#    undo_text = self.redo_text.pop()
#    undo_len = self.redo_len.pop()
#
#    textbuffer = self.get_buffer()
#    if (undo_type_action=="inserted"):
#      self.undoredoInsert(textbuffer,undo_type_action,undo_offset,undo_text,undo_len)
#    else:
#      self.undoredoDelete(textbuffer,undo_type_action,undo_offset,undo_text,undo_len)
#
#    self.undo_add(undo_type_action,undo_offset,undo_text,undo_len)

  #############################################################################
  # @brief Decrease font size
  #############################################################################
  def decrease_font_size(self):
    self.font_size=self.font_size-1
    self.modify_font (pango.FontDescription ("Monospace "+str(self.font_size)))

  #############################################################################
  # @brief register syntax highlighters
  #############################################################################
  def register_syntax (self, obj):  
    self.__syntax[obj.language] = obj
    obj.populate_buffer (self.get_buffer ())

  #############################################################################
  # @brief Return syntax dictionary
  #############################################################################
  def get_syntaxes (self):
    return self.__syntax

  def get_language(self):
    return self.__language

  def set_language(self,language):

    prefs = Preferences()
    prefs.load()

    if language==None:
      language=prefs.default_language
    self.__language=language
    self.buffer.set_language(self.language_manager.get_language(language))
    self.set_indent_width(prefs.indent_width)
    self.set_auto_indent(prefs.auto_indent)
    self.set_tab_width(prefs.tab_width)
    self.set_insert_spaces_instead_of_tabs(prefs.use_space)
    self.set_indent_on_tab(prefs.indent_on_tab)
    self.set_show_line_numbers(prefs.show_line_numbers)

  #############################################################################
  # @brief Find a substring
  #############################################################################
  def search(self,text):
    buffer = self.get_buffer ()
    if self.searched_text != text:
      self.searched_text = text
      self.searched_text_count = 0

      start, end = buffer.get_bounds ()
      self.search_results_list=[]

      search_results = self.findSubstring(buffer.get_text (start, end), text)
      for index in search_results:
        found_start = buffer.get_iter_at_offset (index.span()[0])
        found_end   = buffer.get_iter_at_offset (index.span()[1])
        buffer.apply_tag_by_name ("search_hilight", found_start, found_end)
        self.search_results_list.append(found_start.get_offset())

    else:
      self.searched_text_count =  self.searched_text_count + 1
    try:
      iter = buffer.get_iter_at_offset(self.search_results_list[self.searched_text_count])
      buffer.place_cursor(iter)
      self.scroll_to_iter(iter,0)
    except IndexError:
      print "End text reached"

  #############################################################################
  # @brief Find a substring
  #############################################################################
  def findSubstring(self,text, substring):
    searched_text=re.compile(substring)
    return searched_text.finditer(text)

  #############################################################################
  # @brief Copy code to clipboard
  #############################################################################
  def copy_to_clipboard (self):
    buffer = self.get_buffer ()
    try:
      start, end = buffer.get_selection_bounds ()

      clipboard = gtk.clipboard_get (gtk.gdk.SELECTION_CLIPBOARD)
      clipboard.set_text( buffer.get_text (start, end))
    except:
      print "No selection"

  #############################################################################
  # @brief Cut code to clipboard
  #############################################################################
  def cut_to_clipboard (self):
    buffer = self.get_buffer ()
    start, end = buffer.get_selection_bounds ()
    try:
      clipboard = gtk.clipboard_get (gtk.gdk.SELECTION_CLIPBOARD)
      clipboard.set_text( buffer.get_text (start, end))
      self.user_action = self.user_action + 1
      buffer.delete(start,end)
      self.user_action = self.user_action - 1
    except:
      print "No selection"

  #############################################################################
  # @brief On paste
  #############################################################################
  def on_paste(self,widget, text,data=None):
    buffer = self.get_buffer ()
    try:
      self.user_action = self.user_action + 1
      start,end = buffer.get_selection_bounds()
      buffer.delete(start,end)
      buffer.insert(start,text)
    except:
      buffer.insert_at_cursor(text)
      start = buffer.get_iter_at_mark(buffer.get_insert())
      soff = start.get_offset() - len(text)
      if soff<0:
        soff=0
      start=buffer.get_iter_at_offset(soff)
    self.user_action = self.user_action - 1

  #############################################################################
  # @brief Paste code from clipboard
  #############################################################################
  def paste_from_clipboard (self):
    buffer = self.get_buffer ()
    clipboard = gtk.clipboard_get (gtk.gdk.SELECTION_CLIPBOARD)
    clipboard.request_text(self.on_paste,None)

  #############################################################################
  # @brief Occurs when buffer is modified
  #############################################################################
  def on_modified (self, buffer,data=None):
    start, end = buffer.get_bounds ()
    buffer.remove_tag_by_name("search_hilight",start, end)

    if (self.saved):
      notebook=self.get_parent().get_parent()
      notebook_cur_page=notebook.get_current_page()
      cur_text=notebook.get_tab_label_text(notebook.get_children()[notebook_cur_page])
      notebook.set_tab_label_text(notebook.get_children()[notebook_cur_page],"*"+cur_text)
      self.saved=False

  #############################################################################
  # @brief Populate popup menu
  #############################################################################
  def on_populate_popup (self, textview, popup, data=None):
    item = gtk.MenuItem ('Highlight')
    item.show ()
    popup.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    l = [(obj.description, obj.language) for obj in self.__syntax.values()]
    l.sort ()
    l = [('Default', None)] + l

    group = None
    for desc, language in l:
      item = gtk.RadioMenuItem (group, desc)
      item.connect ('activate', self.on_syntax_selection, language)
      item.set_active (language == self.__language)
      item.show ()
      menu.append (item)

      if not group: group = item

  #############################################################################
  # @brief Change syntax highlight
  #############################################################################
  def on_syntax_selection (self, item, data=None):
    self.__language = data

  #############################################################################
  # @brief Change syntax highlight
  #############################################################################
  def on_syntax_selection (self, item, data=None):
    self.__language = data  

  #############################################################################
  # @brief Clean useless space
  #############################################################################
  def clean_space(self):
    prefs = Preferences()
    prefs.load()
    if prefs.useless_space == True:
      buffer = self.get_buffer()
      offset = buffer.get_iter_at_mark(buffer.get_insert()).get_offset()
      start, end = buffer.get_bounds ()
      buffer.set_text(re.sub('\n\s+\n','\n\n',buffer.get_text(start,end)))
      buffer.place_cursor(buffer.get_iter_at_offset(offset))

  #############################################################################
  # @brief Insert tabulation
  #############################################################################
  def indent_tab (self):
    buffer = self.get_buffer()
    prefs = Preferences()
    prefs.load()
    if prefs.use_space == True:
      #self.get_buffer().insert_at_cursor(' ' * prefs.tab_width)
      tabul=' ' * prefs.tab_width
    else:
      #self.get_buffer().insert_at_cursor('\t')
      tabul='\t'
    if (buffer.get_selection_bounds()!=()):
      start,end=buffer.get_selection_bounds()
      for line in range(start.get_line(),end.get_line()+1):
        buffer.insert(buffer.get_iter_at_line(line),tabul)
    else:
      buffer.insert(buffer.get_iter_at_line(buffer.get_iter_at_mark(buffer.get_insert()).get_line()),tabul)

  #############################################################################
  # @brief Unindent tabulation
  #############################################################################
  def unindent_tab (self):
    buffer = self.get_buffer()
    prefs = Preferences()
    prefs.load()
    if (buffer.get_selection_bounds()!=()):
      start,end=buffer.get_selection_bounds()
      for line in range(start.get_line(),end.get_line()+1):
        r = buffer.get_iter_at_line(line)
        c = buffer.get_text(r,r)
        if c == '\t':
          buffer.delete(r,r)
        else:
          e=buffer.get_iter_at_offset(r.get_offset()+ prefs.tab_width)
          if (buffer.get_text(r,e)==' '* prefs.tab_width):
            buffer.delete(r,e)
    else:
      r = buffer.get_iter_at_line(buffer.get_iter_at_mark(buffer.get_insert()).get_line())
      c = buffer.get_text(r,r)
      if c == '\t':
        buffer.delete(r,r)
      else:
        e=buffer.get_iter_at_offset(r.get_offset()+ prefs.tab_width)
        if (buffer.get_text(r,e)==' '* prefs.tab_width):
          buffer.delete(r,e)

  #############################################################################
  # @brief Comment string
  #############################################################################
  def get_comment_string(self):
    if self.get_language()=='cpp':
      return '//'
    if self.get_language()=='php':
      return '//'
    elif self.get_language()=='python':
      return '#'
    elif self.get_language()=='sql':
      return '--'
    else:
      return ''

  #############################################################################
  # @brief Comment / Uncomment
  #############################################################################
  def comment_text (self):
    ctext = self.get_comment_string()
    if ctext != '':
      comment = True
      buffer = self.get_buffer()
      prefs = Preferences()
      prefs.load()
      r = buffer.get_iter_at_line(buffer.get_iter_at_mark(buffer.get_insert()).get_line())
      e=buffer.get_iter_at_offset(r.get_offset()+ len(ctext))
      if (buffer.get_text(r,e)==ctext): 
        comment=False
      if comment==True:
        if (buffer.get_selection_bounds()!=()):
          start,end=buffer.get_selection_bounds()
          for line in range(start.get_line(),end.get_line()+1):
            buffer.insert(buffer.get_iter_at_line(line),ctext)
        else:
          buffer.insert(buffer.get_iter_at_line(buffer.get_iter_at_mark(buffer.get_insert()).get_line()),ctext)
      else:
         if (buffer.get_selection_bounds()!=()):
           start,end=buffer.get_selection_bounds()
           for line in range(start.get_line(),end.get_line()+1):
             r = buffer.get_iter_at_line(line)
             e=buffer.get_iter_at_offset(r.get_offset()+ len(ctext))
             if (buffer.get_text(r,e) == ctext):
               buffer.delete(r,e)
         else:
           r = buffer.get_iter_at_line(buffer.get_iter_at_mark(buffer.get_insert()).get_line())
           e=buffer.get_iter_at_offset(r.get_offset()+ len(ctext))
           if (buffer.get_text(r,e) == ctext):
             buffer.delete(r,e)
          