/*
 * This file is part of pwsafe
 *
 * Copyright (C) 2005 Nokia Corporation.
 * Copyright (C) 2005 HolisTech Limited.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
                     
#include <src/callbacks.h>
#include <src/interface.h>
#include <src/crypto.h>
#include <gtk/gtk.h>
#include <libintl.h>
/* strlen needed from string.h */
#include <string.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib.h>
#include <mce/dbus-names.h>

#if HILDON == 1
#include <hildon/hildon.h>
#else
#include <hildon-widgets/hildon-app.h>
#endif
#include <gdk/gdkkeysyms.h>

/* Privates: */
void safeReplaceString(char** oldstr, const gchar *newstr, MainView *mainview);
void maybe_save_file(MainView *mainview);
void save_mod_file(MainView *mainview);
void setCurrentGroup(MainView *mainview, gchar *group);

/* activate grid */
void callback_activate_treeView(GtkTreeView *treeView, GtkTreePath *path, GtkTreeViewColumn *column, MainView *mainview) {
	callback_hildon_activate_treeView(treeView, path, mainview);
}

/* activate hildon grid */
void callback_hildon_activate_treeView(GtkTreeView *treeView, GtkTreePath *path, MainView *mainview) {
	GtkTreeIter iter;
	rec *onrec=NULL;
	gchar *label=NULL;
	gboolean groupExpand=FALSE;
	g_assert(mainview != NULL && mainview->data != NULL );
	if ( !gtk_tree_model_get_iter(GTK_TREE_MODEL(mainview->store), &iter, path) ) {
		return;
	}
	gtk_tree_model_get(GTK_TREE_MODEL(mainview->store), &iter, LABEL_COLUMN, &label, DATA_COLUMN, &onrec, GROUP_COLUMN, &groupExpand, -1);
	if (onrec!=NULL) {
		show_rec(mainview, onrec, FALSE);
		setCurrentGroup(mainview, onrec->group);
	} else {
		setCurrentGroup(mainview, label);
		if (groupExpand) {
			gtk_tree_view_collapse_row (treeView, path);
		}
		gtk_tree_store_set(mainview->store, &iter, GROUP_COLUMN, !groupExpand, -1);
	}
}

gboolean callback_key_press(GtkWidget* widget, GdkEventKey* event,  MainView *mainview) {
	switch (event->keyval) {
		case GDK_F6: /* fullscreen */
			interface_fullscreen(mainview);
			return TRUE;
		case GDK_F7: /* button + */
			return TRUE;
		case GDK_F8: /* button - */
			return TRUE;
	}
	return FALSE;
}

void safeReplaceString(char** oldstr, const gchar *newstr, MainView *mainview) {
/* scrub and delete oldstr, malloc and duplicate newstr, return pointer and tag file as changed... */
/* but only if the strings are different (and be careful of null old string pointers) */
	if ((*oldstr)==NULL) {
		if (newstr[0]!='\0') {
			(*oldstr)=g_strdup(newstr);
			mainview->file_edited=TRUE; /* tag file as changed */
		}
	} else {
		if (strcmp((*oldstr),newstr)!=0) { /* check they differ before copying stuff around */
			trashMemory((*oldstr), strlen((*oldstr)));
			g_free((*oldstr));
			(*oldstr)=g_strdup(newstr);
			mainview->file_edited=TRUE; /* tag file as changed */
		}
	}
}

const gchar *entryGetText(GtkWidget *widget) {
	return gtk_entry_get_text(GTK_ENTRY(widget));
}

/* record detail events */
void callback_rec_response(GtkDialog *dialog, gint response, MainView *mainview) {
	RecWindow *recwindow = NULL;
	rec *rec=NULL;
	GtkTextBuffer *buffer=NULL;
	GtkTextIter start;
	GtkTextIter end;

	g_assert(mainview != NULL && mainview->data != NULL );
	
	recwindow = mainview->rec_win;
	rec=recwindow->rec;
	switch (response) {
		case 1: /* save any changes */
			safeReplaceString(&rec->title, entryGetText(recwindow->title), mainview);
			safeReplaceString(&rec->user, entryGetText(recwindow->username), mainview);
			safeReplaceString(&rec->password, entryGetText(recwindow->password), mainview);
#if MAEMO_VERSION_MAJOR < 5
			safeReplaceString(&rec->group, gtk_combo_box_get_active_text(GTK_COMBO_BOX(recwindow->group)), mainview);
#else
			safeReplaceString(&rec->group, hildon_button_get_value(HILDON_BUTTON(recwindow->group)), mainview);
#endif
			buffer=gtk_text_view_get_buffer (GTK_TEXT_VIEW(recwindow->notes));
			if (gtk_text_buffer_get_modified(buffer)) { /* update notes if buffer has changed */
				gtk_text_buffer_get_start_iter (buffer, &start);
				gtk_text_buffer_get_end_iter (buffer, &end);
				safeReplaceString(&rec->notes, gtk_text_buffer_get_text (buffer, &start, &end, FALSE), mainview);
			}
			resort_rec(rec);
			unshow_rec(mainview);  /* close rec window */
			populate_treeView(mainview); /* redraw grid */
			break;
		case 2:
			gtk_clipboard_set_text(mainview->clipboard, rec->user, -1);
			break;
		case 3:
			gtk_clipboard_set_text(mainview->clipboard, rec->password, -1);
			break;
		case 4:
			if ((recwindow->editable) && (CONFRESP_YES==interface_yes_no_note(mainview, _("Confirm Delete ?")))) {
				unshow_rec(mainview);
				delete_rec(rec);
				populate_treeView(mainview); /* redraw grid */
			}
			break;
		case 5:
			interface_set_rec_editable(recwindow, TRUE);
			break;
		default:
			//Cancel for Fremantle (out of dialog click)
			unshow_rec(mainview);  /* close rec window */
			if (recwindow->create) {
				delete_rec(rec);
			}
			populate_treeView(mainview); /* redraw grid */
			break;
	}
}

void setUrlTag(GtkTextBuffer *buffer, gint posStart, gint posEnd) {
	GtkTextIter start, end;
	gtk_text_buffer_get_iter_at_offset(buffer, &start, posStart);
	gtk_text_buffer_get_iter_at_offset(buffer, &end, posEnd);
	gtk_text_buffer_apply_tag_by_name(buffer, "url", &start, &end);

}

#if MAEMO_VERSION_MAJOR < 5
void enhanceBufferWithString(GtkTextBuffer *buffer, const gchar *search) {
	GtkTextIter start, end;
	gchar *text, *bout;
	gint posStart, posEnd, index, listIdx;
	gchar **list, **searchList;
	
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	text = gtk_text_buffer_get_slice(buffer, &start, &end, TRUE);
	if (text == NULL || text[0] == '\0') {
		return;
	}
	list = g_strsplit_set(text, " \t\r\n", -1);
	searchList = g_strsplit(search, " ", -1);
	index = 0;
	posStart = 0;
	while (list[index]) {
		posEnd = posStart + g_utf8_strlen(list[index], -1);
		if (posEnd > posStart) {
			bout = g_ascii_strdown(list[index], -1);
			listIdx = 0;
			while (searchList[listIdx]) {
				if (g_str_has_prefix(bout, searchList[listIdx])) {
					setUrlTag(buffer, posStart, posEnd);
					break;
				}
				listIdx ++;
			}
			g_free(bout);
		}
		posStart =  posEnd + 1;
		index++;
	}
	g_strfreev(list);
	g_strfreev(searchList);
}
#else
void enhanceBufferWithRegex(GtkTextBuffer *buffer, GRegex *regex) {
	GtkTextIter start, end;
	gchar *text;
	GMatchInfo *matchInfo;
	GError *error = NULL;
	gint posStart, posEnd;
	
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	text = gtk_text_buffer_get_slice(buffer, &start, &end, TRUE);
	
	g_regex_match_full(regex, text, -1, 0, 0, &matchInfo, &error);
	while (g_match_info_matches(matchInfo)) {
		if (g_match_info_fetch_pos(matchInfo, 0, &posStart, &posEnd)) {
			setUrlTag(buffer, posStart, posEnd);
		}
		g_match_info_next(matchInfo, &error);
	}
	g_match_info_free(matchInfo);
	if (error != NULL) {
		g_printerr("Error while mathing: %s\n", error->message);
		g_error_free(error);
	}
}
#endif

void callback_buffer_changed(GtkTextBuffer *buffer, MainView *mainview) {
	GtkTextIter start, end;
	
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	gtk_text_buffer_remove_all_tags(buffer, &start, &end);

#if MAEMO_VERSION_MAJOR < 5
	enhanceBufferWithString(buffer, "http:// https:// www.");
#else
	enhanceBufferWithRegex(buffer, mainview->regexHttp);
	enhanceBufferWithRegex(buffer, mainview->regexWww);
#endif
}

static void launch_browser (gchar *url, MainView *mainView) {
  DBusGConnection *connection;
  GError *error = NULL;

  DBusGProxy *proxy;

  connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (connection == NULL) {
      msgbox(mainView, error->message);
      g_error_free (error);
      return;
    }

  proxy = dbus_g_proxy_new_for_name (connection,
				     "com.nokia.osso_browser",
				     "/com/nokia/osso_browser/request",
				     "com.nokia.osso_browser");

  error = NULL;
  if (!dbus_g_proxy_call (proxy, "load_url", &error,
			  G_TYPE_STRING, url,
			  G_TYPE_INVALID,
			  G_TYPE_INVALID)) {
      msgbox(mainView, error->message);
      g_error_free (error);
    }
}


gboolean callback_tag_event(GtkTextTag *tag, GObject *object, GdkEvent *event, GtkTextIter *iter, MainView *mainView) {
	GtkTextIter start, end;
	gchar *text;
	if (event->type == GDK_BUTTON_PRESS && !gtk_text_view_get_editable(GTK_TEXT_VIEW(object))) {
		start = *iter; 
		while (!gtk_text_iter_begins_tag(&start, tag)) {
			if (!gtk_text_iter_backward_char(&start)) {
				break;
			}
		}
		end = *iter; 
		while (!gtk_text_iter_ends_tag(&end, tag)) {
			if (!gtk_text_iter_forward_char(&end)) {
				break;
			}
		}
		text = gtk_text_iter_get_text(&start, &end);
//		g_print("text associ au tag %s\n", text);
		launch_browser(text, mainView);
	}
	return FALSE;
}

/* Callback for hardware D-BUS events */
void hw_event_handler(osso_hw_state_t *state, MainView *mainview) {
 	g_assert(mainview != NULL && mainview->data != NULL );

	if ((state->shutdown_ind)	/* system shutdown */
			|| (state->system_inactivity_ind)) {	/* system sleeping */
		save_mod_file(mainview);
		close_file(mainview);
		return;
	}
	if (state->memory_low_ind) {	/* memory low... */
		unshow_rec(mainview);	/* close the rec window... changes not saved... */
	}
    save_mod_file(mainview);

    /*
	if (state->save_unsaved_data_ind && mainview->file_edited) {	/ * save unsaved data... * /
		write_pwsafe_file(mainview);
/ *		if (CONFRESP_YES==interface_save_changes_note(mainview)) {
		} * /
	} 
    */
    
}

/* application forced exit callback  */
//void exit_event_handler (gboolean die_now, MainView *mainview) {
gboolean exit_event_handler( GtkWidget *widget, GdkEvent  *event, MainView *mainview) {
    if (mainview && mainview->data) {
        save_mod_file(mainview);
        close_file(mainview);
    }
	gtk_main_quit();
	return FALSE;
}


void maybe_save_file(MainView *mainview) {
/* save changes if file changed, with requester*/
    if (mainview->pass && mainview->file_edited) {
        if (CONFRESP_YES==interface_yes_no_note(mainview, _("Save Changes ?"))) { 
            if (!mainview->file_name) {  /* should never happen because file name set on open */
                mainview->file_name = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
            }
            write_pwsafe_file( mainview );
        } 
    }
}

void save_mod_file(MainView *mainview) {
/* save changes if file changed, don't ask */
    if (mainview->pass && mainview->file_edited) {
        if (!mainview->file_name) {  /* should never happen because file name set on open */
            mainview->file_name = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
        }
        write_pwsafe_file( mainview );
    }
}

gchar * getCurrentGroup(MainView *mainview ) {
	return mainview->currentGroup;
}

void setCurrentGroup(MainView *mainview, gchar *group) {
	if (group != NULL && group[0] != '\0' && strcmp(group, _(DEFAULT_GROUP)) != 0) {
		g_free(mainview->currentGroup);
		mainview->currentGroup = g_strdup(group);
	}
}

/* add entry */
void callback_add_entry ( GtkAction * action, MainView *mainview ) {
	rec* newrec=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	newrec=add_new_record("", getCurrentGroup(mainview));
	populate_treeView(mainview);
	show_rec(mainview, newrec, TRUE);
	interface_set_rec_editable(mainview->rec_win, TRUE);
}

/* change password */
void callback_change_password(GtkAction * action, MainView *mainview) {
	char *newpass=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	newpass=get_password(mainview, _("Change Password"), PWD_DLG_CHANGE);
	if (newpass) {
		mainview->pass=newpass;
		mainview->file_edited = TRUE;
	}
}

/* new */
void callback_file_new(GtkAction * action, MainView *mainview) {
	char *newpass;
	g_assert(mainview != NULL && mainview->data != NULL );
	/* maybe_save_file(mainview); */
	save_mod_file(mainview);
	close_file(mainview);
	newpass=get_password(mainview, _("New Password"), PWD_DLG_NEW);
	if (newpass) {
		mainview->pass=newpass;
//		add_new_record("new", "Group");
//		populate_treeView(mainview);
		/* maybe_save_file(mainview); */
		write_pwsafe_file(mainview);  /* force set file name */
		if (mainview->file_name != NULL) {
			interface_show_state(mainview) ;
		}
	}
}

/* open */
void callback_file_open(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
 	/* maybe_save_file(mainview); */
    save_mod_file(mainview);
	mainview->file_name = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_OPEN );
	read_pwsafe_file(mainview);
}

void callback_file_open_last ( GtkAction * action, MainView *mainview ) {
	g_assert(mainview != NULL && mainview->data != NULL );
 	/* maybe_save_file(mainview); */
    save_mod_file(mainview);
	mainview->file_name = mainview->last_file;
	read_pwsafe_file(mainview);
}

/* close file */
void callback_file_close ( GtkAction * action, MainView *mainview ) {
	g_assert(mainview != NULL && mainview->data != NULL );
	maybe_save_file(mainview);
	close_file(mainview);
}

/* save */
void callback_file_save(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	write_pwsafe_file(mainview);
}


/* save as... */
void callback_file_saveas(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	mainview->file_name=NULL;
	write_pwsafe_file(mainview);
}


/* fullscreen */
void callback_fullscreen( GtkAction * action, gpointer data ) {
	interface_fullscreen((MainView *) data);
}

gint searchicase(const char *str1, const char *str2) {
	if (str1 == NULL) {
		return 1;
	}
	if (str2 == NULL) {
		return 0;
	}
	return strcasestr(str1, str2)? 0: 1;
}

/* filter */

void callback_filter(GtkWidget *widget, MainView *mainview ) {
    const gchar *ftxt = NULL;
    GList *onrec=NULL;

    ftxt = gtk_entry_get_text(GTK_ENTRY(widget));
    
    onrec=g_list_first(recs);
    if (ftxt[0] == '\0') { /* no filter, skip= none */
        while (onrec!=NULL) {
            rec *data=(rec *) onrec->data;
            data->skip=0;
            onrec=g_list_next(onrec);
        }
    } else {
        while (onrec!=NULL) {
            rec *data=(rec *) onrec->data;
            
            data->skip= searchicase(data->group,ftxt);
            if (data->skip != 0) {
                data->skip= searchicase(data->title,ftxt);
                if (data->skip != 0) {
                    data->skip= searchicase(data->user,ftxt);
                    if (data->skip != 0) {
                        data->skip= searchicase(data->notes,ftxt);
                        if (data->skip != 0) {
                            data->skip= searchicase(data->password,ftxt);
                        }
                    }
                }
            }
            onrec=g_list_next(onrec);
        }
    }
    populate_treeView(mainview);
}


gboolean callback_tree_key_pressed(GtkTreeView *treeView, GdkEventKey *event, MainView *mainview) {
	gchar *utf8String = NULL;
	guint32 ucs4Char = gdk_keyval_to_unicode(event->keyval);
	if (g_unichar_isalnum(ucs4Char)) { // event->keyval >= GDK_space && event->keyval <= GDK_asciitilde
//	if (event->string[0]!='\0' && event->string[0]!='\n') {
		utf8String = g_strnfill(10, '\0');
		if (g_unichar_to_utf8(ucs4Char, utf8String) > 0) {
			gtk_entry_set_text(GTK_ENTRY(mainview->filter_tb), utf8String);
			callback_filter(mainview->filter_tb, mainview);
			gtk_widget_grab_focus(mainview->filter_tb);
			gtk_editable_set_position(GTK_EDITABLE(mainview->filter_tb), -1);
			g_free(utf8String);
			return FALSE;
		}
	}
	g_free(utf8String);
	return TRUE;
}

gint getEntryLength(GtkEntry *entry) {
#if MAEMO_VERSION_MAJOR < 5
	gchar *text;
	text = gtk_entry_get_text(entry);
	return (text == NULL)?0:strlen(text);
#else
	return gtk_entry_get_text_length(entry);
#endif	
}

gboolean isTextEqual(const gchar *text1, const gchar *text2) {
	if (text1 == NULL && text2 == NULL) {
		return TRUE;
	} else if (text1 == NULL || text2 == NULL) {
		return FALSE;
	}
#if MAEMO_VERSION_MAJOR < 5
	return (strcmp(text1, text2) == 0);
#else
	return (g_strcmp0(text1, text2) == 0);
#endif	
}

/* password Dialog */
void callback_pwdialog_changed(GtkWidget *widget, PasswordUI *pwUI) {
	gboolean activate = FALSE;
	gint len;
	if (pwUI->type == PWD_DLG_GET) {
		activate = (getEntryLength(GTK_ENTRY(widget)) != 0);
	} else if (pwUI->type == PWD_DLG_NEW) {
		len = getEntryLength(GTK_ENTRY(pwUI->entry1));
		activate = (len != 0 && getEntryLength(GTK_ENTRY(pwUI->entry2)) == len);
		if (activate) {
			activate = isTextEqual(gtk_entry_get_text(GTK_ENTRY(pwUI->entry1)), gtk_entry_get_text(GTK_ENTRY(pwUI->entry2)));
		}
		if (!activate) {
			g_free(pwUI->msg);
			if (len == 0) {
				pwUI->msg = g_strdup(_("Empty Password Not Allowed"));
			} else {
				pwUI->msg = g_strdup(_("Password Mismatch"));
			}
		}
	} else {
		activate = (getEntryLength(GTK_ENTRY(pwUI->entry1)) != 0);
		if (activate) {
			len = getEntryLength(GTK_ENTRY(pwUI->entry2));
			activate = (len != 0 && getEntryLength(GTK_ENTRY(pwUI->entry3)) == len);
			if (activate) {
				activate = isTextEqual(gtk_entry_get_text(GTK_ENTRY(pwUI->entry2)), gtk_entry_get_text(GTK_ENTRY(pwUI->entry3)));
			}
			if (!activate) {
				g_free(pwUI->msg);
				if (len == 0) {
					pwUI->msg = g_strdup(_("Empty Password Not Allowed"));
				} else {
					pwUI->msg = g_strdup(_("Password Mismatch"));
				}
			}
		} else {
			g_free(pwUI->msg);
			pwUI->msg = g_strdup(_("Empty Password Not Allowed"));
		}
	} 
	gtk_widget_set_sensitive(pwUI->button, activate);
}

void callback_pwdialog_activate(GtkWidget *widget, PasswordUI *pwUI) {
	if (GTK_WIDGET_IS_SENSITIVE(pwUI->button)) {
		gtk_button_clicked(GTK_BUTTON(pwUI->button));
	} else if (pwUI->msg != NULL) {
		msgbox(pwUI->main, pwUI->msg);
		g_free(pwUI->msg);
		pwUI->msg = NULL;
	}
}

void callback_pwdialog_cancel(GtkWidget *widget, PasswordUI *pwUi) {
	gtk_dialog_response(GTK_DIALOG(pwUi->dialog), GTK_RESPONSE_CANCEL);
}

void callback_pwdialog_button(GtkWidget *widget, PasswordUI *pwUi) {
	if (pwUi->type == PWD_DLG_GET) {
		pwUi->result=g_strdup(gtk_entry_get_text(GTK_ENTRY(pwUi->entry1)));
	} else if (pwUi->type == PWD_DLG_NEW) {
		pwUi->result=g_strdup(gtk_entry_get_text(GTK_ENTRY(pwUi->entry1)));
		if (!isTextEqual(pwUi->result, gtk_entry_get_text(GTK_ENTRY(pwUi->entry2)))) {
			msgbox(pwUi->main, _("Password Mismatch"));
			g_free(pwUi->result);
//				trashFreeMemory(result, -1);
			pwUi->result = NULL;
		}
	} else {
		pwUi->result=g_strdup(gtk_entry_get_text(GTK_ENTRY(pwUi->entry1)));
		if (!isTextEqual(pwUi->result, pwUi->main->pass)) {
			msgbox(pwUi->main, _("Invalid Password"));
			g_free(pwUi->result);
//				trashFreeMemory(result, -1);
			pwUi->result = NULL;
		} else {
			pwUi->result=g_strdup(gtk_entry_get_text(GTK_ENTRY(pwUi->entry2)));
			if (!isTextEqual(pwUi->result, gtk_entry_get_text(GTK_ENTRY(pwUi->entry3)))) {
				msgbox(pwUi->main, _("Password Mismatch"));
				g_free(pwUi->result);
//					trashFreeMemory(result, -1);
				pwUi->result = NULL;
			}
		}
	}
	if (pwUi->result != NULL) {
		gtk_dialog_response(GTK_DIALOG(pwUi->dialog), GTK_RESPONSE_OK);
	}
}

/* export */
void callback_export ( GtkAction * action, MainView *mainview ) {
    char *fname;
    
    fname = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
    if (fname != NULL) {
        FILE *outf;
        outf = fopen(fname,"w");
        if (outf != (FILE *) NULL) {
            GList *onrec=NULL;
     
            onrec=g_list_first(recs);
            while (onrec!=NULL) {
                rec *data=(rec *) onrec->data;
                   
                fprintf(outf,"Group: %s\n",data->group);
                fprintf(outf,"Title: %s\n",data->title);
                fprintf(outf,"User: %s\n",data->user);
                fprintf(outf,"Password: %s\n",data->password);
                fprintf(outf,"Notes: %s\n",data->notes);
                fprintf(outf,"\n--\n");
                       
                onrec=g_list_next(onrec);
            }
        } else {
            interface_error(PWSAFE_ERROR_SAVE_FAILED, mainview);
        }
    }
}

DBusHandlerResult dbus_handle_mce_message(DBusConnection *con, DBusMessage *msg, MainView *mainView ) {
	DBusMessageIter iter;
	dbus_bool_t value;

	if (dbus_message_is_signal(msg, MCE_SIGNAL_IF, MCE_INACTIVITY_SIG)) {
		if (dbus_message_iter_init(msg, &iter)) {
			dbus_message_iter_get_basic(&iter, &value);
			if (value) {
				interfaceCallLastItem(mainView) ;
			}
			g_printerr("INFO: New activity status is now: %d\n", value);
		}
	}
	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}


