#include <QtGui>
#include "jigsawpuzzleitem.h"
#include "jigsawpuzzleboard.h"
#include "util.h"

JigsawPuzzleItem::JigsawPuzzleItem(const QPixmap &pixmap, const QSize &unitSize, QGraphicsItem *parent, QGraphicsScene *scene) :
    QObject(scene),
    PuzzleItem(pixmap, parent, scene),
    _unit(unitSize),
    _dragging(false),
    _canMerge(false),
    _tolerance(5),
    _weight(randomInt(50, 950) / 1000.0)
{
}

bool JigsawPuzzleItem::canMerge() const
{
    return _canMerge;
}

void JigsawPuzzleItem::setMerge(bool canMerge)
{
    _canMerge = canMerge;
}

void JigsawPuzzleItem::enableMerge()
{
    setMerge(true);
}

void JigsawPuzzleItem::disableMerge()
{
    setMerge(false);
}

double JigsawPuzzleItem::weight()
{
    return _weight;
}

void JigsawPuzzleItem::setTolerance(int t)
{
    _tolerance = t;
}

int JigsawPuzzleItem::tolerance()
{
    return _tolerance;
}

const QSize &JigsawPuzzleItem::unit()
{
    return _unit;
}

bool JigsawPuzzleItem::merge(JigsawPuzzleItem *item)
{
    if (isNeighbourOf(item))
    {
        foreach (PuzzleItem *n, item->neighbours())
        {
            item->removeNeighbour(n);
            this->addNeighbour(n);
        }

        item->_canMerge = _canMerge = false;

        QPoint vector = item->puzzleCoordinates() - puzzleCoordinates();
        int x1, x2, y1, y2, u1, v1;
        if (vector.x() >= 0)
        {
            x1 = 0;
            u1 = 0;
            x2 = vector.x() * _unit.width();
        }
        else
        {
            x1 = - vector.x() * _unit.width();
            u1 = - vector.x();
            x2 = 0;
        }
        if (vector.y() >= 0)
        {
            y1 = 0;
            v1 = 0;
            y2 = vector.y() * _unit.height();
        }
        else
        {
            y1 = - vector.y() * _unit.height();
            v1 = - vector.y();
            y2 = 0;
        }

        QPixmap pix(max(x1 + pixmap().width(), x2 + item->pixmap().width()),
                    max(y1 + pixmap().height(), y2 + item->pixmap().height()));
        pix.fill(Qt::transparent);

        QPainter p;
        p.begin(&pix);
        p.setClipping(false);

        p.drawPixmap(x1, y1, pixmap());
        p.drawPixmap(x2, y2, item->pixmap());

        p.end();
        setPixmap(pix);
        setPuzzleCoordinates(puzzleCoordinates() - QPoint(u1, v1));
        setPos(pos().x() - x1, pos().y() - y1);
        _dragStart = _dragStart + QPointF(x1, y1);
        item->hide();
        delete item;
        _canMerge = true;

        if (neighbours().count() == 0)
        {
            _dragging = false;
            _canMerge = false;
            QPointF newPos((scene()->width() - pixmap().width()) / 2, (scene()->height() - pixmap().height()) / 2);
#if QT_VERSION >= QT_VERSION_CHECK(4, 6, 0)
            QPropertyAnimation *anim = new QPropertyAnimation(this, "pos", this);
            anim->setEndValue(newPos);
            anim->setDuration(1000);
            anim->setEasingCurve(QEasingCurve(QEasingCurve::OutElastic));
            connect(anim, SIGNAL(finished()), this, SIGNAL(noNeighbours()));
            anim->start(QAbstractAnimation::DeleteWhenStopped);
#else
            setPos(newPos);
            emit noNeighbours();
#endif
        }

        return true;
    }
    return false;
}

void JigsawPuzzleItem::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->button() == Qt::LeftButton && _canMerge)
    {
        _dragging = true;
        _dragStart = event->pos();
        raise();
    }
}

void JigsawPuzzleItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->button() == Qt::LeftButton)
    {
        verifyCoveredSiblings();
        _dragging = false;
    }
}

void JigsawPuzzleItem::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    if (_dragging)
    {
        setPos(pos() + event->pos() - _dragStart);

        if (_canMerge)
        {
            foreach (PuzzleItem *p, neighbours())
            {
                JigsawPuzzleItem *w = (JigsawPuzzleItem*) p;
                QPointF posDiff1 = pos() - w->pos();
                QPointF posDiff2 = (puzzleCoordinates() * _unit - w->puzzleCoordinates() * _unit);
                QPointF relative = posDiff1 - posDiff2;

                if (abs(relative.x()) < _tolerance && abs(relative.y()) < _tolerance)
                {
                    merge(w);
                }
            }

            verifyPosition();
        }
    }
}

void JigsawPuzzleItem::verifyPosition()
{
    int x = pos().x();// + puzzleCoordinates().x() * _unit.width();
    int maxX = scene()->width() - _unit.width() / 3;
    int minX = - pixmap().width() + _unit.width() / 3;

    int y = pos().y();// + puzzleCoordinates().y() * _unit.height();
    int maxY = scene()->height() - _unit.height() / 3;
    int minY = - pixmap().height() + _unit.height() / 3;

    if (!(x < maxX && x > (minX) && y < maxY && y > (minY)))
    {
        int pX = CLAMP(x, minX + _unit.width() / 2, maxX - _unit.width() / 2);
        int pY = CLAMP(y, minY + _unit.height() / 2, maxY - _unit.height() / 2);
        _dragging = false;
#if QT_VERSION >= QT_VERSION_CHECK(4, 6, 0)
        QPropertyAnimation *anim = new QPropertyAnimation(this, "pos", this);
        anim->setEndValue(QPointF(pX, pY));
        anim->setDuration(200);
        anim->setEasingCurve(QEasingCurve(QEasingCurve::OutBounce));
        if (canMerge())
        {
            disableMerge();
            connect(anim, SIGNAL(finished()), this, SLOT(enableMerge()));
        }
        anim->start(QAbstractAnimation::DeleteWhenStopped);
#else
        setPos(pX, pY);
#endif
    }
}

void JigsawPuzzleItem::raise()
{
    QGraphicsItem *maxItem = this;
    foreach (QGraphicsItem *item, scene()->items())
    {
        if (item->zValue() > maxItem->zValue())
        {
            maxItem = item;
        }
    }
    if (maxItem != this)
    {
        qreal max = maxItem->zValue();
        foreach (QGraphicsItem *item, scene()->items())
        {
            if (item->zValue() > this->zValue())
            {
                item->setZValue(item->zValue() - 1);
            }
            else if (item != this && item->zValue() == this->zValue())
            {
                item->stackBefore(this);
            }
        }
        setZValue(max);
    }
}

void JigsawPuzzleItem::verifyCoveredSiblings()
{
    foreach (QGraphicsItem *gi, scene()->items())
    {
        JigsawPuzzleItem *item = (JigsawPuzzleItem*)gi;
        if (item != this &&
                item->zValue() < zValue() &&
                item->pos().x() >= pos().x() &&
                item->pos().y() >= pos().y() &&
                item->pos().x() + item->pixmap().width() < pos().x() + pixmap().width() &&
                item->pos().y() + item->pixmap().height() < pos().y() + pixmap().height() &&
                item->pixmap().width() < pixmap().width() &&
                item->pixmap().height() < pixmap().height())
        {
            item->raise();
            break;
        }
    }
}
