/*
    Copyright (C) <2010>  <Markus Scharnowski markus.scharnowski@gmail.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "timekeeper.h"
#include <sstream>
#include <iomanip>

using namespace std;

timeKeeper::timeKeeper()
{
  start();
  isRunning = false;
}

int timeKeeper::start()
{
  gettimeofday(&tvPrev,NULL);
  gettimeofday(&tvLast,NULL);
  gettimeofday(&tvStart,NULL);
  gettimeofday(&tvStop,NULL);
  timeStart = time(NULL);
  timeStop  = timeStart;
  timePrev  = timeStart;
  timeLast  = timeStart;
  pauseTimeSec = 0;
  pauseTotalTimeSec = 0;
  isRunning = true;
  isPauseActive = false;
  wasLastPauseLogged = true;
  disableDateForStrings();
  return EXIT_SUCCESS;
}

int timeKeeper::stop()
{
  if(false == isRunning)
  {
    return EXIT_FAILURE;
  }

  stopPause();
  timeStop = time(NULL);
  timeLast = timeStop;
  gettimeofday(&tvLast,NULL);
  gettimeofday(&tvStop,NULL);
  isRunning = false;

  return EXIT_SUCCESS;
}

int timeKeeper::update()
{
  if(false == isRunning)
  {
    return EXIT_FAILURE;
  }

  stopPause();
  timePrev = timeLast;
  timeLast = time(NULL);

  tvPrev = tvLast;
  gettimeofday(&tvLast,NULL);

  return EXIT_SUCCESS;
}

double timeKeeper::getDelta(struct timeval tvLater, struct timeval tvEarlier)
{
  double tvTime = (double)(tvLater.tv_sec - tvEarlier.tv_sec);

  tvTime = tvTime*1e6+(double)(tvLater.tv_usec - tvEarlier.tv_usec);
  tvTime /=1e6;

  return tvTime;
}

double timeKeeper::getDeltaLastToStart()
{
  double retval = 0;

  if(false == isRunning)
  {
    retval = getDelta(tvStop,tvStart);
  }
  else
  {
    retval = getDelta(tvLast,tvStart);
  }
  retval -= pauseTotalTimeSec;

  return retval;
}

double timeKeeper::getDeltaNowToStart()
{
  double retval = 0;
  struct timeval now;

  gettimeofday(&now,NULL);

  if(false == isRunning)
  {
    retval = getDelta(tvStop,tvStart);
  }
  else
  {
    retval = getDelta(now,tvStart);
  }
  retval -= pauseTotalTimeSec;

  return retval;
}

double timeKeeper::getDeltaLastToPrev()
{
  double retval = 0;

  if(false == isRunning)
  {
    return -1;
  }

  retval = getDelta(tvLast,tvPrev);

  if(false == wasLastPauseLogged)
  {
    retval -= pauseTimeSec;
    wasLastPauseLogged = true;
  }

  return retval;
}

double timeKeeper::getDeltalNowToPrev()
{
  double retval = 0;
  struct timeval now;

  gettimeofday(&now,NULL);

  if(false == isRunning)
  {
    retval = getDelta(tvStop,tvPrev);
  }
  else
  {
    retval = getDelta(now,tvPrev);
  }
  retval -= pauseTotalTimeSec;

  return retval;
}

string timeKeeper::makeString(time_t timer)
{
  stringstream sstr;
  struct tm *helper=NULL;

  helper=localtime(&timer);
  sstr << setfill('0');
  if(true == dateForStrings)
  {
    sstr << setw(2) << 1900+helper->tm_year << "-";
    sstr << setw(2) << helper->tm_mon << "-";
    sstr << setw(2) << helper->tm_mday << " ";
  }
  sstr << setw(2) << helper->tm_hour << ":";
  sstr << setw(2) << helper->tm_min << ":";
  sstr << setw(2) << helper->tm_sec;

  return sstr.str();
}

string timeKeeper::getStartTimeString()
{
  return makeString(timeStart);
}

string timeKeeper::getPreviousTimeString()
{
  return makeString(timePrev);
}

string timeKeeper::getLastTimeString()
{
  return makeString(timeLast);
}

string timeKeeper::getNowTimeString()
{
  time_t timeNow = time(NULL);
  return makeString(timeNow);
}

int timeKeeper::pause()
{
  if(false == isRunning)
  {
    return EXIT_FAILURE;
  }

  startPause();

  return EXIT_SUCCESS;
}

int timeKeeper::startPause()
{
  if(true == isPauseActive)
  {
    return stopPause();
  }
  gettimeofday(&tvPauseStart,NULL);
  isPauseActive = true;

  return EXIT_SUCCESS;
}

int timeKeeper::stopPause()
{
  if(false == isPauseActive)
  {
    return EXIT_FAILURE;
  }

  gettimeofday(&tvPauseStop,NULL);
  pauseTimeSec = getDelta(tvPauseStop,tvPauseStart);
  pauseTotalTimeSec += pauseTimeSec;
  wasLastPauseLogged = false;
  isPauseActive = false;

  return EXIT_SUCCESS;
}

int timeKeeper::isPaused()
{
  return isPauseActive;
}

/*
Date string handling
*/
bool timeKeeper::enableDateForStrings()
{
  dateForStrings = true;
  return dateForStrings;
}

bool timeKeeper::disableDateForStrings()
{
  dateForStrings = false;
  return dateForStrings;
}

bool timeKeeper::isDateForStringsEnabled()
{
  return dateForStrings;
}
