#include "ptssWidget.h"
#include "ptssConfiguration.h"

#include <QVBoxLayout>
#include <QCheckBox>
#include <QPushButton>

#include <QApplication>
#include <QTime>
#include <QDate>

#include <iostream>

inline int max(int a, int b) { return a>b?a:b; }

PtssWidget::PtssWidget(QString path, QWidget*)
{
	setAttribute(Qt::WA_Maemo5StackedWindow);
	initUI();
	
	db=new PtssDb(path);
	
	//default settings
	if (db->getSetting("timeResultsCount").isEmpty())
		db->setSetting("timeResultsCount", "15");
	
	if (db->getSetting("timeTolerance").isEmpty())
		db->setSetting("timeTolerance", "5");
	
	
	groupList->addItems(db->getGroupsLabels());
	
	updateTimes();
	
	t.start(5*60*1000);
}



PtssWidget::~PtssWidget()
{
	delete db;
}



void PtssWidget::initUI()
{
	QVBoxLayout* myLayout = new QVBoxLayout(this);
	QHBoxLayout *topLayout = new QHBoxLayout();
	
	groupList=new QComboBox();
	timesList=new QListWidget();
	
	timesList->setStyleSheet("::item {margin: 0px; padding: 0px; height: 1.1em;}");
	topLayout->addWidget(groupList);
	myLayout->addLayout(topLayout);
	myLayout->addWidget(timesList);

#ifndef MAEMO
	QPushButton *cfgBtn=new QPushButton("&Configure");
	connect(cfgBtn, SIGNAL(clicked()), SLOT(showConfigureDialog()));
	topLayout->addWidget(cfgBtn);
#endif

	setLayout(myLayout);
	setWindowTitle("Public Transport Service Schedule");

	connect(groupList, SIGNAL(currentIndexChanged(int)), this, SLOT(updateTimes()));
	
	connect(&t, SIGNAL(timeout()), this, SLOT(updateTimes()));
}



void PtssWidget::showConfigureDialog()
{
	configureDialog = new PtssConfiguration(db, this);
	
	connect(configureDialog, SIGNAL(closing()), this, SLOT(configureDialogTerminated()));
	configureDialog->show();
}


void PtssWidget::configureDialogTerminated()
{
	QString group=groupList->currentText();
	
	groupList->clear();
	groupList->addItems(db->getGroupsLabels());
	
	groupList->setCurrentIndex(
		max(0, groupList->findText(group))
	);
	
	updateTimes();
	
	configureDialog = NULL; /* the widget is deleted automatically */
}



QTextStream out(stdout);


QStringList getCurrentTimes(PtssDb *db, QString group)
{
	int timeTolerance=db->getSetting("timeTolerance").toInt();
	int timeCount=db->getSetting("timeResultsCount").toInt();
	
	out << db->getSetting("timeResultsCount") << endl;
	out << timeCount << endl;
	
	QTime ct=QTime::currentTime();
	QDate cd=QDate::currentDate();
	
	int day=cd.dayOfWeek()-Qt::Monday;
	int min=ct.hour()*60+ct.minute()-timeTolerance;
	
	if (min<0) {
		//fix near-midnight condition
		min+=24*60;
		
		day--;
		if (day<0) day+=7;
	}
	
	QTextStream out(stdout);
	
	return db->getConnsFromGroupForTime(group, day, min, timeCount);
}

void PtssWidget::updateTimes()
{
	timesList->clear();
	timesList->insertItems(
		0,
		getCurrentTimes(db, groupList->currentText())
	);
}

int main(int argc, char** argv)
{
	QApplication app(argc, argv);
	QString dbpath = QString(getenv("HOME")) + "/.ptss.sqlite";

	if (argc>=3 && QString(argv[1])=="print") {
		PtssDb db(dbpath);
		QTextStream out(stdout);
		
		QStringList l=getCurrentTimes(&db, argv[2]);
		
		for (int i = 0; i < l.size(); i++)
			out << l[i] << endl;

		return 0;
	} else {
		PtssWidget win(dbpath);
		win.show();

		return app.exec();
	}
}
