#include "ptssDb.h"
#include <QFile>
#include <assert.h>

static QString dayNames[]= { "Mo", "Tu", "We", "Th", "Fr", "Sa", "Su" };

QList<int> DBGetDepartMinutes(QSqlDatabase db, QString linkId, int fromStop, int toStop, DayFlagArray day, bool exactDay, int hour=-1)
{
	QList<int> rv;
	
	QString q;
	QTextStream(&q) <<
		"SELECT day,hour,minute FROM conns, conns_single WHERE " <<
		"linkId = '" << linkId << "' " << " AND " <<
		"fromStop = " << fromStop << " AND " <<
		"toStop = " << toStop << " AND " <<
		"conns.id=conns_single.connId " <<
		"ORDER BY minute"
		;
	
	QSqlQuery query(q, db);
	
	while (query.next()) {
		DayFlagArray d = (DayFlagArray)query.value(0).toUInt();
		int h = query.value(1).toInt();
		int m = query.value(2).toInt();
		
		if (exactDay) {
			if (d!=day) continue;
		} else {
			if ((d & day) != day) continue;
		}
		
		if (hour==-1)
			rv.append(h*60+m);
		else if (hour==h)
			rv.append(h*60+m);
	}
	
	return rv;
}

int DBIdOfConn(QSqlDatabase db, QString linkId, int fromStop, int toStop)
{
	QString q;
	QTextStream(&q) <<
		"SELECT id FROM conns WHERE " <<
		"linkId = '" << linkId << "' " << " AND " <<
		"fromStop = " << fromStop << " AND " <<
		"toStop = " << toStop
		;
	
	QSqlQuery query(q, db);
	
	if (!query.next())
		return -1;
	else
		return query.value(0).toInt();
}


int DBAddConn(QSqlDatabase db, QString linkId, int fromStop, int toStop)
{
	QString qi;
	QTextStream(&qi) <<
		"INSERT INTO conns(linkId, fromStop, toStop) VALUES ('" <<
		linkId << "', " <<
		fromStop << ", " <<
		toStop << ")"
		;

	QSqlQuery ins(qi, db);
	
	return ins.lastInsertId().toInt();
}


PtssDb::PtssDb(QString path)
{
	QFile f(path);
	bool initDb=!f.exists();
		
	db=QSqlDatabase::addDatabase("QSQLITE");
	db.setDatabaseName(path);
	db.open();
	
	if (initDb) {
		QSqlQuery query;
		// groups
		query.exec("CREATE TABLE groups(id integer primary key asc, label string);");
		query.exec("CREATE TABLE groupContents(groupId int, connId int);");
		
		// bus/tram/metro stops 
		query.exec("CREATE TABLE stops(id integer primary key asc, name string);");
		
		// connections' times
		query.exec("CREATE TABLE conns(id integer primary key asc, linkId string, fromStop int, toStop int);");
		query.exec("CREATE TABLE conns_single(connId int, day int, hour int, minute int);");
		
		//settings
		query.exec("CREATE TABLE settings(name string primary key, value string);");
	}
	
	initializeModels();
}



PtssDb::~PtssDb()
{
	delete _stopsModel;
	delete _timesModel;
	delete _groupContentsModel;
	if (_connModel) 
		delete _connModel;
}



void PtssDb::initializeModels()
{
	_stopsModel=new PtssStopsTableModel(this, db);
	_stopsModel->setTable("stops");
	_stopsModel->setEditStrategy(QSqlTableModel::OnRowChange);
	_stopsModel->select();
	//_stopsModel->removeColumn(0); // don't show the ID
	_stopsModel->setHeaderData(0, Qt::Horizontal, tr("ID"));
	_stopsModel->setHeaderData(1, Qt::Horizontal, tr("Stop name"));
	
	_timesModel=new PtssTimesTableModel(this, db);
	
	_groupContentsModel=new PtssGroupContentsModel(this, db);
	
	_connModel = NULL;
}

PtssConnTableModel *PtssDb::getConnTableModel()
{
	// TODO: this is a way too nasty workaround for updating the table view...
	if (_connModel) {
		delete _connModel;
	}
	
	_connModel = new PtssConnTableModel (this, db);
	
	return _connModel;
}

QStringList PtssDb::getGroupsLabels()
{
	QStringList rv;
	
	QSqlQuery query("SELECT label FROM groups", db);
	
	while (query.next()) {
		rv.append(query.value(0).toString());
	}
	
	return rv;
}


void PtssDb::removeStopRecord(const QModelIndex& r)
{
	_stopsModel->removeRows(r.row(), 1);
}

unsigned int PtssDb::timetableDays(QString linkId, int fromStop, int toStop, bool onlyBiggestGroup)
{
	QString q;
	QTextStream(&q) <<
		"SELECT DISTINCT conns_single.day FROM conns, conns_single WHERE " <<
		"linkId = '" << linkId << "' " << " AND " <<
		"fromStop = " << fromStop << " AND " <<
		"toStop = " << toStop << " AND " <<
		"conns.id=conns_single.connId"
		;
	
	QSqlQuery query(q, db);
	
	DayFlagArray rv=0;
	while (query.next()) {
		DayFlagArray f=query.value(0).toUInt();
		if (!onlyBiggestGroup) {
			rv|=f;
		} else {
			if (rv<f) rv=f;
		}
	}
	
	return rv;
}


QStringList PtssDb::getUsedLinkIds()
{
	QSqlQuery query(db);
	QStringList rv;
	
	query.exec("SELECT DISTINCT linkId FROM conns;");
	
	while (query.next()) {
		rv.append(query.value(0).toString());
	}
	
	return rv;
}


QString PtssDb::getSetting(QString name)
{
	QSqlQuery query(db);
	
	query.exec(QString("SELECT value FROM settings WHERE name='")+name+"';");
	
	if (!query.next())
		return QString();
	
	return query.value(0).toString();
}

void PtssDb::setSetting(QString name, QString val)
{
	QSqlQuery query(db);
	
	if (getSetting(name)==QString()) {
		query.exec(QString("INSERT INTO settings VALUES('")+name+"', '"+val+"')");
	} else {
		query.exec(QString("UPDATE settings SET value='")+val+"' WHERE name='"+name+"'");
	}
}


QVariant PtssStopsTableModel::data(const QModelIndex &item, int role) const
{
	if (item.row()==(rowCount()-1)) {
		if (role==Qt::DisplayRole) {
			if (item.column()==0)
				return QVariant(-1);
			else
				return QString("<new value>");
		} else {
			return QVariant();
		}
	} else {
		return QSqlTableModel::data(item, role);
	}
}


bool PtssStopsTableModel::setData(const QModelIndex &item, const QVariant &value, int role)
{
	if (item.row()==(rowCount()-1)) {
		if (role==Qt::DisplayRole || role==Qt::EditRole) {
			QSqlField n("name", QVariant::String);
			n.setValue(value.toString());
			
			QSqlRecord r;
			r.append(n);
			
			return insertRecord(rowCount()-1, r);
		} else
			return false;
	} else {
		return QSqlTableModel::setData(item, value, role);
	}
}


Qt::ItemFlags PtssTimesTableModel::flags(const QModelIndex &index) const
{
	Qt::ItemFlags flags = QAbstractTableModel::flags(index);
	if (index.column() == 1)
		flags |= Qt::ItemIsEditable;
	return flags;
}

QVariant PtssTimesTableModel::headerData ( int section, Qt::Orientation orientation, int role) const
{
	if (orientation==Qt::Horizontal && role==Qt::DisplayRole) {
		if (section==0) return QString("Hour");
		if (section==1) return QString("Minutes");
	}
	
	return QVariant();
}


QVariant PtssTimesTableModel::data(const QModelIndex &index, int role) const
{
	QVariant value=QString("");
	
	if (role == Qt::DisplayRole || role==Qt::EditRole) {
		if (index.column()==0) {
			QString str;
			str.setNum(index.row());
			value=str;
		} else {
			QList<int> times=DBGetDepartMinutes(qDb, qLink, qStopFrom, qStopTo, qDays, true, index.row());
			
			QString str;
			
			for (int i=0; i<times.length(); i++) {
				QString n;
				n.setNum(times[i]-(index.row())*60);
				str+=n;
				if (i!=(times.length()-1))
					str+=" ";
			}
			value=str;
		}
		
		return value;
	}
	
	return QVariant();
}


bool PtssTimesTableModel::setData(const QModelIndex &index, const QVariant &value, int /* role */)
{
	if (index.column() != 1)
		return false;

	int connId=DBIdOfConn(qDb, qLink, qStopFrom, qStopTo);
	if (connId==-1) {
		connId=DBAddConn(qDb, qLink, qStopFrom, qStopTo);
	}

	int hour=index.row();
	
	//first delete all data for that particular hour of day and conn
	QString qd;
	QTextStream(&qd) <<
		"DELETE FROM conns_single WHERE " <<
		"connId = " << connId << " AND " <<
		"day = " << qDays << " AND " <<
		"hour = " << hour
		;
	
	QSqlQuery del(qd, qDb);
	
	QStringList vals=value.toString().split(" ");
	
	for (int i=0; i<vals.length(); i++) {
		bool isInt;
		int min=vals[i].toInt(&isInt);
		
		if (!isInt || min<0 || min>59)
			continue;
		
		QString qi;
		QTextStream(&qi) <<
			"INSERT INTO conns_single VALUES (" <<
			connId << ", " <<
			qDays << ", " <<
			hour << ", " <<
			min << ")"
			;

		QSqlQuery ins(qi, qDb);
	}
	return true;
}

PtssConnTableModel::PtssConnTableModel(QObject *parent, QSqlDatabase db): QSqlQueryModel(parent)
{
	setQuery(
		"SELECT linkId, stop1.name, stop2.name, conns.id, fromStop, toStop "
		"FROM conns, stops as stop1, stops as stop2 WHERE "
		"conns.fromStop=stop1.id AND "
		"conns.toStop=stop2.id", db
	);
}


QVariant PtssConnTableModel::headerData ( int section, Qt::Orientation orientation, int role) const
{
	if (orientation==Qt::Horizontal && role==Qt::DisplayRole) {
		if (section==0) return QString("Link ID");
		if (section==1) return QString("From Stop");
		if (section==2) return QString("To Stop");
		if (section==3) return QString("ID");
	}
	
	return QVariant();
}


QVariant PtssConnTableModel::data(const QModelIndex &item, int role) const
{
	if (role == Qt::DisplayRole) {
		return QSqlQueryModel::data(item, role);
	} else if (role == Qt::EditRole) {
		if (item.column()==0) return QSqlQueryModel::data(item, Qt::DisplayRole);
		if (item.column()==1) return QSqlQueryModel::data(index(item.row(), 4), Qt::DisplayRole);
		if (item.column()==2) return QSqlQueryModel::data(index(item.row(), 5), Qt::DisplayRole);
		else return QSqlQueryModel::data(item, role);
	}
	
	return QVariant();
}



void PtssGroupContentsModel::setGroupName(QString groupName)
{
	_groupName=groupName;
	
	QString q;
	QTextStream(&q) <<
		"SELECT linkId, stop1.name, stop2.name, conns.id " <<
		"FROM groups, groupContents, stops as stop1, stops as stop2, conns WHERE " <<
		"groups.label = '" << groupName << "' AND " <<
		"groups.id=groupContents.groupId AND " <<
		"groupContents.connId=conns.id AND " <<
		"conns.fromStop=stop1.id AND " <<
		"conns.toStop=stop2.id";
	
	setQuery(q, qDb);
}

void PtssGroupContentsModel::insertConnection(int connId)
{
	QString q;
	QTextStream(&q) <<
		"SELECT id " <<
		"FROM groups WHERE " <<
		"label = '" << _groupName << "'";
	
	QSqlQuery qGroupId(q, qDb);
	int groupId;
	
	if (qGroupId.next()) {
		groupId=qGroupId.value(0).toInt();
	} else {
		QString qg;
		QTextStream(&qg) <<
			"INSERT INTO groups(label) VALUES ('" << _groupName << "')";

		QSqlQuery ins(qg, qDb);
		
		groupId=ins.lastInsertId().toInt();
	}
	
	QString qi;
	QTextStream(&qi) <<
		"INSERT INTO groupContents(groupId, connId) VALUES (" <<
		groupId << ", " <<
		connId << ")";
		;
	
	QSqlQuery query(qi, qDb);
	
	//TODO: doesn't do anything :-/
	reset();
}


void PtssGroupContentsModel::removeConnection(int connId)
{
	QString q;
	QTextStream(&q) <<
		"SELECT id " <<
		"FROM groups WHERE " <<
		"label = '" << _groupName << "'";
	
	QSqlQuery qGroupId(q, qDb);
	assert(qGroupId.next());
	int groupId=qGroupId.value(0).toInt();
	
	QString qd;
	QTextStream(&qd) <<
		"DELETE FROM groupContents WHERE " <<
		"groupId = " << groupId << " AND " <<
		"connId = " << connId 
		;
	
	QSqlQuery query(qd, qDb);
	
	//TODO: doesn't do anything :-/
	reset();
}


QVariant PtssGroupContentsModel::data(const QModelIndex &item, int role) const
{
	if (role == Qt::DisplayRole) {
		if (item.column() == 0) {
			QString link=QSqlQueryModel::data(QSqlQueryModel::index(item.row(), 0), role).toString();
			QString s1	=QSqlQueryModel::data(QSqlQueryModel::index(item.row(), 1), role).toString();
			QString s2	=QSqlQueryModel::data(QSqlQueryModel::index(item.row(), 2), role).toString();
		
			return link+" ("+s1+" => "+ s2+ ")";
		} else if (item.column() == 1) {
			return QSqlQueryModel::data(QSqlQueryModel::index(item.row(), 3), role);
		}
	}
	
	return QVariant();
}

QStringList PtssDb::getConnsFromGroupForTime(QString groupName, int day, int minute, int count)
{
	QStringList rv;
	
	// Plan: let's take all the connections from current group for current and following day
	// then filter out too early connections (sooner than the minute specified) and return
	// the first "count" items... trivial stuff... :)
	
	QList<SingleConnectionTime> times;
	times += getConnsFromGroupForDay(groupName, 1 << (6-day), day);
	times += getConnsFromGroupForDay(groupName, 1 << (6-((day+1)%7)), (day+1)%7 );
	
	QList<SingleConnectionTime>::iterator timesIt;
	
	for (timesIt = times.begin(); timesIt != times.end() && count != 0; timesIt++) {
		if ((timesIt->minute >= minute) || (timesIt->tag != day)) {
			rv << timesIt->linkId + " / " + dayNames[timesIt->tag] + " " +
						QString("%1").arg(int(timesIt->minute / 60), 2, 10, QChar('0')) + ":" +
						QString("%1").arg(int(timesIt->minute % 60), 2, 10, QChar('0')) +
						" (" + timesIt->fromStop + " => " + timesIt->toStop + ")";

			count--;
		}
	}
	
	return rv;
}

QList<SingleConnectionTime> PtssDb::getConnsFromGroupForDay(QString groupName, DayFlagArray day, int tag)
{
	QList<SingleConnectionTime> rv;
	
	QString q;
	QTextStream(&q) <<
		"SELECT day,hour,minute, linkId, stop1.name, stop2.name " <<
		"FROM groups, groupContents, stops as stop1, stops as stop2, conns, conns_single WHERE " <<
		"groups.label = '" << groupName << "' AND " <<
		"groups.id=groupContents.groupId AND " <<
		"groupContents.connId=conns.id AND " <<
		"conns.fromStop=stop1.id AND " <<
		"conns.toStop=stop2.id AND " <<
		"conns_single.connId=conns.id " <<
		"ORDER BY hour,minute"
		;
	
	QSqlQuery query(q, db);
	
	while (query.next()) {
		DayFlagArray d = (DayFlagArray)query.value(0).toUInt();
		if (d & day) {
			SingleConnectionTime t;
			t.linkId=query.value(3).toString();
			t.fromStop=query.value(4).toString();
			t.toStop=query.value(5).toString();
			
			t.day=d;
			t.minute=query.value(1).toInt()*60 + query.value(2).toInt();
			t.tag=tag;
			
			rv+=t;
		}
	}
	return rv;
}
