#include "ptssConfiguration.h"
#include "ptssChooseConnection.h"
#include <assert.h>
#include <QLineEdit>
#include <QMessageBox>


inline static int GetComboBoxPathID(QComboBox *w)
{
	int i=w->currentIndex();
	
	QAbstractItemModel *m=w->model();
	QVariant d=m->data(m->index(i, 0));
	
	return d.toInt();
}


PtssConfiguration::PtssConfiguration(PtssDb *db, QWidget *parent)
: QWidget(parent)
{
	_db=db;
	
	ui.setupUi(this);
	
	setAttribute(Qt::WA_DeleteOnClose);
	
#ifdef Q_WS_MAEMO_5
	setAttribute(Qt::WA_Maemo5StackedWindow);
	setWindowFlags(windowFlags() | Qt::Window);
#endif
	
	initGroups();
	initStops();
	initTimes();
	
	initSettings();
	
	updateTimesView_conn();
}

void PtssConfiguration::closeEvent(QCloseEvent *)
{
	_db->setSetting("timeResultsCount", QString::number(ui.timeResultsCount_spinBox->value()));
	_db->setSetting("timeTolerance", QString::number(ui.timeTolerance_spinBox->value()));
}

void PtssConfiguration::initSettings()
{
	QString val;
	
	val=_db->getSetting("timeResultsCount");
	if (!val.isEmpty())
		ui.timeResultsCount_spinBox->setValue(val.toInt());
	
	val=_db->getSetting("timeTolerance");
	if (!val.isEmpty())
		ui.timeTolerance_spinBox->setValue(val.toInt());
}

void PtssConfiguration::initGroups()
{
	connect(ui.group_combo, SIGNAL(editTextChanged(QString)), this, SLOT(updateGroupView()));
	connect(ui.group_addToList, SIGNAL(clicked()), this, SLOT(addConnToGroup()));
	connect(ui.group_rmFromList, SIGNAL(clicked()), this, SLOT(rmConnFromGroup()));
	
	updateGroupView_full();
}

void PtssConfiguration::initStops()
{
	stopsModel=_db->getStopsTableModel();
	
	ui.stops_list->setModel(stopsModel);
	ui.stops_list->setModelColumn(1);
	
	connect(ui.stops_rm, SIGNAL(clicked()), this, SLOT(rmStop()));
	
	connect(ui.stops_list->selectionModel(), SIGNAL(selectionChanged(QItemSelection, QItemSelection)), this, SLOT(stopSelected()));
	stopSelected();
}

void PtssConfiguration::initTimes()
{
	ui.times_linkNo->insertItems(0, _db->getUsedLinkIds());
	QStringList linkIds;
	
	QStringList days;
	days << "All week" << "Work days" << "Weekend" << "Monday" << "Tuesday" << "Wednesday" << "Thursday" << "Friday" << "Saturday" << "Sunday";
	ui.times_day->addItems(days);
	
	ui.times_pathFrom->setModel(stopsModel);
	ui.times_pathFrom->setModelColumn(1);
	ui.times_pathTo->setModel(stopsModel);
	ui.times_pathTo->setModelColumn(1);
	
	connect(ui.times_linkNo, SIGNAL(editTextChanged(QString)), this, SLOT(updateTimesView_conn()));
	connect(ui.times_pathFrom, SIGNAL(currentIndexChanged(int)), this, SLOT(updateTimesView_conn()));
	connect(ui.times_pathTo, SIGNAL(currentIndexChanged(int)), this, SLOT(updateTimesView_conn()));
	
	connect(ui.times_day, SIGNAL(currentIndexChanged(int)), this, SLOT(updateTimesView_day()));
	connect(ui.times_unused, SIGNAL(clicked(bool)), this, SLOT(updateTimesView_unused(bool)));
}


void PtssConfiguration::addConnToGroup()
{
	PtssChooseConnection cc(_db, this);
	
	QString groupName=ui.group_combo->currentText();
	
	assert(groupName != "");

	int connId;
	
	cc.exec();
	
	connId=cc.getSelected();
	if (connId <= -1) {
		return;
	}
	
	static_cast<PtssGroupContentsModel*>(ui.group_list->model())->insertConnection(connId);
	
	updateGroupView_full();
}


void PtssConfiguration::rmConnFromGroup()
{
	QModelIndex l=ui.group_list->selectionModel()->currentIndex();
	
	PtssGroupContentsModel *m=static_cast<PtssGroupContentsModel*>(ui.group_list->model());
	
	if (l.isValid()) {
		m->removeConnection(
			m->data(m->index(l.row(), 1), Qt::DisplayRole).toInt()
		);
	}
	
	updateGroupView_full();
}


void PtssConfiguration::rmStop()
{
	QModelIndex l=ui.stops_list->selectionModel()->currentIndex();
	
	if (l.isValid())
		_db->removeStopRecord(l);
}



void PtssConfiguration::stopSelected()
{
	QModelIndexList l=ui.stops_list->selectionModel()->selectedIndexes();
	
	ui.stops_rm->setEnabled(
		(!l.empty()) &&
		(l[0].row() != (ui.stops_list->model()->rowCount()-1))
	);
}


void PtssConfiguration::updateGroupView_full()
{
	QString cg=ui.group_combo->currentText();
	
	ui.group_combo->blockSignals(true);
	ui.group_combo->clear();
	ui.group_combo->addItems(_db->getGroupsLabels());
	ui.group_combo->lineEdit()->setText(cg);
	ui.group_combo->blockSignals(false);
	
	updateGroupView();
}

void PtssConfiguration::updateGroupView()
{
	QString cg=ui.group_combo->currentText();
	
	ui.group_list->setModel(_db->getGroupContentsModel(cg));
	
	ui.group_addToList->setEnabled(cg != "");
	ui.group_rmFromList->setEnabled(ui.group_list->model()->rowCount() != 0);
}


void PtssConfiguration::updateTimesView_conn()
{
	QString linkNo=ui.times_linkNo->currentText();
	int pathFrom=GetComboBoxPathID(ui.times_pathFrom);
	int pathTo=GetComboBoxPathID(ui.times_pathTo);
	
	unsigned int timetableDays=_db->timetableDays(linkNo, pathFrom, pathTo, true);
	
	ui.times_day->setCurrentIndex(0);
		
	if (timetableDays) {
		for (unsigned int i=0; i<sizeof(DayIdxTable)/sizeof(DayFlagArray); i++) {
			if (timetableDays == DayIdxTable[i]) {
				ui.times_day->setCurrentIndex(i);
				break;
			}
		}
	}
	
	updateTimesView_day();
}


bool isColumnEmpty(QAbstractItemModel *model, int rows, int column)
{
	bool dataAvailable=false;
	
	for (int i=0; i<rows; i++) {
		dataAvailable |= (!model->data(model->index(i, column)).toString().isEmpty());
	}
	
	return !dataAvailable;
}

void PtssConfiguration::updateTimesView_day()
{
	QString linkNo=ui.times_linkNo->currentText();
	int pathFrom=GetComboBoxPathID(ui.times_pathFrom);
	int pathTo=GetComboBoxPathID(ui.times_pathTo);
	int days=ui.times_day->currentIndex();
	
	QAbstractItemModel *timesModel=_db->getTimesTableModel(linkNo, pathFrom, pathTo, DayIdxTable[days]);
	ui.times_list->setModel(timesModel);
	ui.times_list->reset();
	ui.times_list->resizeColumnToContents(0);
	
	bool listEmpty=isColumnEmpty(timesModel, HOURS_PER_DAY, 1);
	
	ui.times_unused->setChecked(listEmpty);
	ui.times_list->setEnabled(!listEmpty);
}


void PtssConfiguration::updateTimesView_unused(bool unused)
{
	if (unused) {
		if (!isColumnEmpty(ui.times_list->model(), HOURS_PER_DAY, 1)) {
			
			if (QMessageBox::question(this, "Remove records?",
						"Remove time records for the current connection and day selection?",
						QMessageBox::Yes | QMessageBox::No)
				== QMessageBox::No)
			{
				ui.times_unused->setChecked(false);
				return;
			}
			
			//clear the table then...
			
			for (int i=0; i<HOURS_PER_DAY; i++) {
				QAbstractItemModel *m=ui.times_list->model();
				
				m->setData(m->index(i, 1), "", Qt::EditRole);
			}
		}
	}
	
		ui.times_list->setEnabled(!unused);
}
