/*
 * Mesa 3-D graphics library
 * Version:  4.0
 *
 * Copyright (C) 1999-2001  Brian Paul   All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * BRIAN PAUL BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * Author: Sami Kystil <sami.kyostila@gmail.com>
 *
 * Note: This API is a temporary hack until the GLX/DRI version is done.
 */


/* glesapi.c - public interface to GLES/Mesa functions */


#ifdef HAVE_CONFIG_H
#include "conf.h"
#endif

#if defined(__linux__)
#include <dlfcn.h>
#else
#error please port me
#endif

/* FIXME */
#define GLES
#if defined(GLES)
#include "glesdrv.h"

#include "drivers/common/driverfuncs.h"
#include "framebuffer.h"

/* Signify an unsupported (no-op) function */
#define GLES_UNSUPPORTED() \
   do { \
      if (GLES_DEBUG) { \
         fprintf(stderr, "Unsupported GLES operation: %s\n", __func__); \
      } \
   } while (0)

/* Check that the EGL error status is clear and return if not */
#define GLES_CHECK_EGL_ERROR_RETURN(CTX) \
   do { \
      EGLint _err = CTX->eglGetError(); \
      if (_err != EGL_SUCCESS) { \
         if (GLES_DEBUG) { \
            fprintf(stderr, "EGL/GLX operation %s:%d failed due to error 0x%x\n", __func__, __LINE__, _err); \
         } \
         return; \
      } \
  } while (0)

/* Check that the EGL error status is clear and go to a label if not */
#define GLES_CHECK_EGL_ERROR_GOTO(CTX, LABEL) \
   do { \
      EGLint _err = CTX->eglGetError(); \
      if (_err != EGL_SUCCESS ) { \
         if (GLES_DEBUG) { \
            fprintf(stderr, "EGL/GLX operation %s:%d failed due to error 0x%x\n", __func__, __LINE__, _err); \
         } \
         goto LABEL; \
      } \
  } while (0)

static glesMesaContext glesMesaCurrentCtx       = NULL;
static EGLint glesMesaConfigId                  = 0;
static glesGLXContext glesMesaCurrentGLXContext = NULL;

typedef void (*EGLFunc)(void);

/*
 * Create a new GLES/Mesa context and return a handle to it.
 */
glesMesaContext GLAPIENTRY
glesMesaCreateContext(glesGLXContext glxContext)
{
   glesMesaContext glesMesa = NULL;
   GLcontext *ctx = NULL, *shareCtx = NULL;
   struct dd_function_table functions;
   const char *str = NULL;

   /* Visual properties */
   GLboolean doubleBuffer = GL_FALSE;
   GLuint depthSize = 16, alphaSize = 0, stencilSize = 0, accumSize = 0;
   GLuint redBits = 5, greenBits = 6, blueBits = 5;

   glesMesa = (glesMesaContext)CALLOC_STRUCT(tglesMesaContext);
   if (!glesMesa) {
      str = "private context";
      goto errorhandler;
   }

   glesMesa->glxContext = glxContext;

   /* If we have a valid config, create a full rendering context */
   if (glxContext->config) {
      glxContext->eglGetConfigAttrib(glxContext->display, glxContext->config, EGL_RED_SIZE,     &redBits);
      glxContext->eglGetConfigAttrib(glxContext->display, glxContext->config, EGL_GREEN_SIZE,   &greenBits);
      glxContext->eglGetConfigAttrib(glxContext->display, glxContext->config, EGL_BLUE_SIZE,    &blueBits);
      glxContext->eglGetConfigAttrib(glxContext->display, glxContext->config, EGL_ALPHA_SIZE,   &alphaSize);
      glxContext->eglGetConfigAttrib(glxContext->display, glxContext->config, EGL_DEPTH_SIZE,   &depthSize);
      glxContext->eglGetConfigAttrib(glxContext->display, glxContext->config, EGL_STENCIL_SIZE, &stencilSize);

      if (stencilSize > STENCIL_BITS) {
         stencilSize = STENCIL_BITS;
      }

      glesMesa->glVis = _mesa_create_visual(GL_TRUE,	/* RGB mode */
                                          doubleBuffer,
                                          GL_FALSE,	/* stereo */
                                          redBits,	/* RGBA.R bits */
                                          greenBits,	/* RGBA.G bits */
                                          blueBits,	/* RGBA.B bits */
                                          alphaSize,	/* RGBA.A bits */
                                          0,		/* index bits */
                                          depthSize,	/* depth_size */
                                          stencilSize,	/* stencil_size */
                                          accumSize,
                                          accumSize,
                                          accumSize,
                                          alphaSize ? accumSize : 0,
                                          1);
      if (!glesMesa->glVis) {
         str = "_mesa_create_visual";
         goto errorhandler;
      }

      _mesa_init_driver_functions(&functions);
      ctx = glesMesa->glCtx = _mesa_create_context(glesMesa->glVis, shareCtx,
                                                   &functions, (void *) glesMesa);
      if (!ctx) {
         str = "_mesa_create_context";
         goto errorhandler;
      }

      if (!glesDDInitGLESMesaContext(glesMesa)) {
         str = "glesDDInitGLESMesaContext";
         goto errorhandler;
      }

      glesMesa->glBuffer = _mesa_create_framebuffer(glesMesa->glVis);
      
      if (!glesMesa->glBuffer) {
         str = "_mesa_create_framebuffer";
         goto errorhandler;
      }
   }
   return glesMesa;

errorhandler:
   if (glesMesa) {
      if (glesMesa->glBuffer) {
         _mesa_unreference_framebuffer(&glesMesa->glBuffer);
      }
      if (glesMesa->glVis) {
         _mesa_destroy_visual(glesMesa->glVis);
      }
      if (glesMesa->glCtx) {
         _mesa_destroy_context(glesMesa->glCtx);
      }
      FREE(glesMesa);
   }

   fprintf(stderr, "glesMesaCreateContext: ERROR: %s\n", str);
   return NULL;
}

/*
 * Destroy the given GLES/Mesa context.
 */
void GLAPIENTRY
glesMesaDestroyContext(glesMesaContext glesMesa)
{
   if (!glesMesa)
      return;

   if (glesMesa->glBuffer) {
      _mesa_unreference_framebuffer(&glesMesa->glBuffer);
   }
   if (glesMesa->glVis) {
      _mesa_destroy_visual(glesMesa->glVis);
   }
   if (glesMesa->glCtx) {
      glesDDDestroyGLESMesaContext(glesMesa); /* must be before _mesa_destroy_context */
      _mesa_destroy_context(glesMesa->glCtx);
   }

   FREE(glesMesa);

   if (glesMesa == glesMesaCurrentCtx)
      glesMesaCurrentCtx = NULL;
}


/*
 * Make the specified GLES/Mesa context the current one.
 */
void GLAPIENTRY
glesMesaMakeCurrent(glesMesaContext glesMesa)
{
   GLcontext *ctx = glesMesa->glCtx;
   EGLSurface surface;
   EGLint width, height;

   if (!glesMesa) {
      _mesa_make_current(NULL, NULL, NULL);
      glesMesaCurrentCtx = NULL;
      return;
   }

   /* Make the context current if it already isn't */
   if (glesMesaCurrentCtx != glesMesa ||
       glesMesaCurrentCtx->glCtx != _mesa_get_current_context()) {
      glesMesaCurrentCtx = glesMesa;
      _mesa_make_current(glesMesa->glCtx, glesMesa->glBuffer, glesMesa->glBuffer);
      glesSetupDDPointers(glesMesa->glCtx);
   }

   /* Check whether the window size has changed */
   glesMesa->glxContext->eglQuerySurface(glesMesa->glxContext->display, glesMesa->glxContext->surface, EGL_WIDTH,  &width);
   glesMesa->glxContext->eglQuerySurface(glesMesa->glxContext->display, glesMesa->glxContext->surface, EGL_HEIGHT, &height);

   if (width != ctx->DrawBuffer->Width || height != ctx->DrawBuffer->Height) {
      _mesa_resize_framebuffer(ctx, glesMesa->glBuffer, width, height);
   }
}

/*
 * Swap front/back buffers for current context if double buffered.
 */
void GLAPIENTRY
glesMesaSwapBuffers(void)
{
   if (glesMesaCurrentCtx) {
      _mesa_notifySwapBuffers(glesMesaCurrentCtx->glCtx);
   }
}

/*
 *  GLX functions
 */

#define COPY_ATTRIB(NAME) \
   case GLX_ ## NAME: \
      *attribsOut++ = EGL_ ## NAME; \
      break

#define TRANSLATE_ATTRIB(GLXNAME, EGLNAME) \
   case GLX_ ## GLXNAME: \
      *attribsOut++ = EGL_ ## EGLNAME; \
      break

XVisualInfo* glXChooseVisual(Display *dpy, int screen,
                             int *attrib_list)
{
   EGLint attribs[64];
   EGLint *attribsOut = attribs;
   EGLint numConfigs;
   EGLConfig config;
   VisualID visualId = 0;
   XVisualInfo* visualInfo = NULL;
   glesGLXContext context;

   /* Create a temporary context for querying the visual */
   context = glXCreateContext(dpy, NULL, NULL, False);

   if (!context) {
      goto out_error;
   }

   /* Always specify the surface type */
   *attribsOut++ = EGL_SURFACE_TYPE;
   *attribsOut++ = EGL_WINDOW_BIT;

   /* A depth buffer is not often requested even if required? */
   *attribsOut++ = EGL_DEPTH_SIZE;
   *attribsOut++ = 16;

   /* Transform the GLX attributes to EGL attributes */
   while (attrib_list && *attrib_list != None && attribsOut < attribs + sizeof(attribs) - 1) {
      if (GLES_DEBUG) {
         fprintf(stderr, "glXChooseVisual: GLX attribute 0x%x = %d\n", *attrib_list, *(attrib_list + 1));
      }

      switch (*attrib_list++) {
      COPY_ATTRIB(BUFFER_SIZE);
      COPY_ATTRIB(LEVEL);
      COPY_ATTRIB(RED_SIZE);
      COPY_ATTRIB(GREEN_SIZE);
      COPY_ATTRIB(BLUE_SIZE);
      COPY_ATTRIB(ALPHA_SIZE);
      COPY_ATTRIB(DEPTH_SIZE);
      COPY_ATTRIB(STENCIL_SIZE);
      COPY_ATTRIB(MAX_PBUFFER_WIDTH);
      COPY_ATTRIB(MAX_PBUFFER_HEIGHT);
      COPY_ATTRIB(SAMPLE_BUFFERS);
      COPY_ATTRIB(SAMPLES);
      /* These attributes have no values */
      case GLX_RGBA:
      case GLX_USE_GL:
         continue;
      /* Ignore these attributes */
      case GLX_DOUBLEBUFFER:
      case GLX_STEREO:
      case GLX_ACCUM_RED_SIZE:
      case GLX_ACCUM_GREEN_SIZE:
      case GLX_ACCUM_BLUE_SIZE:
      case GLX_ACCUM_ALPHA_SIZE:
         attrib_list++;
         continue;
      default:
         /* Unknown attrib */
         if (GLES_DEBUG) {
            fprintf(stderr, "glXChooseVisual: Ignoring unknown attribute 0x%x\n", *(attrib_list - 1));
         }
         attrib_list++;
         continue;
      }
      /* Copy the attribute value */
      *attribsOut++ = *attrib_list++;
   }
   *attribsOut++ = EGL_NONE;

   if (GLES_DEBUG) {
      int i;
      for (i = 0; attribs[i] != EGL_NONE; i+=2) {
         fprintf(stderr, "glXChooseVisual: EGL attribute 0x%x = %d\n", attribs[i], attribs[i + 1]);
      }
   }

   fprintf(stderr, "glXChooseVisual: eglChooseConfig\n");
   context->eglChooseConfig(context->display, attribs, &config, 1, &numConfigs);
   GLES_CHECK_EGL_ERROR_GOTO(context, out_error);

   if (numConfigs != 1) {
      if (GLES_DEBUG) {
         fprintf(stderr, "glXChooseVisual: No suitable config found.\n");
      }
      return NULL;
   }

   if (context->eglGetConfigAttrib(context->display, config, EGL_NATIVE_VISUAL_ID,
                                   (EGLint*)&visualId) == EGL_FALSE ||
       !visualId) {
      /* Use the default visual when all else fails */
      XVisualInfo vi_in;
      int out_count;
      vi_in.screen = screen;

      visualInfo = XGetVisualInfo(dpy, VisualScreenMask, &vi_in, &out_count);
   } else {
      XVisualInfo vi_in;
      int out_count;

      vi_in.screen   = screen;
      vi_in.visualid = visualId;
      visualInfo = XGetVisualInfo(dpy, VisualScreenMask | VisualIDMask, &vi_in, &out_count);
   }

   /* Store the config id for later construction */
   context->eglGetConfigAttrib(context->display, config, EGL_CONFIG_ID, &glesMesaConfigId);
   GLES_CHECK_EGL_ERROR_GOTO(context, out_error);

   /* Tear down the context */
   glXDestroyContext(dpy, context);
   context = NULL;

   if (!visualInfo && GLES_DEBUG) {
      fprintf(stderr, "glXChooseVisual: No suitable visual found.\n");
   }

   return visualInfo;

out_error:
   if (context) {
      glXDestroyContext(dpy, context);
   }

   return NULL;
}

#define GLES_LOAD_EGL_FUNC(NAME) \
   do { \
      EGLFunc* _func = (EGLFunc*)dlsym(context->eglLibrary, #NAME); \
      *((EGLFunc*)&(context->NAME)) = _func; \
      if (!context->NAME) { \
         fprintf(stderr, "Unable to load EGL symbol: %s\n", #NAME); \
         assert(0); \
      } \
   } while( 0)

glesGLXContext glXCreateContext(Display *dpy, XVisualInfo *vis,
                                glesGLXContext share_list, Bool direct)
{
   glesGLXContext context = (glesGLXContext)CALLOC_STRUCT(tglesGLXContext);
   EGLint numConfigs;
   EGLint attribs[3];

   if (!context) {
      if (GLES_DEBUG) {
         fprintf(stderr, "glXCreateContext: Out of memory.\n");
      }
      goto out_error;
   }

   /* Load the required EGL functions */
   context->eglLibrary  = dlopen(EGL_LIBRARY_NAME, RTLD_NOW);
   context->thisLibrary = dlopen(NULL, RTLD_LAZY);

   if (!context->eglLibrary) {
      fprintf(stderr, "Unable to open EGL library: %s\n", dlerror());
      goto out_error;
   }

   GLES_LOAD_EGL_FUNC(eglGetError);
   GLES_LOAD_EGL_FUNC(eglGetDisplay);
   GLES_LOAD_EGL_FUNC(eglInitialize);
   GLES_LOAD_EGL_FUNC(eglTerminate);
   GLES_LOAD_EGL_FUNC(eglQueryString);
   GLES_LOAD_EGL_FUNC(eglGetProcAddress);
   GLES_LOAD_EGL_FUNC(eglGetConfigs);
   GLES_LOAD_EGL_FUNC(eglChooseConfig);
   GLES_LOAD_EGL_FUNC(eglGetConfigAttrib);
   GLES_LOAD_EGL_FUNC(eglCreateWindowSurface);
   GLES_LOAD_EGL_FUNC(eglCreatePixmapSurface);
   GLES_LOAD_EGL_FUNC(eglCreatePbufferSurface);
   GLES_LOAD_EGL_FUNC(eglDestroySurface);
   GLES_LOAD_EGL_FUNC(eglQuerySurface);
   GLES_LOAD_EGL_FUNC(eglSurfaceAttrib);
   GLES_LOAD_EGL_FUNC(eglBindTexImage);
   GLES_LOAD_EGL_FUNC(eglReleaseTexImage);
   GLES_LOAD_EGL_FUNC(eglSwapInterval);
   GLES_LOAD_EGL_FUNC(eglCreateContext);
   GLES_LOAD_EGL_FUNC(eglDestroyContext);
   GLES_LOAD_EGL_FUNC(eglMakeCurrent);
   GLES_LOAD_EGL_FUNC(eglGetCurrentContext);
   GLES_LOAD_EGL_FUNC(eglGetCurrentSurface);
   GLES_LOAD_EGL_FUNC(eglGetCurrentDisplay);
   GLES_LOAD_EGL_FUNC(eglQueryContext);
   GLES_LOAD_EGL_FUNC(eglWaitGL);
   GLES_LOAD_EGL_FUNC(eglWaitNative);
   GLES_LOAD_EGL_FUNC(eglSwapBuffers);
   GLES_LOAD_EGL_FUNC(eglCopyBuffers);

   context->display = context->eglGetDisplay(dpy);
   context->eglInitialize(context->display, NULL, NULL);
   GLES_CHECK_EGL_ERROR_GOTO(context, out_error);

   /* Look up the previously chosen config */
   if (vis) {
      attribs[0] = EGL_CONFIG_ID;
      attribs[1] = glesMesaConfigId;
      attribs[2] = EGL_NONE;
      context->eglChooseConfig(context->display, attribs, &context->config, 1, &numConfigs);
      GLES_CHECK_EGL_ERROR_GOTO(context, out_error);

      if (numConfigs != 1) {
         if (GLES_DEBUG) {
            fprintf(stderr, "glXCreateContext: No suitable config found.\n");
         }
         goto out_error;
      }

      context->context = context->eglCreateContext(context->display, context->config,
                                                   share_list ? share_list->context : EGL_NO_CONTEXT,
                                                   NULL);
      GLES_CHECK_EGL_ERROR_GOTO(context, out_error);
   }

   context->glesContext = glesMesaCreateContext(context);
   if (!context->glesContext) {
      goto out_error;
   }

   return context;

out_error:
   if (context) {
      if (context->context) {
         context->eglDestroyContext(context->display, context->context);
      }
      if (context->glesContext) {
         glesMesaDestroyContext(context->glesContext);
      }
      if (context->display) {
         context->eglTerminate(context->display);
      }
      if (context->eglLibrary) {
         dlclose(context->eglLibrary);
      }
      if (context->thisLibrary) {
         dlclose(context->thisLibrary);
      }
      FREE(context);
   }

   return NULL;
}

void glXDestroyContext(Display *dpy, glesGLXContext context)
{
   if (context) {
      if (context->context) {
         context->eglDestroyContext(context->display, context->context);
      }
      if (context->surface) {
         context->eglDestroySurface(context->display, context->surface);
      }
      if (context->glesContext) {
         glesMesaDestroyContext(context->glesContext);
      }
      if (context->display) {
         context->eglTerminate(context->display);
      }
      if (context->eglLibrary) {
         dlclose(context->eglLibrary);
      }
      if (context->thisLibrary) {
         dlclose(context->thisLibrary);
      }
      FREE(context);
   }
   if (context == glesMesaCurrentGLXContext) {
      glesMesaCurrentGLXContext = NULL;
   }
}

Bool glXMakeCurrent(Display *dpy, GLXDrawable drawable,
                    glesGLXContext context)
{
   if (!context || !context->display || !context->config || !context->context) {
      glesMesaCurrentGLXContext = NULL;
      return False;
   }

   /* Create the actual window surface if it doesn't exist yet */
   if (!context->surface) {
      context->surface = context->eglCreateWindowSurface(context->display, context->config,
                                                         drawable, NULL);
      GLES_CHECK_EGL_ERROR_GOTO(context, out_error);
   }

   context->eglMakeCurrent(context->display, context->surface,
                           context->surface, context->context);
   GLES_CHECK_EGL_ERROR_GOTO(context, out_error);

   glesMesaMakeCurrent(context->glesContext);
   glesMesaCurrentGLXContext = context;
   return True;

out_error:
   return False;
}

#define MAP_ATTRIB(NAME) \
   case GLX_ ## NAME: attrib = EGL_ ## NAME; break

int glXGetConfig(Display *dpy, XVisualInfo *vis,
                 int attrib, int *value)
{
   glesGLXContext context = glesMesaCurrentGLXContext;

   if (!context || !context->display || !context->config) {
      return GLX_BAD_ATTRIBUTE;
   }

   /* Translate the GLX attribute to an EGL attribute */
   switch (attrib) {
      MAP_ATTRIB(RED_SIZE);
      MAP_ATTRIB(GREEN_SIZE);
      MAP_ATTRIB(BLUE_SIZE);
      MAP_ATTRIB(ALPHA_SIZE);
      MAP_ATTRIB(BUFFER_SIZE);
      MAP_ATTRIB(DEPTH_SIZE);
      MAP_ATTRIB(STENCIL_SIZE);
      MAP_ATTRIB(SAMPLE_BUFFERS);
      MAP_ATTRIB(SAMPLES);
      default:
         return GLX_BAD_ATTRIBUTE;
   }
   if (context->eglGetConfigAttrib(context->display, context->config,
                                   attrib, (EGLint*)value) == EGL_FALSE) {
      return GLX_BAD_ATTRIBUTE;
   }
   return 0;
}

#undef MAP_ATTRIB

void glXSwapBuffers(Display *dpy, GLXDrawable drawable)
{
   glesGLXContext context = glesMesaCurrentGLXContext;

   if (!context || !context->display || !context->surface || !context->surface) {
      return;
   }
   glesMesaSwapBuffers();
   context->eglSwapBuffers(context->display, context->surface);
   GLES_CHECK_EGL_ERROR_RETURN(context);
}

void glXWaitGL(void)
{
   glesGLXContext context = glesMesaCurrentGLXContext;

   if (!context) {
      return;
   }
   context->eglWaitGL();
}

void glXWaitX(void)
{
   glesGLXContext context = glesMesaCurrentGLXContext;
   
   if (!context) {
      return;
   }
   context->eglWaitNative(EGL_CORE_NATIVE_ENGINE);
}

const char *glXQueryExtensionsString( Display *dpy, int screen )
{
   glesGLXContext context = glesMesaCurrentGLXContext;
   
   if (!context || !context->display) {
      return NULL;
   }
   return context->eglQueryString(context->display, EGL_EXTENSIONS);
}

glesGLXContext glXGetCurrentContext(void)
{
   return glesMesaCurrentGLXContext;
}

Bool glXIsDirect(Display *dpy, glesGLXContext ctx)
{
   return True;
}

Bool glXQueryVersion(Display *dpy, int *major, int *minor)
{
   if (major) *major = 1;
   if (minor) *minor = 0;
   return True;
}

Bool glXQueryExtension(Display *dpy, int *error_base, int *event_base)
{
   if (error_base) *error_base = 0;
   if (event_base) *event_base = 0;
   return True;
}

void* glXGetProcAddressARB (const GLubyte *name)
{
   glesGLXContext context = glesMesaCurrentGLXContext;

   if (!context || !context->thisLibrary) {
      return NULL;
   }
   return dlsym(context->thisLibrary, name);
}

/*
 *  Unsupported stubs
 */

void glXUseXFont(Font font, int first, int count, int list_base)
{
   GLES_UNSUPPORTED();
}

void glXCopyContext(Display *dpy, glesGLXContext src,
                    glesGLXContext dst, unsigned long mask)
{
   GLES_UNSUPPORTED();
}

GLXPixmap glXCreateGLXPixmap(Display *dpy, XVisualInfo *vis,
                             Pixmap pixmap)
{
   GLES_UNSUPPORTED();
   return NULL;
}

GLXDrawable glXGetCurrentDrawable(void)
{
   GLES_UNSUPPORTED();
   return NULL;
}

void glXDestroyGLXPixmap(Display *dpy, GLXPixmap pix)
{
   GLES_UNSUPPORTED();
}

#else

/*
 * Need this to provide at least one external definition.
 */
extern int gl_GLES_dummy_function_api(void);
int
gl_GLES_dummy_function_api(void)
{
   return 0;
}

#endif /* GLES */
