#include "memorycardmanager.h"
#include "ui_memorycardmanager.h"
#include <QFileDialog>
#include "memorycardselect.h"

MemoryCardManager::MemoryCardManager(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MemoryCardManager)
{
    ui->setupUi(this);

    m_Card = NULL;
    m_SelectedSlot = -1;
#ifdef Q_WS_MAEMO_5
    this->setAttribute(Qt::WA_Maemo5StackedWindow);
    ui->listWidget->setIconSize(QSize(48,48));
#endif

    ui->actionSave->setEnabled(false);

#ifdef Q_WS_MAEMO_5
    connect(ui->listWidget, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(SlotChanged(QListWidgetItem*)));
#else
    connect(ui->listWidget, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(SlotChanged(QListWidgetItem*)));
#endif
    connect(ui->actionLoad, SIGNAL(triggered()), this, SLOT(LoadCard()));
    connect(ui->actionSave, SIGNAL(triggered()), this, SLOT(SaveCard()));
    connect(ui->pbnDeleteSlot, SIGNAL(clicked()), this, SLOT(DeleteSlot()));
}

MemoryCardManager::~MemoryCardManager()
{
    if (m_Card != NULL)
        delete m_Card;
    delete ui;
}

void MemoryCardManager::LoadCard()
{
#ifdef Q_WS_MAEMO_5
    QString filename = GetMemoryCardName();
#else
    QString filename = QFileDialog::getOpenFileName(this, tr("Open Memory card"),
                                                    QDir::homePath(),
                                                    tr("Memory cards (*.mcd)"));
#endif
    if (filename.isEmpty())
        return;

    ui->actionSave->setEnabled(true);
    m_Filename = filename;
    if (m_Card != NULL){
        delete m_Card;
        m_Card = NULL;
    }

    m_Card = new PSX_memory_card();
    m_Card->load_file(filename);
    m_Modified = false;

    QFileInfo f(filename);
    setWindowTitle(f.fileName());
    UpdateSlotList();
}

void MemoryCardManager::SaveCard()
{
    if (m_Card == NULL || m_Filename.isEmpty())
        return;

    m_Card->save_file(m_Filename);
    m_Modified = false;
    if (windowTitle().endsWith("*"))
        setWindowTitle(windowTitle().remove(windowTitle().length() - 1, 1));
}

void MemoryCardManager::UpdateSlotList()
{
    ui->listWidget->clear();
    for (int i=0; i<15; i++){
        if (m_Card->get_slot_is_used(i)){
            QString title = QString("%1 [%2]").arg(m_Card->get_slot_title(i)).arg(m_Card->get_slot_Pcode(i));
            QPixmap pixmap = m_Card->get_slot_icon(i);
#ifdef Q_WS_MAEMO_5
            QListWidgetItem* item = new QListWidgetItem(QIcon(pixmap.scaledToHeight(48)), title);
#else
            QListWidgetItem* item = new QListWidgetItem(QIcon(pixmap), title);
#endif
            ui->listWidget->addItem(item);
        }else{
#ifdef Q_WS_MAEMO_5
            QPixmap icon(48, 48);
#else
            QPixmap icon(16, 16);
#endif
            icon.fill(Qt::lightGray);
            QListWidgetItem* item = new QListWidgetItem(QIcon(icon), tr("[Free Slot]"));
            ui->listWidget->addItem(item);
        }
    }

    if (ui->listWidget->count()){
        ui->listWidget->setCurrentRow(0);
        SlotChanged(NULL);
    }

}

void MemoryCardManager::DeleteSlot()
{
    if (m_Card == NULL)
        return;

    if (m_Card->get_slot_is_used(m_SelectedSlot)){
        m_Card->delete_slot(m_SelectedSlot);
        UpdateSlotList();

        m_Modified = true;
        if (!windowTitle().endsWith("*"))
            setWindowTitle(QString("%1 *").arg(windowTitle()));
    }
}

#ifdef Q_WS_MAEMO_5
QString MemoryCardManager::GetMemoryCardName()
{
    MemoryCardSelect* dlg = new MemoryCardSelect(this);
    if (dlg->exec() == QDialog::Accepted)
        return dlg->Selected();
    return "";
}
#endif

void MemoryCardManager::SlotChanged(QListWidgetItem*)
{
    if (m_Card == NULL){
        ui->pbnDeleteSlot->setEnabled(false);
        return;
    }

    m_SelectedSlot = ui->listWidget->currentRow();
    ui->pbnDeleteSlot->setEnabled(m_Card->get_slot_is_used(ui->listWidget->currentRow()));
}

void MemoryCardManager::show()
{
    QMainWindow::show();
    LoadCard();
}
