/*
  Copyright (C) 2010, 2011  Sakari Hyoty

  This file is part of Pomodoro.

  Pomodoro is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Pomodoro is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Pomodoro.  If not, see <http://www.gnu.org/licenses/>.
  */

#include <QSound>
#include <QInputDialog>
#include <QMessageBox>
#include <phonon/AudioOutput>
#include <QSettings>

#ifdef Q_WS_MAEMO_5
#include <QtDBus/QDBusMessage>
#include <QtDBus/QDBusConnection>
#endif // Q_WS_MAEMO_5

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "pomodoromodel.h"
#include "pomodorotimer.h"
#include "pomodoroconfig.h"
#include "taskdialog.h"
#include "sounddialog.h"

QString secondsToString(int totalSeconds)
{
    int minutes = totalSeconds / 60;
    int seconds = totalSeconds % 60;

    QString str("%1:%2");
    str = str.arg(QVariant(minutes).toString(), 2, '0');
    str = str.arg(QVariant(seconds).toString(), 2, '0');

    return str;
}

void showSystemNotification(QString text)
{
    qDebug("showSystemNotification text=\"%s\"", text.toAscii().constData());
#ifdef Q_WS_MAEMO_5
    QDBusMessage msg =
        QDBusMessage::createMethodCall(
            "org.freedesktop.Notifications",
            "/org/freedesktop/Notifications",
            "org.freedesktop.Notifications",
            "SystemNoteInfoprint");

    QList<QVariant> args;
    args.append(text);

    msg.setArguments(args);

    QDBusConnection::systemBus().call(msg);
#endif // Q_WS_MAEMO_5
}

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    pomodoroModel(NULL),
    alertMediaObject(NULL),
    pomodoroTimer(NULL)
{
    ui->setupUi(this);

    // initialize buttons
    connect(ui->removeButton, SIGNAL(clicked()),
            this, SLOT(removeTask()));
    connect(ui->doneButton, SIGNAL(clicked()),
            this, SLOT(completeTask()));
    connect(ui->internalButton, SIGNAL(clicked()),
            this, SLOT(internalInterruption()));
    connect(ui->externalButton, SIGNAL(clicked()),
            this, SLOT(externalInterruption()));

    // initialize timer
    pomodoroTimer = new PomodoroTimer(this);
    pomodoroTimer->setLength(POMODORO_TIMER_LENGTH);

    connect(ui->startButton, SIGNAL(clicked()),
            pomodoroTimer, SLOT(start()));
    connect(ui->stopButton, SIGNAL(clicked()),
            pomodoroTimer, SLOT(stop()));
    connect(pomodoroTimer, SIGNAL(started(int)),
            this, SLOT(pomodoroStarted(int)));
    connect(pomodoroTimer, SIGNAL(stopped(bool)),
            this, SLOT(pomodoroStopped(bool)));
    connect(pomodoroTimer, SIGNAL(secondsLeft(int)),
            this, SLOT(updateProgress(int)));

    // initialize model
    pomodoroModel = new PomodoroModel(ui->tableView);
    QApplication::setOverrideCursor(Qt::WaitCursor);
    bool modelLoadStatus = pomodoroModel->loadModel();
    QApplication::restoreOverrideCursor();

    // initialize tableView
    QTableView *tableView = ui->tableView;
    tableView->setModel(pomodoroModel);
    connect(tableView, SIGNAL(doubleClicked(QModelIndex)),
            this, SLOT(editTask(QModelIndex)));

    // initialize headerView
    QHeaderView *headerView = tableView->horizontalHeader();
    headerView->setClickable(false);
    headerView->setResizeMode(0, QHeaderView::Stretch);
    headerView->setResizeMode(1, QHeaderView::ResizeToContents);
    headerView->setResizeMode(2, QHeaderView::ResizeToContents);
    headerView->setResizeMode(3, QHeaderView::ResizeToContents);
    headerView->setResizeMode(4, QHeaderView::ResizeToContents);

    QItemSelectionModel *selectionModel = tableView->selectionModel();
    connect(selectionModel, SIGNAL(selectionChanged(QItemSelection, QItemSelection)),
            this, SLOT(taskSelected(QItemSelection, QItemSelection)));

    taskDialog = new TaskDialog(this);
    connect(ui->addButton, SIGNAL(clicked()), taskDialog, SLOT(showAdd()));
    connect(taskDialog, SIGNAL(newTask(QString,int)),
            this, SLOT(addTask(QString,int)));
    connect(taskDialog, SIGNAL(taskUpdated(QModelIndex, QString, int)),
            this, SLOT(taskUpdated(QModelIndex, QString, int)));

    soundDialog = new SoundDialog(this);
    connect(soundDialog, SIGNAL(alarmFileChanged(QString)),
            this, SLOT(alarmFileChanged(QString)));
    connect(soundDialog, SIGNAL(alarmFileReset()),
            this, SLOT(alarmFileReset()));

    connect(ui->actionGuide, SIGNAL(triggered()), this, SLOT(showGuide()));
    connect(ui->actionAbout, SIGNAL(triggered()), this, SLOT(showAbout()));
    connect(ui->actionRemoveCompleted, SIGNAL(triggered()), this, SLOT(removeCompletedTasks()));
    connect(ui->actionChangeAlarm, SIGNAL(triggered()), this, SLOT(showChangeAlarmFile()));

    // load alarm sound file path from settings
    QSettings settings(POMODORO_FILE_SETTINGS, QSettings::IniFormat);
    alarmFile = settings.value(POMODORO_SETTINGS_KEY_ALARM,
                               POMODORO_FILE_ALERT).toString();

    alertMediaObject = new Phonon::MediaObject(this);
    Phonon::AudioOutput *audioOutput =
        new Phonon::AudioOutput(Phonon::NoCategory, this);
    Phonon::createPath(alertMediaObject, audioOutput);

    setUiState(IdleNoSelection);

    if (!modelLoadStatus) {
        showGuide();
    }
}

MainWindow::~MainWindow()
{
    delete alertMediaObject;
    alertMediaObject = NULL;

    delete taskDialog;
    taskDialog = NULL;
}

void MainWindow::addTask(QString taskName, int estimate)
{
    QModelIndex newTaskIndex = pomodoroModel->newTask(taskName, estimate);

    if (!pomodoroTimer->isStarted()) {
        selectedIndex = newTaskIndex;
        setUiState(IdleTaskSelected);
        ui->tableView->setCurrentIndex(selectedIndex);
        ui->tableView->scrollTo(
            selectedIndex,
            QAbstractItemView::PositionAtCenter);
    } else {
        showSystemNotification(tr("Task added"));
    }
}

void MainWindow::editTask(const QModelIndex &index)
{
    qDebug("MainWindow::editTask");

    if (!pomodoroModel->isTaskCompleted(index)) {
        taskDialog->showEdit(index, pomodoroModel->taskName(index),
                             pomodoroModel->estimate(index));
    } else {
        showSystemNotification(tr("Unable to edit a completed task"));
    }
}

void MainWindow::taskUpdated(const QModelIndex &index,
                             QString taskName, int estimate)
{
    qDebug("MainWindow::taskUpdated");

    selectedIndex = index;

    pomodoroModel->updateTask(selectedIndex, taskName, estimate);

    setUiState(IdleTaskSelected);
    ui->tableView->setCurrentIndex(selectedIndex);
    ui->tableView->scrollTo(
        selectedIndex,
        QAbstractItemView::PositionAtCenter);
}

void MainWindow::removeTask()
{
    qDebug("MainWindow::removeTask");

    pomodoroModel->deleteTask(selectedIndex);
    selectedIndex = QModelIndex();
    setUiState(IdleNoSelection);
}

void MainWindow::completeTask()
{
    qDebug("MainWindow::completeTask");

    if (pomodoroTimer->isStarted()) {
        pomodoroTimer->stop(false);
        pomodoroModel->pomodoroCompleted(selectedIndex);
    }

    pomodoroModel->taskCompleted(selectedIndex);
    ui->tableView->setCurrentIndex(selectedIndex);
    setUiState(IdleCompletedTaskSelected);
}

void MainWindow::internalInterruption()
{
    qDebug("MainWindow::internalInterruption");

    pomodoroModel->pomodoroInterrupted(selectedIndex, true);
    showSystemNotification(tr("Internal interruption recorded"));
}

void MainWindow::externalInterruption()
{
    qDebug("MainWindow::externalInterruption");

    pomodoroModel->pomodoroInterrupted(selectedIndex, false);
    showSystemNotification(tr("External interruption recorded"));
}


void MainWindow::taskSelected(const QItemSelection &selected,
    const QItemSelection &/*deselected*/)
{
    QModelIndexList indexes = selected.indexes();

    if (indexes.count()) {
        selectedIndex = selected.indexes().at(0);

        qDebug("MainWindow::taskSelected selectedIndex.row()=%i",
            selectedIndex.row());

        if (!pomodoroModel->isTaskCompleted(selectedIndex))
            setUiState(IdleTaskSelected);
        else
            setUiState(IdleCompletedTaskSelected);
    } else {
        qDebug("MainWindow::taskSelected (no selection)");
        setUiState(IdleNoSelection);
    }
}

void MainWindow::pomodoroStarted(int /*lengthInSeconds*/)
{
    qDebug("MainWindow::pomodoroStarted");

    // set task name to UI
    QModelIndex taskNameIndex =
        pomodoroModel->index(selectedIndex.row(), 0, QModelIndex());
    ui->currentTaskName->setText(
            taskNameIndex.data(Qt::DisplayRole).toString());

    // update progress UI
    ui->timerLabel->setText(secondsToString(POMODORO_TIMER_LENGTH));

    setUiState(Started);
}

void MainWindow::pomodoroStopped(bool finished)
{
    qDebug("MainWindow::pomodoroStopped %sfinished", finished?"":"!");

    if (finished) {
        pomodoroModel->pomodoroCompleted(selectedIndex);

        if (QFile(alarmFile).exists()) {
            showSystemNotification(tr("Pomodoro finished"));
            alertMediaObject->setCurrentSource(alarmFile);
        } else {
            showSystemNotification(tr("Selected custom alarm file does not exist"));
            alertMediaObject->setCurrentSource(POMODORO_FILE_ALERT);
        }

        alertMediaObject->play();
    }

    setUiState(IdleTaskSelected);

    // make UI point to correct task
    ui->tableView->setCurrentIndex(selectedIndex);
}

void MainWindow::updateProgress(int secondsLeft)
{
    qDebug("MainWindow::updateProgress secondsLeft=%i", secondsLeft);

    if (secondsLeft == POMODORO_TIMER_ALMOST_FINISHED)
        setUiState(AlmostFinished);

    ui->timerLabel->setText(secondsToString(secondsLeft));
}

void MainWindow::showGuide()
{
    qDebug("MainWindow::showGuide");

    QString title = tr("What is a pomodoro?");
    QString text =
            tr("Pomodoro is a 25 minute session of focused work. Between pomodoros "
               "take a 5 minute break and every 4 pomodoros a longer break. "
               "Internal interruption means that you have been interrupted by "
               "yourself, you did something not in the scope of the task. "
               "External interruption means that someone else interrupted you."
               "\n\n"
               "See http://www.pomodorotechnique.com/ for details on The Pomodoro Technique.");

    QMessageBox::information(this, title, text);

    title = tr("How to use this application?");
    text = tr("Add a task and estimate it in pomodoros. "
              "Start the task and work until the "
              "time runs out. Interruptions can be recorded by clicking "
              "on Int./Ext. button depending on the distraction type."
              "\n\n"
              "Mark the task completed when it's done. Completed tasks can "
              "be mass removed from the menu. Tasks can be edited "
              "by double clicking.\n\n"
              "You can access this guide again at any time from the application menu.");

    QMessageBox::information(this, title, text);
}

void MainWindow::showAbout()
{
    qDebug("MainWindow::showAbout");

    QString title = tr("About");
    QString text =
            tr("Pomodoro %1\nCopyright (C) 2010, 2011 Sakari Hyoty\n\n"
               "%2\n\n"
               "This program comes with ABSOLUTELY NO WARRANTY. This is free software, "
               "and you are welcome to redistribute it under certain conditions; see `%3' "
               "for details.").arg(POMODORO_VERSION).arg(POMODORO_HOMEPAGE).arg(POMODORO_README_FILE);

    QMessageBox::about(this, title, text);
}

void MainWindow::removeCompletedTasks()
{
    qDebug("MainWindow::removeCompletedTasks");

    pomodoroModel->deleteCompletedTasks();
    selectedIndex = QModelIndex();
    setUiState(IdleNoSelection);
}

void MainWindow::showChangeAlarmFile()
{
    qDebug("MainWindow::showChangeAlarmFile");

    soundDialog->show(alarmFile);
}

void MainWindow::alarmFileChanged(QString newAlarmFile)
{
    qDebug("MainWindow::alarmFileChanged");

    alarmFile = newAlarmFile;

    QSettings settings(POMODORO_FILE_SETTINGS, QSettings::IniFormat);
    settings.setValue(POMODORO_SETTINGS_KEY_ALARM, alarmFile);
    settings.sync();
}

void MainWindow::alarmFileReset()
{
    qDebug("MainWindow::alarmFileReset");

    alarmFile = POMODORO_FILE_ALERT;

    QFile file(POMODORO_FILE_SETTINGS);
    file.remove();
}

void MainWindow::changeEvent(QEvent *e)
{
    //qDebug("MainWindow::changeEvent");

    QMainWindow::changeEvent(e);

    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::setUiState(PomodoroUiState state)
{
    qDebug("MainWindow::setUiState state=%i", state);

    // developer: keep lines alphabetically sorted
    switch (state) {
    case IdleNoSelection:
        ui->actionChangeAlarm->setEnabled(true);
        ui->actionRemoveCompleted->setEnabled(true);
        ui->currentTaskName->setText(QString(""));
        ui->currentTaskName->setVisible(false);
        ui->doneButton->setEnabled(false);
        ui->externalButton->setEnabled(false);
        ui->internalButton->setEnabled(false);
        ui->timerLabel->setVisible(false);
        ui->removeButton->setEnabled(false);
        ui->startButton->setEnabled(false);
        ui->stopButton->setEnabled(false);
        ui->tableView->setVisible(true);
        break;
    case IdleTaskSelected:
        ui->actionChangeAlarm->setEnabled(true);
        ui->actionRemoveCompleted->setEnabled(true);
        ui->currentTaskName->setText(QString(""));
        ui->currentTaskName->setVisible(false);
        ui->doneButton->setEnabled(true);
        ui->externalButton->setEnabled(false);
        ui->internalButton->setEnabled(false);
        ui->timerLabel->setVisible(false);
        ui->removeButton->setEnabled(true);
        ui->startButton->setEnabled(true);
        ui->stopButton->setEnabled(false);
        ui->tableView->setVisible(true);
        break;
    case IdleCompletedTaskSelected:
        ui->actionChangeAlarm->setEnabled(true);
        ui->actionRemoveCompleted->setEnabled(true);
        ui->currentTaskName->setText(QString(""));
        ui->currentTaskName->setVisible(false);
        ui->doneButton->setEnabled(false);
        ui->externalButton->setEnabled(false);
        ui->internalButton->setEnabled(false);
        ui->timerLabel->setVisible(false);
        ui->removeButton->setEnabled(true);
        ui->startButton->setEnabled(false);
        ui->stopButton->setEnabled(false);
        ui->tableView->setVisible(true);
        break;
    case Started:
        ui->actionChangeAlarm->setEnabled(false);
        ui->actionRemoveCompleted->setEnabled(false);
        ui->currentTaskName->setVisible(true);
        ui->doneButton->setEnabled(false);
        ui->externalButton->setEnabled(true);
        ui->internalButton->setEnabled(true);
        ui->timerLabel->setVisible(true);
        ui->removeButton->setEnabled(false);
        ui->startButton->setEnabled(false);
        ui->stopButton->setEnabled(true);
        ui->tableView->setVisible(false);
        break;
    case AlmostFinished:
        ui->actionChangeAlarm->setEnabled(false);
        ui->actionRemoveCompleted->setEnabled(false);
        ui->currentTaskName->setVisible(true);
        ui->doneButton->setEnabled(true);
        ui->externalButton->setEnabled(true);
        ui->internalButton->setEnabled(true);
        ui->timerLabel->setVisible(true);
        ui->removeButton->setEnabled(false);
        ui->startButton->setEnabled(false);
        ui->stopButton->setEnabled(true);
        ui->tableView->setVisible(false);
        break;
    }
}
