/*
  Copyright (C) 2010  Sakari Hyoty

  This file is part of Pomodoro.

  Pomodoro is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Pomodoro is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Pomodoro.  If not, see <http://www.gnu.org/licenses/>.
  */

#include <QSound>
#include <QInputDialog>
#include <QMessageBox>
#include <phonon/AudioOutput>

#ifdef Q_WS_MAEMO_5
#include <QtDBus/QDBusMessage>
#include <QtDBus/QDBusConnection>
#endif // Q_WS_MAEMO_5

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "pomodoromodel.h"
#include "pomodorotimer.h"
#include "pomodoroconfig.h"
#include "taskdialog.h"

QString secondsToString(int totalSeconds)
{
    int minutes = totalSeconds / 60;
    int seconds = totalSeconds % 60;

    QString str("%1:%2");
    str = str.arg(QVariant(minutes).toString(), 2, '0');
    str = str.arg(QVariant(seconds).toString(), 2, '0');

    return str;
}

void showSystemNotification(QString text)
{
    qDebug("showSystemNotification text=\"%s\"", text.toAscii().constData());
#ifdef Q_WS_MAEMO_5
    QDBusMessage msg =
        QDBusMessage::createMethodCall(
            "org.freedesktop.Notifications",
            "/org/freedesktop/Notifications",
            "org.freedesktop.Notifications",
            "SystemNoteInfoprint");

    QList<QVariant> args;
    args.append(text);

    msg.setArguments(args);

    QDBusConnection::systemBus().call(msg);
#endif // Q_WS_MAEMO_5
}

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow),
    pomodoroModel(NULL),
    alertMediaObject(NULL),
    pomodoroTimer(NULL)
{
    ui->setupUi(this);

    // initialize buttons
    connect(ui->removeButton, SIGNAL(clicked()),
            this, SLOT(removeTask()));
    connect(ui->doneButton, SIGNAL(clicked()),
            this, SLOT(completeTask()));
    connect(ui->internalButton, SIGNAL(clicked()),
            this, SLOT(internalInterruption()));
    connect(ui->externalButton, SIGNAL(clicked()),
            this, SLOT(externalInterruption()));

    // initialize timer
    pomodoroTimer = new PomodoroTimer(this);
    pomodoroTimer->setLength(POMODORO_TIMER_LENGTH);

    connect(ui->startButton, SIGNAL(clicked()),
            pomodoroTimer, SLOT(start()));
    connect(ui->stopButton, SIGNAL(clicked()),
            pomodoroTimer, SLOT(stop()));
    connect(pomodoroTimer, SIGNAL(started(int)),
            this, SLOT(pomodoroStarted(int)));
    connect(pomodoroTimer, SIGNAL(stopped(bool)),
            this, SLOT(pomodoroStopped(bool)));
    connect(pomodoroTimer, SIGNAL(secondsLeft(int)),
            this, SLOT(updateProgress(int)));

    // initialize model
    pomodoroModel = new PomodoroModel(ui->tableView);
    QApplication::setOverrideCursor(Qt::WaitCursor);
    bool modelLoadStatus = pomodoroModel->loadModel();
    QApplication::restoreOverrideCursor();

    // initialize tableView
    QTableView *tableView = ui->tableView;
    tableView->setModel(pomodoroModel);

    // initialize headerView
    QHeaderView *headerView = tableView->horizontalHeader();
    headerView->setClickable(false);
    headerView->setResizeMode(0, QHeaderView::Stretch);
    headerView->setResizeMode(1, QHeaderView::ResizeToContents);
    headerView->setResizeMode(2, QHeaderView::ResizeToContents);
    headerView->setResizeMode(3, QHeaderView::ResizeToContents);
    headerView->setResizeMode(4, QHeaderView::ResizeToContents);
    connect(tableView, SIGNAL(entered(QModelIndex)),
            this, SLOT(taskSelected(QModelIndex)));
    connect(tableView, SIGNAL(clicked(QModelIndex)),
            this, SLOT(taskSelected(QModelIndex)));
    connect(tableView, SIGNAL(pressed(QModelIndex)),
            this, SLOT(taskSelected(QModelIndex)));

    taskDialog = new TaskDialog(this);
    connect(ui->addButton, SIGNAL(clicked()), taskDialog, SLOT(show()));
    connect(taskDialog, SIGNAL(newTask(QString,int)),
            this, SLOT(addTask(QString,int)));

    connect(ui->actionGuide, SIGNAL(triggered()), this, SLOT(showGuide()));
    connect(ui->actionAbout, SIGNAL(triggered()), this, SLOT(showAbout()));

    alertMediaObject = new Phonon::MediaObject(this);
    Phonon::AudioOutput *audioOutput =
        new Phonon::AudioOutput(Phonon::NoCategory, this);
    Phonon::createPath(alertMediaObject, audioOutput);

    setUiState(IdleNoSelection);

    if (!modelLoadStatus) {
        showGuide();
    }
}

MainWindow::~MainWindow()
{
    delete alertMediaObject;
    alertMediaObject = NULL;

    delete taskDialog;
    taskDialog = NULL;
}

void MainWindow::addTask(QString taskName, int estimate)
{
    QModelIndex newTaskIndex = pomodoroModel->newTask(taskName, estimate);

    if (!pomodoroTimer->isStarted()) {
        selectedIndex = newTaskIndex;
        setUiState(IdleTaskSelected);
        ui->tableView->setCurrentIndex(selectedIndex);
        ui->tableView->scrollTo(
            selectedIndex,
            QAbstractItemView::PositionAtCenter);
    } else {
        showSystemNotification(tr("Task added"));
    }
}

void MainWindow::removeTask()
{
    qDebug("MainWindow::removeTask");

    pomodoroModel->deleteTask(selectedIndex);
    selectedIndex = QModelIndex();
    setUiState(IdleNoSelection);
}

void MainWindow::completeTask()
{
    qDebug("MainWindow::completeTask");

    if (pomodoroTimer->isStarted()) {
        pomodoroTimer->stop(false);
        pomodoroModel->pomodoroCompleted(selectedIndex);
    }

    pomodoroModel->taskCompleted(selectedIndex);
    ui->tableView->setCurrentIndex(selectedIndex);
    setUiState(IdleCompletedTaskSelected);
}

void MainWindow::internalInterruption()
{
    qDebug("MainWindow::internalInterruption");

    pomodoroModel->pomodoroInterrupted(selectedIndex, true);
    showSystemNotification(tr("Internal interruption recorded"));
}

void MainWindow::externalInterruption()
{
    qDebug("MainWindow::externalInterruption");

    pomodoroModel->pomodoroInterrupted(selectedIndex, false);
    showSystemNotification(tr("External interruption recorded"));
}


void MainWindow::taskSelected(const QModelIndex &index)
{
    qDebug("MainWindow::taskSelected");

    selectedIndex = index;

    if (!pomodoroModel->isTaskCompleted(index))
        setUiState(IdleTaskSelected);
    else
        setUiState(IdleCompletedTaskSelected);
}

void MainWindow::pomodoroStarted(int /*lengthInSeconds*/)
{
    qDebug("MainWindow::pomodoroStarted");

    // set task name to UI
    QModelIndex taskNameIndex =
        pomodoroModel->index(selectedIndex.row(), 0, QModelIndex());
    ui->currentTaskName->setText(
            taskNameIndex.data(Qt::DisplayRole).toString());

    // update progress UI
    ui->timerLabel->setText(secondsToString(POMODORO_TIMER_LENGTH));

    setUiState(Started);
}

void MainWindow::pomodoroStopped(bool finished)
{
    qDebug("MainWindow::pomodoroStopped %sfinished", finished?"":"!");

    if (finished) {
        pomodoroModel->pomodoroCompleted(selectedIndex);
        showSystemNotification(tr("Pomodoro finished"));
        alertMediaObject->setCurrentSource(POMODORO_FILE_ALERT);
        alertMediaObject->play();
    }

    setUiState(IdleTaskSelected);

    // make UI point to correct task
    ui->tableView->setCurrentIndex(selectedIndex);
}

void MainWindow::updateProgress(int secondsLeft)
{
    qDebug("MainWindow::updateProgress secondsLeft=%i", secondsLeft);

    if (secondsLeft == POMODORO_TIMER_ALMOST_FINISHED)
        setUiState(AlmostFinished);

    ui->timerLabel->setText(secondsToString(secondsLeft));
}

void MainWindow::showGuide()
{
    qDebug("MainWindow::showGuide");

    QString title = tr("What is a pomodoro?");
    QString text =
            tr("\"Pomodoro\" is a 25 minute session of focused work. Between pomodoros "
               "take a 5 minute break and every 4 pomodoros a longer break. "
               "Internal interruption means that you have been interrupted by "
               "yourself, you did something not in the scope of the task. "
               "External interruption means that someone else interrupted you."
               "\n\n"
               "For details on The Pomodoro Technique see <http://www.pomodorotechnique.com/>.");

    QMessageBox::information(this, title, text);

    title = tr("How to use this application?");
    text = tr("First add a task and estimate it in pomodoros. "
              "Then just start the task and work until "
              "time runs out. For every interruption during a pomodoro click "
              "on Int./Ext. button depending on the distraction type."
              "\n\n"
              "Mark the task completed when it's done. Completed tasks will "
              "be hidden from the task list the next day.");

    QMessageBox::information(this, title, text);
}

void MainWindow::showAbout()
{
    qDebug("MainWindow::showAbout");

    QString title = tr("About Pomodoro ") + POMODORO_VERSION;
    QString text =
            tr("Copyright (C) 2010, Sakari Hyoty\n\n"
               "See detailed licensing information from ") + POMODORO_README_FILE + " .";

    QMessageBox::about(this, title, text);
}

void MainWindow::changeEvent(QEvent *e)
{
    //qDebug("MainWindow::changeEvent");

    QMainWindow::changeEvent(e);

    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::setUiState(PomodoroUiState state)
{
    qDebug("MainWindow::setUiState state=%i", state);

    // developer: keep lines alphabetically sorted
    switch (state) {
    case IdleNoSelection:
        ui->currentTaskName->setText(QString(""));
        ui->currentTaskName->setVisible(false);
        ui->doneButton->setEnabled(false);
        ui->externalButton->setEnabled(false);
        ui->internalButton->setEnabled(false);
        ui->timerLabel->setVisible(false);
        ui->removeButton->setEnabled(false);
        ui->startButton->setEnabled(false);
        ui->stopButton->setEnabled(false);
        ui->tableView->setVisible(true);
        break;
    case IdleTaskSelected:
        ui->currentTaskName->setText(QString(""));
        ui->currentTaskName->setVisible(false);
        ui->doneButton->setEnabled(true);
        ui->externalButton->setEnabled(false);
        ui->internalButton->setEnabled(false);
        ui->timerLabel->setVisible(false);
        ui->removeButton->setEnabled(true);
        ui->startButton->setEnabled(true);
        ui->stopButton->setEnabled(false);
        ui->tableView->setVisible(true);
        break;
    case IdleCompletedTaskSelected:
        ui->currentTaskName->setText(QString(""));
        ui->currentTaskName->setVisible(false);
        ui->doneButton->setEnabled(false);
        ui->externalButton->setEnabled(false);
        ui->internalButton->setEnabled(false);
        ui->timerLabel->setVisible(false);
        ui->removeButton->setEnabled(true);
        ui->startButton->setEnabled(false);
        ui->stopButton->setEnabled(false);
        ui->tableView->setVisible(true);
        break;
    case Started:
        ui->currentTaskName->setVisible(true);
        ui->doneButton->setEnabled(false);
        ui->externalButton->setEnabled(true);
        ui->internalButton->setEnabled(true);
        ui->timerLabel->setVisible(true);
        ui->removeButton->setEnabled(false);
        ui->startButton->setEnabled(false);
        ui->stopButton->setEnabled(true);
        ui->tableView->setVisible(false);
        break;
    case AlmostFinished:
        ui->currentTaskName->setVisible(true);
        ui->doneButton->setEnabled(true);
        ui->externalButton->setEnabled(true);
        ui->internalButton->setEnabled(true);
        ui->timerLabel->setVisible(true);
        ui->removeButton->setEnabled(false);
        ui->startButton->setEnabled(false);
        ui->stopButton->setEnabled(true);
        ui->tableView->setVisible(false);
        break;
    }
}
