/*
  Copyright (C) 2010  Sakari Hyoty

  This file is part of Pomodoro.

  Pomodoro is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Pomodoro is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Pomodoro.  If not, see <http://www.gnu.org/licenses/>.
  */

#include <QFile>
#include <QColor>
#include <QFont>

#include "pomodoromodel.h"
#include "pomodoroconfig.h"

struct PomodoroTask
{
    PomodoroTask()
        : id(0),
        pomodorosEstimated(0),
        pomodorosCompleted(0),
        internalInterruptions(0),
        externalInterruptions(0),
        creationDate(QDate::currentDate()),
        completionDate(0, 0, 0) // default to invalid date
    {
    }

    bool displayInTaskList() const
    {
        return !isCompleted() || completionDate == QDate::currentDate();
    }

    bool isCompleted() const
    {
        return completionDate.isValid();
    }

    quint32 id;
    QString taskName;
    int pomodorosEstimated;
    int pomodorosCompleted;
    int internalInterruptions;
    int externalInterruptions;
    QDate creationDate;
    QDate completionDate;
};

/* data format 1:
   (int) task id
   (QString) task name
   (int) estimate
   (int) pomodoros completed
   (int) internal interruptions
   (int) external interruptions
   (QDate) creation date
   (QDate) completion date, invalid if not completed
   */

bool modelLoader_1(QDataStream &readStream, QVector<PomodoroTask> &vector)
{
    PomodoroTask task;

    while (!readStream.atEnd() && readStream.status() == QDataStream::Ok) {
        readStream
                >> task.id
                >> task.taskName
                >> task.pomodorosEstimated
                >> task.pomodorosCompleted
                >> task.internalInterruptions
                >> task.externalInterruptions
                >> task.creationDate
                >> task.completionDate;

        if (task.displayInTaskList()) {
            vector.append(task);
        }
    }

    return readStream.status() == QDataStream::Ok;
}

bool modelSaver_1(QDataStream &writeStream, const QVector<PomodoroTask> &vector)
{
    PomodoroTask task;

    for (int i = 0; i < vector.count(); i++) {
        task = vector.at(i);
        writeStream
                << task.id
                << task.taskName
                << task.pomodorosEstimated
                << task.pomodorosCompleted
                << task.internalInterruptions
                << task.externalInterruptions
                << task.creationDate
                << task.completionDate;
    }

    return true;
}

PomodoroModel::PomodoroModel(QObject *parent) :
    QAbstractTableModel(parent),
    taskIdRunner(0),
    dirty(false)
{
}

PomodoroModel::~PomodoroModel()
{
}

int PomodoroModel::rowCount(const QModelIndex &/*parent*/) const
{
    //qDebug("PomodoroModel::rowCount");

    return pomodoroVector.count();
}

int PomodoroModel::columnCount(const QModelIndex &/*parent*/) const
{
    //qDebug("PomodoroModel::columnCount");

    return 5;
}

QVariant PomodoroModel::data(const QModelIndex &index, int role) const
{
    //qDebug("PomodoroModel::data");

    QVariant variant;
    int row = index.row();
    int column = index.column();
    const PomodoroTask task = pomodoroVector.at(row);

    if (role == Qt::DisplayRole) {
        if (column == 0) {
            variant.setValue(task.taskName);
        } else if (column == 1){
            QVariant pomodoros(task.pomodorosCompleted);
            variant.setValue(pomodoros.toString());
        } else if (column == 2) {
            if (task.isCompleted()) {
                QString str;
                str.append(tr("Done"));
                variant.setValue(str);
            } else {
                QVariant estimate(task.pomodorosEstimated);
                variant.setValue(estimate.toString());
            }
        } else if (column == 3) {
            QVariant internal(task.internalInterruptions);
            variant.setValue(internal.toString());
        } else if (column == 4) {
            QVariant external(task.externalInterruptions);
            variant.setValue(external.toString());
        }
    } else if (role == Qt::TextAlignmentRole && column > 0) {
        variant.setValue(static_cast<int>(Qt::AlignCenter));
    } else if (role == Qt::BackgroundColorRole) {
        if (column == 2 && task.isCompleted())
            variant.setValue(QColor(0, 255, 0, 65));
        else if (column == 1 && task.pomodorosCompleted > task.pomodorosEstimated)
            variant.setValue(QColor(255, 0, 0, 65));
        else if (column == 3 && task.internalInterruptions)
            variant.setValue(QColor(255, 255, 0, 65));
        else if (column == 4 && task.externalInterruptions)
            variant.setValue(QColor(0, 255, 255, 65));
    }

    return variant;
}

QVariant PomodoroModel::headerData(int section, Qt::Orientation /*orientation*/, int role) const
{
    //qDebug("PomodoroModel::headerData");

    QVariant variant;

    if (role == Qt::DisplayRole) {
        if (section == 0)
            variant.setValue(tr("Task"));
        else if (section == 1)
            variant.setValue(tr("Comp."));
        else if (section == 2)
            variant.setValue(tr("Est."));
        else if (section == 3)
            variant.setValue(tr("Int."));
        else if (section == 4)
            variant.setValue(tr("Ext."));
        else
            qFatal("PomodoroModel::headerData invalid section value (%i)", section);
    } else if (role == Qt::TextAlignmentRole && section == 0) {
        variant.setValue(static_cast<int>(Qt::AlignLeft));
    }

    return variant;
}

QModelIndex PomodoroModel::index(int row, int column, const QModelIndex &/*parent*/) const
{
    //qDebug("PomodoroModel::index row=%i column=%i", row, column);

    return createIndex(row, column, pomodoroVector.at(row).id);
}

bool PomodoroModel::loadModel()
{
    qDebug("PomodoroModel::loadModel");

    QFile file(POMODORO_FILE_MODEL);
    bool retval = false;

    if (file.exists() && file.open(QIODevice::ReadOnly)) {
        QDataStream readStream(&file);
        readStream.setVersion(QDataStream::Qt_4_5);

        int version = 0;

        readStream >> version;
        readStream >> taskIdRunner;

        if (version == 1)
            retval = modelLoader_1(readStream, pomodoroVector);

        if (!retval)
            qFatal(tr("Data file %s is not compatible "
                      "with this version of Pomodoro").toAscii().constData(),
                   POMODORO_FILE_MODEL.toAscii().constData());
    } else {
        qDebug("PomodoroModel::loadModel unable to open data file");
    }

    return retval;
}

bool PomodoroModel::saveModel()
{
    QFile file(POMODORO_FILE_MODEL);
    bool exists = file.exists();
    bool retval = false;

    if (dirty) {
        qDebug("PomodoroModel::saveModel model is dirty");

        if (file.open(QIODevice::ReadWrite)) {
            QDataStream readWriteStream(&file);
            readWriteStream.setVersion(QDataStream::Qt_4_5);

            int version = 0;

            readWriteStream >> version;

            if (!exists || version == 1) {
                file.seek(0);
                file.resize(0);
                readWriteStream.setDevice(&file);
                readWriteStream << 1;
                readWriteStream << taskIdRunner;
                retval = modelSaver_1(readWriteStream, pomodoroVector);
                dirty = false;
            } else {
                qWarning("Data file found is not compatible with this version of Pomodoro.");
            }
        }
    } else {
        qDebug("PomodoroModel::saveModel model not dirty");
        retval = true;
    }

    return retval;
}

QModelIndex PomodoroModel::newTask(QString taskName, int estimate)
{
    qDebug("PomodoroModel::newTask");

    PomodoroTask task;
    task.id = taskIdRunner++;
    task.taskName = taskName;
    task.pomodorosEstimated = estimate;

    pomodoroVector.append(task);
    dirty = true;
    saveModel();
    int lastRow = pomodoroVector.count() - 1;
    insertRow(lastRow, QModelIndex());
    reset(); // notify view

    return createIndex(lastRow, 0, pomodoroVector.at(lastRow).id);
}

void PomodoroModel::deleteTask(const QModelIndex &index)
{
    qDebug("PomodoroModel::deleteTask");

    pomodoroVector.remove(index.row(), 1);
    dirty = true;
    saveModel();
    reset(); // notify view
}

void PomodoroModel::taskCompleted(const QModelIndex &index)
{
    qDebug("PomodoroModel::taskCompleted");

    task(index)->completionDate = QDate::currentDate();
    dirty = true;
    saveModel();
    reset(); // notify view
}

bool PomodoroModel::isTaskCompleted(const QModelIndex &index)
{
    qDebug("PomodoroModel::isTaskCompleted");

    return task(index)->isCompleted();
}

void PomodoroModel::pomodoroCompleted(const QModelIndex &index)
{
    qDebug("PomodoroModel::pomodoroCompleted");

    task(index)->pomodorosCompleted++;
    dirty = true;
    saveModel();
    reset(); // notify view
}

void PomodoroModel::pomodoroInterrupted(const QModelIndex &index, bool internalInterruption)
{
    qDebug("PomodoroModel::pomodoroInterrupted");

    if (internalInterruption)
        task(index)->internalInterruptions++;
    else
        task(index)->externalInterruptions++;

    dirty = true;
    saveModel();
    reset(); // notify view
}

PomodoroTask *PomodoroModel::task(const QModelIndex &index)
{
    qDebug("PomodoroModel::task");

    PomodoroTask *task = NULL;

    for (int i = 0; i < pomodoroVector.count(); i++) {
        if (pomodoroVector.at(i).id == index.internalId()) {
            task = &(pomodoroVector[i]);
            break;
        }
    }

    return task;
}
