/*
  PokeMini - Pokmon-Mini Emulator
  Copyright (C) 2009  JustBurn

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "PokeMini.h"

TMinxColorPRC MinxColorPRC;
uint8_t *PRCColorPixels = NULL;
uint8_t *PRCColorPixelsOld = NULL;
uint8_t *PRCColorMap = NULL;
unsigned int PRCColorOffset = 0;
uint8_t *PRCColorTop = 0x00000000;

// Color Flags
// Bit 0 - New Color Palette
// Bit 1 - Render to RAM
// Bit 2 to 7 - Reserved
uint8_t PRCColorFlags;

//
// Functions
//

int MinxColorPRC_Create(void)
{
	// Create color pixels array
	PRCColorPixels = (uint8_t *)malloc(96*64);
	if (!PRCColorPixels) return 0;
	memset(PRCColorPixels, 0, 96*64);
	PRCColorPixelsOld = (uint8_t *)malloc(96*64);
	if (!PRCColorPixelsOld) return 0;
	memset(PRCColorPixelsOld, 0, 96*64);

	// Reset
	MinxColorPRC_Reset(1);

	return 1;
}

void MinxColorPRC_Destroy(void)
{
	if (PRCColorPixels) {
		free(PRCColorPixels);
		PRCColorPixels = NULL;
	}
	if (PRCColorPixelsOld) {
		free(PRCColorPixelsOld);
		PRCColorPixelsOld = NULL;
	}
}

void MinxColorPRC_Reset(int hardreset)
{
	// Initialize State
	memset((void *)&MinxColorPRC, 0, sizeof(TMinxColorPRC));
}

int MinxColorPRC_LoadState(FILE *fi, uint32_t bsize)
{
	POKELOADSS_START(16);
	POKELOADSS_A(MinxColorPRC.Reserved, 16);
	POKELOADSS_END(16);

}

int MinxColorPRC_SaveState(FILE *fi)
{
	POKESAVESS_START(16);
	POKESAVESS_A(MinxColorPRC.Reserved, 16);
	POKESAVESS_END(16);
}

//
// Unofficial Color Support
//

const uint8_t PRCStaticColorMap[8] = {0x00, 0xF0, 0x00, 0xF0, 0x00, 0xF0, 0x00, 0xF0};

static inline void MinxPRC_DrawSprite8x8_Color8(uint8_t cfg, int X, int Y, int DrawT, int MaskT)
{
	uint8_t *ColorMap;
	int yC, xC, xP, level, out;
	uint8_t sdata, smask;

	// No point to proceed if it's offscreen
	if (X >= 96) return;
	if (Y >= 64) return;

	// Pre calculate
	level = (((MinxLCD.Contrast + 2) & 0x3C) << 2) - 0x80;
	ColorMap = (uint8_t *)PRCColorMap + (MinxPRC.PRCSprBase >> 2) + (DrawT << 1) - PRCColorOffset;
	if ((ColorMap < PRCColorMap) || (ColorMap >= PRCColorTop)) ColorMap = (uint8_t *)PRCStaticColorMap;

	// Draw sprite
	for (yC=0; yC<8; yC++) {
		if ((Y >= 0) && (Y < 64)) {
			for (xC=0; xC<8; xC++) {
				if ((X >= 0) && (X < 96)) {
					xP = (cfg & 0x01) ? (7 - xC) : xC;

					smask = MinxPRC_OnRead(0, MinxPRC.PRCSprBase + (MaskT * 8) + xP);
					if (cfg & 0x02) smask = PRCInvertBit[smask];
					smask = smask & (1 << (yC & 7));

					// Write result
					if (!smask) {
						sdata = MinxPRC_OnRead(0, MinxPRC.PRCSprBase + (DrawT * 8) + xP);
						if (cfg & 0x02) sdata = PRCInvertBit[sdata];
						if (cfg & 0x04) sdata = ~sdata;
						sdata = sdata & (1 << (yC & 7));

						out = level + (int)(sdata ? ColorMap[1] : *ColorMap);
						if (out > 255) out = 255;
						if (out < 0) out = 0;
						PRCColorPixels[Y * 96 + X] = (uint8_t)out;
					}
				}
				X++;
			}
			X -= 8;
		}
		Y++;
	}
}

void MinxPRC_Render_Color8(void)
{
	int xC, yC, tx, ty, tileidxaddr, ltileidxaddr, outaddr, level, out;
	int tiledataddr = 0;
	uint8_t *ColorMap = (uint8_t *)PRCStaticColorMap;
	uint8_t tileidx = 0, tdata, data;

	int SprTB, SprAddr;
	int SprX, SprY, SprC;

	if (!PRCColorMap) return;
	if (VidEnableHighcolor) memcpy(PRCColorPixelsOld, PRCColorPixels, 96*64);
	if (PRCColorFlags & 2) MinxPRC_Render_Mono();

	// Color contrast level
	level = (((MinxLCD.Contrast + 2) & 0x3C) << 2) - 0x80;

	if (PRCRenderBD) {
		for (xC=0; xC<96*64; xC++) PRCColorPixels[xC] = 0x00;
	}

	if ((PRCRenderBG) && (PMR_PRC_MODE & 0x02)) {
		outaddr = 0;
		ltileidxaddr = -1;
		for (yC=0; yC<64; yC++) {
			ty = yC + MinxPRC.PRCMapPY;
			for (xC=0; xC<96; xC++) {
				tx = xC + MinxPRC.PRCMapPX;
				tileidxaddr = 0x1360 + (ty >> 3) * MinxPRC.PRCMapTW + (tx >> 3);

				// Read tile index
				if (ltileidxaddr != tileidxaddr) {
					ltileidxaddr = tileidxaddr;
					tileidx = MinxPRC_OnRead(0, tileidxaddr);
					tiledataddr = MinxPRC.PRCBGBase + (tileidx << 3);
					ColorMap = (uint8_t *)PRCColorMap + (MinxPRC.PRCBGBase >> 2) + (tileidx << 1) - PRCColorOffset;
					if ((ColorMap < PRCColorMap) || (ColorMap >= PRCColorTop)) ColorMap = (uint8_t *)PRCStaticColorMap;
				}

				// Read tile data
				tdata = MinxPRC_OnRead(0, tiledataddr + (tx & 7));
				if (PMR_PRC_MODE & 0x01) tdata = ~tdata;
				data = (tdata & (1 << (ty & 7))) ? ColorMap[1] : *ColorMap;

				// Write result
				out = level + (int)data;
				if (out > 255) out = 255;
				if (out < 0) out = 0;
				PRCColorPixels[outaddr++] = (uint8_t)out;
			}
		}
	}

	if ((PRCRenderSpr) && (PMR_PRC_MODE & 0x04)) {
		SprAddr = 0x1300 + (24 * 4);
		do {
			SprC = MinxPRC_OnRead(0, --SprAddr);
			SprTB = MinxPRC_OnRead(0, --SprAddr) << 3;
			SprY = (MinxPRC_OnRead(0, --SprAddr) & 0x7F) - 16;
			SprX = (MinxPRC_OnRead(0, --SprAddr) & 0x7F) - 16;
			if (SprC & 0x08) {
				MinxPRC_DrawSprite8x8_Color8(SprC, SprX + (SprC & 0x01 ? 8 : 0), SprY + (SprC & 0x02 ? 8 : 0), SprTB+2, SprTB);
				MinxPRC_DrawSprite8x8_Color8(SprC, SprX + (SprC & 0x01 ? 8 : 0), SprY + (SprC & 0x02 ? 0 : 8), SprTB+3, SprTB+1);
				MinxPRC_DrawSprite8x8_Color8(SprC, SprX + (SprC & 0x01 ? 0 : 8), SprY + (SprC & 0x02 ? 8 : 0), SprTB+6, SprTB+4);
				MinxPRC_DrawSprite8x8_Color8(SprC, SprX + (SprC & 0x01 ? 0 : 8), SprY + (SprC & 0x02 ? 0 : 8), SprTB+7, SprTB+5);
			}
		} while (SprAddr > 0x1300);
	}
}

static inline void MinxPRC_DrawSprite8x8_Color4(uint8_t cfg, int X, int Y, int DrawT, int MaskT)
{
	uint8_t *ColorMap;
	int yC, xC, xP, level, quad, out;
	uint8_t sdata, smask;

	// No point to proceed if it's offscreen
	if (X >= 96) return;
	if (Y >= 64) return;

	// Pre calculate
	level = (((MinxLCD.Contrast + 2) & 0x3C) << 2) - 0x80;
	ColorMap = (uint8_t *)PRCColorMap + MinxPRC.PRCSprBase + (DrawT << 3) - PRCColorOffset;
	if ((ColorMap < PRCColorMap) || (ColorMap >= PRCColorTop)) ColorMap = (uint8_t *)PRCStaticColorMap;

	// Draw sprite
	for (yC=0; yC<8; yC++) {
		if ((Y >= 0) && (Y < 64)) {
			for (xC=0; xC<8; xC++) {
				if ((X >= 0) && (X < 96)) {
					quad = (yC & 4) + ((xC & 4) >> 1);
					xP = (cfg & 0x01) ? (7 - xC) : xC;

					smask = MinxPRC_OnRead(0, MinxPRC.PRCSprBase + (MaskT * 8) + xP);
					if (cfg & 0x02)	smask = PRCInvertBit[smask];
					smask = smask & (1 << (yC & 7));

					// Write result
					if (!smask) {
						sdata = MinxPRC_OnRead(0, MinxPRC.PRCSprBase + (DrawT * 8) + xP);
						if (cfg & 0x02)	sdata = PRCInvertBit[sdata];
						if (cfg & 0x04) sdata = ~sdata;
						sdata = sdata & (1 << (yC & 7));

						out = level + (int)(sdata ? ColorMap[quad+1] : ColorMap[quad]);
						if (out > 255) out = 255;
						if (out < 0) out = 0;
						PRCColorPixels[Y * 96 + X] = (uint8_t)out;
					}
				}
				X++;
			}
			X -= 8;
		}
		Y++;
	}
}

void MinxPRC_Render_Color4(void)
{
	int xC, yC, tx, ty, tileidxaddr, ltileidxaddr, outaddr, level, quad, out;
	int tiledataddr = 0;
	uint8_t *ColorMap = (uint8_t *)PRCStaticColorMap;
	uint8_t tileidx = 0, tdata, data;

	int SprTB, SprAddr;
	int SprX, SprY, SprC;

	if (!PRCColorMap) return;
	if (VidEnableHighcolor) memcpy(PRCColorPixelsOld, PRCColorPixels, 96*64);
	if (PRCColorFlags & 2) MinxPRC_Render_Mono();

	// Color contrast level
	level = (((MinxLCD.Contrast + 2) & 0x3C) << 2) - 0x80;

	if (PRCRenderBD) {
		for (xC=0; xC<96*64; xC++) PRCColorPixels[xC] = 0x00;
	}

	if ((PRCRenderBG) && (PMR_PRC_MODE & 0x02)) {
		outaddr = 0;
		ltileidxaddr = -1;
		for (yC=0; yC<64; yC++) {
			ty = yC + MinxPRC.PRCMapPY;
			for (xC=0; xC<96; xC++) {
				tx = xC + MinxPRC.PRCMapPX;
				quad = (ty & 4) + ((tx & 4) >> 1);
				tileidxaddr = 0x1360 + (ty >> 3) * MinxPRC.PRCMapTW + (tx >> 3);

				// Read tile index
				if (ltileidxaddr != tileidxaddr) {
					ltileidxaddr = tileidxaddr;
					tileidx = MinxPRC_OnRead(0, tileidxaddr);
					tiledataddr = MinxPRC.PRCBGBase + (tileidx << 3);
					ColorMap = (uint8_t *)PRCColorMap + MinxPRC.PRCBGBase + (tileidx << 3) - PRCColorOffset;
					if ((ColorMap < PRCColorMap) || (ColorMap >= PRCColorTop)) ColorMap = (uint8_t *)PRCStaticColorMap;
				}

				// Read tile data
				tdata = MinxPRC_OnRead(0, tiledataddr + (tx & 7));
				if (PMR_PRC_MODE & 0x01) tdata = ~tdata;
				data = (tdata & (1 << (ty & 7))) ? ColorMap[quad+1] : ColorMap[quad];

				// Write result
				out = level + (int)data;
				if (out > 255) out = 255;
				if (out < 0) out = 0;
				PRCColorPixels[outaddr++] = (uint8_t)out;
			}
		}
	}
	if (PRCRenderSpr && (PMR_PRC_MODE & 0x04)) {
		SprAddr = 0x1300 + (24 * 4);
		do {
			SprC = MinxPRC_OnRead(0, --SprAddr);
			SprTB = MinxPRC_OnRead(0, --SprAddr) << 3;
			SprY = (MinxPRC_OnRead(0, --SprAddr) & 0x7F) - 16;
			SprX = (MinxPRC_OnRead(0, --SprAddr) & 0x7F) - 16;
			if (SprC & 0x08) {
				MinxPRC_DrawSprite8x8_Color4(SprC, SprX + (SprC & 0x01 ? 8 : 0), SprY + (SprC & 0x02 ? 8 : 0), SprTB+2, SprTB);
				MinxPRC_DrawSprite8x8_Color4(SprC, SprX + (SprC & 0x01 ? 8 : 0), SprY + (SprC & 0x02 ? 0 : 8), SprTB+3, SprTB+1);
				MinxPRC_DrawSprite8x8_Color4(SprC, SprX + (SprC & 0x01 ? 0 : 8), SprY + (SprC & 0x02 ? 8 : 0), SprTB+6, SprTB+4);
				MinxPRC_DrawSprite8x8_Color4(SprC, SprX + (SprC & 0x01 ? 0 : 8), SprY + (SprC & 0x02 ? 0 : 8), SprTB+7, SprTB+5);
			}
		} while (SprAddr > 0x1300);
	}
}
