// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QDebug>
#include <QMessageBox>
#include <QFileDialog>
#include <QToolBar>
#include <QMenuBar>
#include <QCloseEvent>
#include <QSettings>
#include <QCalendarWidget>
#include "plansplant/widgets.hpp"
#include "plansplant/tasks_changes.hpp"
#include "plansplant/export.hpp"
#include "plansplant/utility.hpp"
#ifdef Q_WS_MAEMO_5
#include <QStackedWidget>
#include <QX11Info>
#include <X11/Xcursor/Xcursor.h>
#include <QAbstractKineticScroller>
#include <QScrollBar>
#include <QStyleFactory>
#endif // Q_WS_MAEMO_5
#ifdef __GLIBC__
#include <langinfo.h>
#else
#include <QLocale>
#endif // __GLIBC__
#include "ui_search_dialog.h"
#include "ui_move_times_dialog.h"

namespace PlansPlant
{
  //! \todo Move to some "System" class. 
  Qt::DayOfWeek first_day_of_week()
  {
#ifdef __GLIBC__
    return Qt::DayOfWeek( ( int( *nl_langinfo( _NL_TIME_FIRST_WEEKDAY ) ) + 5 ) % 7 + 1 ); // It's very strange function and returns either 0 or 7 for Sunday.
#else
    //! \todo Add other locales
    switch( QLocale().language() )
    {
    case QLocale::Russian:
      return Qt::Monday;
    default:
      return Qt::Sunday;
    }
#endif // __GLIBC__
  } // first_day_of_week()
#ifdef Q_WS_MAEMO_5
  void disable_kinetic_scroller( QAbstractScrollArea* Widget )
  {
    if( Widget )
      if( QAbstractKineticScroller *Scroller = Widget->property( "kineticScroller" ).value<QAbstractKineticScroller*>() )
      {
	Scroller->setEnabled( false );
	if( !MainWindow::desktop_view() ) //! \todo Find a way to determine that we're using Hildon style.
	  if( QStyle* GoodStyle = QStyleFactory::create( "GTK+" ) )
	  {
	    Widget->horizontalScrollBar()->setStyle( GoodStyle );
	    Widget->horizontalScrollBar()->setContextMenuPolicy( Qt::PreventContextMenu );
	    Widget->verticalScrollBar()->setStyle( GoodStyle );
	    Widget->verticalScrollBar()->setContextMenuPolicy( Qt::PreventContextMenu );
	  }
      }
  } // disable_kinetic_scroller( QAbstractScrollArea* )
#endif // Q_WS_MAEMO_5

  TimeSpanSpinBox::TimeSpanSpinBox( QWidget* Parent ) : QSpinBox( Parent ) {}
  QString TimeSpanSpinBox::textFromValue( int Value ) const
  {
    QString Result;
    int Days = Value;
    if( Days < 0 )
    {
      Result += '-';
      Days = -Days;
    }
    int Secs = Days % 60;
    Days /= 60;
    int Mins = Days % 60;
    Days /= 60;
    int Hours = Days % 24;
    Days /= 24;
    Result += QString::number( Days ) + ' ';
    if( Hours < 10 ) Result += '0';
    Result += QString::number( Hours ) + ':';
    if( Mins < 10 ) Result += '0';
    Result += QString::number( Mins ) + ':';
    if( Secs < 10 ) Result += '0';
    Result += QString::number( Secs );
    return Result;
  } // textFromValue( int ) const
  int TimeSpanSpinBox::valueFromText( const QString& Text ) const
  { //! \todo Write parser methods or better class.
    int Result = 0; qDebug() << "Parse" << Text;
    QString::const_iterator It = Text.begin();
    while( It != Text.end() && It->isSpace() ) ++It;
    if( It != Text.end() )
    {
      int Sign = 1;
      if( *It == '-' )
      {
	Sign = -1;
	++It;
      }
      for( ; It != Text.end() && It->isDigit(); ++It ) Result = Result * 10 + It->digitValue();
      if( It != Text.end() && It->isSpace() )
      {
	while( It != Text.end() && It->isSpace() ) ++It;
	int Val = 0;
	for( ; It != Text.end() && It->isDigit(); ++It ) Val = Val * 10 + It->digitValue();
	if( Val < 24 )
	{
	  Result = Result * 24 + Val;
	  if( It != Text.end() && *It == ':' )
	  {
	    Val = 0;
	    for( ++It; It != Text.end() && It->isDigit(); ++It ) Val = Val * 10 + It->digitValue();
	    if( Val < 60 )
	    {
	      Result = Result * 60 + Val;
	      if( It != Text.end() && *It == ':' )
	      {
		Val = 0;
		for( ++It; It != Text.end() && It->isDigit(); ++It ) Val = Val * 10 + It->digitValue();
		if( Val < 60 ) Result = ( Result * 60 + Val ) * Sign;
		// if( It != end() ) trailing garbage
		// else more then 59 seconds
	      } // else no seconds or wrong separator
	    } // else more then 59 minutes
	  } // else no minutes or wrong separator
	} // else More then 23 hours 
      } // else no days or no space after days
    } // else no valid numbers.
    qDebug() << "Result is:" << Result;
    return Result;
  } // valueFromText( const QString ) const
  QValidator::State TimeSpanSpinBox::validate( QString& Text, int& Pos ) const
  {
    qDebug() << "Validate" << Text << "pos" << Pos;
    QValidator::State Result = QValidator::Intermediate;
    int Len = Text.length();
    int I = 0;
    while( I < Len && Text[ I ].isSpace() ) ++I; // Skip leading spaces
    if( I < Len && Text[ I ] == '-' ) ++I;
    bool CanBeAcceptable = I < Len && Text[ I ].isDigit();
    while( I < Len && Text[ I ].isDigit() ) ++I;
    if( I < Len && Text[ I ] == ' ' ) ++I;
    else CanBeAcceptable = false;
    if( !( I < Len && Text[ I ].isDigit() ) ) CanBeAcceptable = false;
    int Val = 0;
    for( ; I < Len && Text[ I ].isDigit(); ++I ) Val = Val*10 + Text[ I ].digitValue();
    if( I < Len && Text[ I ] == ':' ) ++I;
    else CanBeAcceptable = false;
    if( !( Val < 24 && I < Len && Text[ I ].isDigit() ) ) CanBeAcceptable = false;
    for( Val = 0; I < Len && Text[ I ].isDigit(); ++I ) Val = Val*10 + Text[ I ].digitValue();
    if( I < Len && Text[ I ] == ':' ) ++I;
    else CanBeAcceptable = false;
    if( !( Val < 60 && I < Len && Text[ I ].isDigit() ) ) CanBeAcceptable = false;
    for( Val = 0; I < Len && Text[ I ].isDigit(); ++I ) Val = Val*10 + Text[ I ].digitValue();
    if( I < Len ) Result = QValidator::Invalid;
    else if( CanBeAcceptable && Val < 60 ) Result = QValidator::Acceptable;
    return Result;
  } // validate( QString& Text, int& Pos ) const

  class TasksIterator
  {
  public:
    TasksIterator( TasksFile& File0 ) : File( File0 ), Current( 0 ) { if( !File.roots().empty() ) Current = File.roots().front(); }
    Task* task() const { return Current; }
    Task* operator*() const { return Current; }
    Task* operator->() const { return Current; }
    TasksIterator& operator++();
  protected:
    TasksFile& File;
    Task* Current;
  }; // TasksIterator
  TasksIterator& TasksIterator::operator++()
  {
    if( Current )
    {
      if( !Current->subtasks().empty() ) Current = Current->subtasks().front();
      else
      {
	Task* Next = Current;
	Current = 0;
	for( Task* Parent = Next->supertask(); Next && !Current; Parent = Next ? Next->supertask() : 0 )
	{
	  const Task::List& Tasks = Parent ? Parent->subtasks() : File.roots();
	  int ItemIndex = Tasks.indexOf( Next );
	  if( ItemIndex >= 0 && ItemIndex < Tasks.size()-1 )
	    Current = Tasks[ ItemIndex+1 ];
	  else
	    Next = Parent;
	}
      }
    }
    return *this;
  } // ++TasksIterator

  class MoveTimesDialog : public QDialog
  {
  public:
    MoveTimesDialog( QWidget* Parent = 0 ) : QDialog( Parent )
    {
      UI.setupUi( this );
      UI.FromTime->calendarWidget()->setFirstDayOfWeek( first_day_of_week() );
      UI.ToTime->calendarWidget()->setFirstDayOfWeek( first_day_of_week() );
      QSettings Set;
      Set.beginGroup( "Status/Recent/MoveTimes" );
      UI.FromTime->setDateTime( Set.value( "FromTime" ).toDateTime() );
      UI.ToTime->setDateTime( Set.value( "ToTime" ).toDateTime() );
      UI.MoveTime->setValue( Set.value( "MoveTime" ).toInt() );
    } // MoveTimesDialog( QWidget* )
    QDateTime from() const { return UI.FromTime->dateTime(); }
    QDateTime to() const { return UI.ToTime->dateTime(); }
    int move() const { return UI.MoveTime->value(); }
  protected:
    void accept()
    {
      QSettings Set;
      Set.beginGroup( "Status/Recent/MoveTimes" );
      Set.setValue( "FromTime", from() );
      Set.setValue( "ToTime", to() );
      Set.setValue( "MoveTime", move() );
      QDialog::accept();
    } // accept()
    Ui::MoveTimesDialog UI;
  }; // MoveTimesDialog
  

  TaskSelectDialog::TaskSelectDialog( TasksModel& Model0, QWidget* Parent, Task* CurrentTask0, const QString& Title )
    : QDialog( Parent ), CurrentTask( CurrentTask0 ), TasksTree( 0 ), Buttons( 0 )
  {
    setWindowTitle( Title );
    TasksTree = new TasksTreeWidget( this, &Model0 );
    TasksTree->setMinimumHeight( 300 );
    QVBoxLayout* Layout = new QVBoxLayout( this );
    Layout->addWidget( TasksTree );
#ifdef PLANSPLANT_HANDHELD
    if( !MainWindow::desktop_view() ) Layout->setContentsMargins( 0, 0, 0, 0 );
    foreach( QAction* Act, TasksTree->actions() )
      Act->setShortcuts( QList<QKeySequence>() );
#endif // PLANSPLANT_HANDHELD
    Buttons = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( CurrentTask );
    connect( Buttons, SIGNAL( accepted() ), SLOT( accept() ) );
    connect( Buttons, SIGNAL( rejected() ), SLOT( reject() ) );
    Layout->addWidget( Buttons );
    setLayout( Layout );
    connect( TasksTree->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ), SLOT( selection_changed( const QItemSelection& ) ) );
    TasksTree->select_task( CurrentTask );
  } // TaskSelectDialog( TasksTreeModel&, QWidget*, Task* )
  void TaskSelectDialog::accept()
  {
    CurrentTask = TasksTree->selected_task();
    if( CurrentTask ) QDialog::accept();
    else QMessageBox::warning( this, tr( "PlansPlant" ), tr( "Select a task before." ) );
  } // accept()
  void TaskSelectDialog::selection_changed( const QItemSelection& New )
  {
    QModelIndex NewIndex;
    if( !New.indexes().isEmpty() )
      NewIndex = New.indexes().front(); // We've got indexes for every column. Take only the first one.
    selection_changed( NewIndex );
  } // selection_changed( const QItemSelection& )
  void TaskSelectDialog::selection_changed( const QModelIndex& NewIndex )
  {
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( TasksTree->tasks() && TasksTree->tasks()->task_from_index( NewIndex ) );
  } // selection_changed( const QModelIndex& )

  ParentSelectDialog::ParentSelectDialog( TasksModel& Model0, QWidget* Parent, const Task::List& Blockers0, Task* ForTask0, Task* SuperTask0 )
    : TaskSelectDialog( Model0, Parent, SuperTask0, tr( "Select new supertask for " ) + ( ForTask0 ? ( '\"' + ForTask0->name() + '\"' ) : tr( "new task" ) ) ),
      ForTask( ForTask0 ), Blockers( Blockers0 )
  {
    selection_changed( TasksTree->selected_index() );
  } // ParentSelectDialog( TasksTreeModel&, Task&, QWidget*, Task* )
  void ParentSelectDialog::selection_changed( const QModelIndex& NewIndex )
  {
    bool Enable = false;
    if( TasksTree && TasksTree->tasks() )
      if( Task* NewParent = TasksTree->tasks()->task_from_index( NewIndex ) )
	Enable = !NewParent->check_loop( Blockers, ForTask );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( Enable );
  } // selection_changed( const QModelIndex& )
  BlockerSelectDialog::BlockerSelectDialog( TasksModel& Model0, QWidget* Parent, Task* ForTask0, Task* Blocker0 )
    : TaskSelectDialog( Model0, Parent, Blocker0, tr( "Select new dependency for " ) + ( ForTask0 ? ( '\"' + ForTask0->name() + '\"' ) : tr( "new task" ) ) + '\"' ),
      ForTask( ForTask0 )
  {
    selection_changed( TasksTree->selected_index() );
  } // BlockerSelectDialog( TasksTreeModel&, Task&, QWidget*, Task* )
  void BlockerSelectDialog::selection_changed( const QModelIndex& NewIndex )
  {
    bool Enable = false;
    if( TasksTree )
      if( TasksModel* Tasks = TasksTree->tasks() )
	if( Task* NewBlocker = Tasks->task_from_index( NewIndex ) )
	  Enable = !ForTask || ( !ForTask->blockers().contains( NewBlocker ) && !ForTask->check_loop( *NewBlocker ) );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( Enable );
  } // selection_changed( const QModelIndex& )

  class BlockersListModel : public QAbstractItemModel
  {
  public:
    static Task* task_from_index( const QModelIndex& Index ) { return static_cast<Task*>( Index.internalPointer() ); }
    BlockersListModel( QObject* Parent = 0, const Task::List& Blockers0 = Task::List() ) : QAbstractItemModel( Parent ), Blockers( Blockers0 ) {}
    int columnCount( const QModelIndex& Parent = QModelIndex() ) const;
    int rowCount( const QModelIndex& Parent = QModelIndex() ) const;
    QModelIndex index( int Row, int Column, const QModelIndex& Parent = QModelIndex() ) const;
    QModelIndex parent( const QModelIndex& Index ) const;
    QVariant data( const QModelIndex& Index, int Role = Qt::DisplayRole ) const;
    QVariant headerData( int Section, Qt::Orientation Orient, int Role = Qt::DisplayRole ) const;
    const Task::List& blockers() const { return Blockers; }
    void add_blocker( Task& NewBlocker );
    void remove_blocker( Task& OldBlocker );
    void move_up( const QModelIndex& Index );
    void move_down( const QModelIndex& Index );
  protected:
    Task::List Blockers;
  }; // BlockersListModel
  int BlockersListModel::columnCount( const QModelIndex& /*Parent*/ ) const { return TasksTreeModel::TotalCols; }
  int BlockersListModel::rowCount( const QModelIndex& Parent ) const { return Parent.isValid() ? 0 : Blockers.size(); }
  QModelIndex BlockersListModel::index( int Row, int Column, const QModelIndex& Parent ) const
  {
    QModelIndex Result;
    if( !Parent.isValid() && Row >= 0 && Row < Blockers.size() ) 
      Result = createIndex( Row, Column, Blockers[ Row ] );
    return Result;
  } // index( int Row, int Column, const QModelIndex& Parent ) const
  QModelIndex BlockersListModel::parent( const QModelIndex& /*Index*/ ) const { return QModelIndex(); }
  QVariant BlockersListModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    const int Row = Index.row(); 
    if( Index.isValid() && Row >= 0 && Row < Blockers.size() ) 
    {
      Task* Blocker = Blockers[ Row ];
      if( Role == Qt::DisplayRole )
	switch( Index.column() )
	{
	case TasksTreeModel::NameCol: Result = Blocker->name(); break;
	case TasksTreeModel::CompletedCol: Result = QString::number( Blocker->completed() * 100 ) + '%'; break;
	case TasksTreeModel::PlanStartCol:
	  if( Blocker->plan_start().isValid() )
	    Result = Blocker->plan_start();
	  break;
	case TasksTreeModel::PlanFinishCol:
	  if( Blocker->plan_finish().isValid() )
	    Result = Blocker->plan_finish();
	  break;
	case TasksTreeModel::EstimationCol:
	  if( Blocker->estimation() > 0 )
	    Result = QString::number( Blocker->estimation() ) + ' ' + Task::units_short_name( Blocker->estimation_units() );
	  break;
	default: break;
	}
      else if( Role == Qt::DecorationRole )
      {
	if( Index.column() == TasksTreeModel::NameCol )
	  Result = plansplant_icon( "dependency" );
      }
      else if( Role == Qt::CheckStateRole )
      {
	if( Index.column() == TasksTreeModel::CompletedMarkCol )
	{
	  if( Blocker->completed() >= 1 )
	    Result = Qt::Checked;
	  else
	  { if( !Blocker->blocked() ) Result = Qt::Unchecked; }
	}
      }
#ifdef PLANSPLANT_HANDHELD
      else if( Role == Qt::SizeHintRole )
      { if( !MainWindow::desktop_view() ) Result = QSize( 300, 42 ); }
#endif
    }
    return Result;
  } // data( const QModelIndex& Index, int Role = Qt::DisplayRole ) const
  QVariant BlockersListModel::headerData( int Section, Qt::Orientation Orient, int Role ) const
  {
    QVariant Result;
    if( Role == Qt::DisplayRole && Orient == Qt::Horizontal )
      switch( Section )
      {
      case TasksTreeModel::NameCol: Result = TasksTreeModel::tr( "Name" ); break;
      case TasksTreeModel::CompletedMarkCol: break;
      case TasksTreeModel::CompletedCol: Result = TasksTreeModel::tr( "Completed" ); break;
      case TasksTreeModel::PlanStartCol: Result = TasksTreeModel::tr( "Start" ); break;
      case TasksTreeModel::PlanFinishCol: Result = TasksTreeModel::tr( "Finish" ); break;
      case TasksTreeModel::EstimationCol: Result = TasksTreeModel::tr( "Estimation" ); break;
      default: Result = QString::number( Section ); break;
      }
    return Result;
  } // headerData( int, Qt::Orientation, int ) const
  void BlockersListModel::add_blocker( Task& NewBlocker )
  {
    if( !Blockers.contains( &NewBlocker ) )
    {
      beginInsertRows( QModelIndex(), Blockers.size(), Blockers.size() );
      Blockers.push_back( &NewBlocker );
      endInsertRows();
    }
  } // add_blocker( Task& )
  void BlockersListModel::remove_blocker( Task& OldBlocker )
  {
    int Index = Blockers.indexOf( &OldBlocker );
    if( Index >= 0 )
    {
      beginRemoveRows( QModelIndex(), Index, Index );
      Blockers.removeAt( Index );
      endRemoveRows();
    }
  } // remove_blocker( Task& )
  void BlockersListModel::move_up( const QModelIndex& Index )
  {
    if( Index.isValid() )
    {
      int NewRow = Index.row()-1;
      if( NewRow >= 0 )
	if( beginMoveRows( parent( Index ), Index.row(), Index.row(), parent( Index ), NewRow ) )
	{
	  Blockers.swap( Index.row(), NewRow );
	  endMoveRows();
	}
    }
  } // move_up( const QModelIndex& )
  void BlockersListModel::move_down( const QModelIndex& Index )
  {
    if( Index.isValid() )
    {
      int NewRow = Index.row()+1;
      if( NewRow < Blockers.size() )
	if( beginMoveRows( parent( Index ), Index.row(), Index.row(), parent( Index ), NewRow+1 ) )
	{
	  Blockers.swap( Index.row(), NewRow );
	  endMoveRows();
	}
    }
  } // move_down( const QModelIndex& )

  BlockersEditor::BlockersEditor( TasksModel& Model0, Task* Object0, QWidget* Parent )
    : QWidget( Parent ), Model( Model0 ), Blockers( 0 ), Object( Object0 ), SuperTask( 0 )
  {
    UI.setupUi( this );
    UI.BlockerAddButton->setIcon( load_icon( "add" ) );
    UI.BlockerRemoveButton->setIcon( load_icon( "remove" ) );
    UI.BlockerMoveUpButton->setIcon( load_icon( "go-up" ) );
    UI.BlockerMoveDownButton->setIcon( load_icon( "go-down" ) );
    UI.BlockersTreeSwitchButton->setIcon( plansplant_icon( "show-tree" ) );
    UI.BlockersPoolTree->setVisible( UI.BlockersTreeSwitchButton->isChecked() );
    foreach( QAction* Act, UI.BlockersPoolTree->actions() )
#ifndef PLANSPLANT_HANDHELD
      Act->setShortcutContext( Qt::WidgetShortcut );
#else // PLANSPLANT_HANDHELD
      Act->setShortcuts( QList<QKeySequence>() );
    if( MainWindow::desktop_view() ) disable_kinetic_scroller( UI.BlockersList );
    else UI.BlockersSplitter->setMinimumHeight( 220 );
#endif // PLANSPLANT_HANDHELD
    if( Object )
    {
      SuperTask = Object->supertask();
      Blockers = new BlockersListModel( this, Object->blockers() );
    }
    else
      Blockers = new BlockersListModel( this );
    UI.BlockersList->setModel( Blockers );
    UI.BlockersList->setColumnWidth( TasksTreeModel::NameCol, 400 );
    if( MainWindow::desktop_view() )
    {
      UI.BlockersList->setColumnWidth( TasksTreeModel::CompletedMarkCol, 24 );
      UI.BlockersList->setColumnWidth( TasksTreeModel::CompletedCol, 43 );
    }
    else
    {
      UI.BlockersList->setColumnWidth( TasksTreeModel::CompletedMarkCol, 55 );
      UI.BlockersList->setColumnWidth( TasksTreeModel::CompletedCol, 70 );
      UI.BlockersList->setColumnWidth( TasksTreeModel::PlanStartCol, 128 );
      UI.BlockersList->setColumnWidth( TasksTreeModel::PlanFinishCol, 128 );
    }
    UI.BlockersPoolTree->tasks( &Model );
    if( Object ) UI.BlockersPoolTree->select_task( Object->blockers().empty() ? Object : Object->blockers().back() );
    else if( SuperTask ) UI.BlockersPoolTree->select_task( SuperTask );
    connect( UI.BlockerAddButton, SIGNAL( clicked() ), SLOT( add_blocker() ) );
    connect( UI.BlockerRemoveButton, SIGNAL( clicked() ), SLOT( remove_blocker() ) );
    connect( UI.BlockerMoveUpButton, SIGNAL( clicked() ), SLOT( move_blocker_up() ) );
    connect( UI.BlockerMoveDownButton, SIGNAL( clicked() ), SLOT( move_blocker_down() ) );
    connect( UI.BlockersPoolTree->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
	     SLOT( blockers_pool_selection_changed( const QItemSelection& ) ) );
    connect( UI.BlockersList->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
	     SLOT( blockers_list_selection_changed( const QItemSelection& ) ) );
    connect( UI.BlockersTreeSwitchButton, SIGNAL( toggled( bool ) ), SLOT( blockers_pool_switched( bool ) ) );
  } // BlockersEditor( TasksTreeModel&, Task*, QWidget* )
  const Task::List& BlockersEditor::blockers() const { return Blockers->blockers(); }
  QModelIndex BlockersEditor::selected_blocker() const
  {
    QModelIndex Result;
    if( UI.BlockersList )
      if( QItemSelectionModel* SelectionModel = UI.BlockersList->selectionModel() )
      {
	QModelIndexList Indexes = SelectionModel->selectedIndexes();
	if( !Indexes.isEmpty() )
	  Result = Indexes.front();
      }
    return Result;
  } // selected_blocker() const
  void BlockersEditor::supertask( Task* NewSuperTask )
  {
    if( SuperTask != NewSuperTask )
    {
      SuperTask = NewSuperTask;
      blockers_pool_switched( UI.BlockersTreeSwitchButton->isChecked() );
    }
  } // supertask( Task* )
  void BlockersEditor::add_blocker()
  {
    if( Blockers )
      if( Task* Blocker = UI.BlockersPoolTree->selected_task() )
	if( can_add_blocker( *Blocker ) )
	{
	  Blockers->add_blocker( *Blocker );
	  blockers_list_selection_changed( selected_blocker() );
	  blockers_pool_switched( UI.BlockersTreeSwitchButton->isChecked() ); // To update buttons
	  emit blockers_changed();
	}
  } // add_blocker()
  void BlockersEditor::remove_blocker()
  {
    if( Blockers )
      if( Task* Blocker = BlockersListModel::task_from_index( selected_blocker() ) )
      {
	qDebug() << "Remove blocker" << Blocker->name() << "from list.";
	Blockers->remove_blocker( *Blocker );
	blockers_list_selection_changed( selected_blocker() );
	blockers_pool_switched( UI.BlockersTreeSwitchButton->isChecked() ); // To update buttons
	emit blockers_changed();
      }
  } // remove_blocker()
  void BlockersEditor::move_blocker_up()
  {
    if( Blockers )
    {
      Blockers->move_up( selected_blocker() );
      blockers_list_selection_changed( selected_blocker() );
    }
  } // move_blocker_up()
  void BlockersEditor::move_blocker_down()
  {
    if( Blockers )
    {
      Blockers->move_down( selected_blocker() );
      blockers_list_selection_changed( selected_blocker() );
    }
  } // move_blocker_down()
  bool BlockersEditor::can_add_blocker( Task& NewBlocker )
  {
    bool Result = false;
    if( &NewBlocker != Object && !( Blockers && Blockers->blockers().contains( &NewBlocker ) )
	&& !( SuperTask && SuperTask->check_loop( NewBlocker ) ) )
    {
      Result = true;
      if( Object )
	for( Task::List::const_iterator It = Object->dependents().begin(); Result && It != Object->dependents().end(); It++ )
	  Result = !(*It)->check_loop( NewBlocker );
    }
    return Result;
  } // can_add_blocker( Task& )
  void BlockersEditor::blockers_pool_switched( bool On )
  {
    if( On )
      blockers_pool_selection_changed( UI.BlockersPoolTree->selected_index() );
    else
      UI.BlockerAddButton->setEnabled( false );
  } // blockers_pool_switched( bool )
  void BlockersEditor::blockers_pool_selection_changed( const QItemSelection& New )
  {
    QModelIndex NewIndex;
    if( !New.indexes().isEmpty() )
      NewIndex = New.indexes().front();
    blockers_pool_selection_changed( NewIndex );
  } // blockers_pool_selection_changed( const QItemSelection& )
  void BlockersEditor::blockers_pool_selection_changed( const QModelIndex& NewIndex )
  {
    bool CanAdd = false;
    if( Task* NewBlocker = Model.task_from_index( NewIndex ) )
      CanAdd = can_add_blocker( *NewBlocker );
    UI.BlockerAddButton->setEnabled( CanAdd );
  } // blockers_pool_selection_changed( const QModelIndex& )
  void BlockersEditor::blockers_list_selection_changed( const QItemSelection& New )
  {
    QModelIndex NewIndex;
    if( !New.indexes().isEmpty() )
      NewIndex = New.indexes().front();
    blockers_list_selection_changed( NewIndex );
  } // blockers_list_selection_changed( const QItemSelection& )
  void BlockersEditor::blockers_list_selection_changed( const QModelIndex& NewIndex )
  {
    if( NewIndex != selected_blocker() )
      qDebug() << "Wrong list index:" << NewIndex << selected_blocker();
    if( NewIndex.isValid() && NewIndex.internalPointer() )
    {
      UI.BlockerRemoveButton->setEnabled( true );
      UI.BlockerMoveUpButton->setEnabled( NewIndex.row() > 0 );
      UI.BlockerMoveDownButton->setEnabled( Blockers && NewIndex.row() < Blockers->blockers().size()-1 );
    }
    else
    {
      UI.BlockerRemoveButton->setEnabled( false );
      UI.BlockerMoveUpButton->setEnabled( false );
      UI.BlockerMoveDownButton->setEnabled( false );
    }
  } // blockers_list_selection_changed( const QModelIndex& )

  BlockersEditorDialog::EditorWidget::EditorWidget( TasksModel& Model0, Task* Object0, QDialog* Parent )
    : BlockersEditor( Model0, Object0, Parent )
  {
    QDialogButtonBox* Buttons = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this );
    UI.ButtonsLayout->addWidget( Buttons );
#ifdef PLANSPLANT_HANDHELD
    if( !MainWindow::desktop_view() )
    {
      UI.ButtonsLayout->setSpacing( 0 );
      layout()->setSpacing( 0 );
    }
#endif // PLANSPLANT_HANDHELD
    Parent->connect( Buttons, SIGNAL( accepted() ), SLOT( accept() ) );
    Parent->connect( Buttons, SIGNAL( rejected() ), SLOT( reject() ) );
    UI.BlockersLabel->setText( Object ? BlockersEditorDialog::tr( "Task" ) + ": " + Object->name() : BlockersEditorDialog::tr( "Unknown task" ) );
    UI.BlockersTreeSwitchButton->setChecked( true );
  } // EditorWidget( TasksTreeModel&, Task*, QWidget* )
  bool BlockersEditorDialog::EditorWidget::accept()
  {
    bool Result = true;
    if( Blockers && Object )
      Model.file().change_blockers( *Object, Blockers->blockers(), SuperTask );
    else
      Result = false;
    return Result;
  } // accept()
  BlockersEditorDialog::BlockersEditorDialog( TasksModel& Model0, Task* Object0, QWidget* Parent )
  : QDialog( Parent ), Editor( Model0, Object0, this )
  {
    setWindowTitle( tr( "Dependencies" ) );
    QVBoxLayout* Layout = new QVBoxLayout( this );
    Layout->addWidget( &Editor );
    setLayout( Layout );
#ifdef Q_WS_MAEMO_5
    if( !MainWindow::desktop_view() )
    {
      Qt::WindowFlags Flags = windowFlags() & ~Qt::Dialog | Qt::Window;
      setWindowFlags( Flags );
      setAttribute( Qt::WA_Maemo5StackedWindow );
      Layout->setContentsMargins( 0, 0, 0, 0 );
    }
#endif // Q_WS_MAEMO_5
  } // BlockersEditorDialog( TasksTreeModel&, Task*, QWidget* )
  QSize BlockersEditorDialog::sizeHint() const { return QSize( 800, 480 ); }
  void BlockersEditorDialog::accept()
  {
    if( Editor.accept() )
      QDialog::accept();
  } // accept()

  TaskDialog::TimeActions::TimeActions() : Button( 0 ), Default( Now ) { for( int I = 0; I < ActionsCount; I++ ) Actions[ I ] = 0; }
  void TaskDialog::TimeActions::create( QToolButton* But )
  {
    Button = But;
    QMenu* Menu = new QMenu( Button );
    Actions[ Now ] = new QAction( tr( "Now" ), Menu );
    Menu->addAction( Actions[ Now ] );
    Actions[ StartOfDay ] = new QAction( tr( "Start of day" ), Menu );
    Menu->addAction( Actions[ StartOfDay ] );
    Actions[ EndOfDay ] = new QAction( tr( "End of day" ), Menu );
    Menu->addAction( Actions[ EndOfDay ] );
    Actions[ Estimated ] = new QAction( tr( "Estimated" ), Menu );
    Menu->addAction( Actions[ Estimated ] );
    Actions[ FromSubItems ] = new QAction( tr( "From subitems" ), Menu );
    Menu->addAction( Actions[ FromSubItems ] );
    Actions[ FromSubtasks ] = new QAction( tr( "From subtasks" ), Menu );
    Menu->addAction( Actions[ FromSubtasks ] );
    Actions[ BlockersStart ] = new QAction( tr( "Blockers start" ), Menu );
    Menu->addAction( Actions[ BlockersStart ] );
    Actions[ BlockersFinish ] = new QAction( tr( "Blockers finish" ), Menu );
    Menu->addAction( Actions[ BlockersFinish ] );
    Actions[ FromSupertask ] = new QAction( tr( "From supertask" ), Menu );
    Menu->addAction( Actions[ FromSupertask ] );
    Button->setMenu( Menu );
    default_action( Default );
  } // create( QMenu* )
  void TaskDialog::TimeActions::default_action( Action NewDefault )
  {
    Default = NewDefault;
    Button->setDefaultAction( Actions[ Default ] );
  } // default_action( Action )
  
  TaskDialog::TaskDialog( TasksModel& Model0, Task* Object0, QWidget* Parent ) : QDialog( Parent ), Model( Model0 ), Object( Object0 ), SuperTask( 0 ), Blockers( 0 )
  {
    UI.setupUi( this );
#ifdef Q_WS_MAEMO_5
    if( !MainWindow::desktop_view() )
    {
      Qt::WindowFlags Flags = windowFlags() & ~Qt::Dialog | Qt::Window;
      setWindowFlags( Flags );
      setAttribute( Qt::WA_Maemo5StackedWindow );
    }
#endif // Q_WS_MAEMO_5
    for( int I = 2; I > -3; --I )
      UI.Priority->addItem( Task::priority_name( I ), I );
    UI.Priority->setCurrentIndex( UI.Priority->findData( 0 ) );
    UI.EstimatedUnits->addItem( tr( "seconds" ), Task::Seconds );
    UI.EstimatedUnits->addItem( tr( "minutes" ), Task::Minutes );
    UI.EstimatedUnits->addItem( tr( "hours" ), Task::Hours );
    UI.EstimatedUnits->addItem( tr( "workdays" ), Task::WorkDays );
    UI.EstimatedUnits->addItem( tr( "days" ), Task::Days );
    UI.EstimatedUnits->addItem( tr( "workweeks" ), Task::WorkWeeks );
    UI.EstimatedUnits->addItem( tr( "weeks" ), Task::Weeks );
    UI.EstimatedUnits->addItem( tr( "months" ), Task::Months );
    UI.EstimatedUnits->addItem( tr( "quarters" ), Task::Quarters );
    UI.EstimatedUnits->addItem( tr( "years" ), Task::Years );
    int RecentUnits = QSettings().value( "Status/RecentUnits", Task::Hours ).toInt();
    UI.EstimatedUnits->setCurrentIndex( UI.EstimatedUnits->findData( RecentUnits ) );
    UI.PlanStart->calendarWidget()->setFirstDayOfWeek( first_day_of_week() );
    UI.PlanFinish->calendarWidget()->setFirstDayOfWeek( first_day_of_week() );
    PlanStartActions.create( UI.PlanStartButton );
    UI.PlanStartButton->setEnabled( false );
    PlanFinishActions.create( UI.PlanFinishButton );
    UI.PlanFinishButton->setEnabled( false );
    PlanStartActions.action( TimeActions::Estimated )->setEnabled( UI.HasFinish->isChecked() );
    PlanStartActions.action( TimeActions::Estimated )->connect( UI.HasFinish, SIGNAL( toggled( bool ) ), SLOT( setEnabled( bool ) ) );
    PlanFinishActions.action( TimeActions::Estimated )->setEnabled( UI.HasStart->isChecked() );
    PlanFinishActions.action( TimeActions::Estimated )->connect( UI.HasStart, SIGNAL( toggled( bool ) ), SLOT( setEnabled( bool ) ) );
#ifdef PLANSPLANT_HANDHELD
    if( !MainWindow::desktop_view() )
    {
      UI.Description->setMinimumHeight( 128 );
      UI.Comment->setMinimumHeight( 128 );
      QMargins Margins = UI.PlanGroup->layout()->contentsMargins();
      Margins.setTop( 20 );
      UI.PlanGroup->layout()->setContentsMargins( Margins );
      layout()->setContentsMargins( 0, 0, 0, 0 );
    }
#endif // PLANSPLANT_HANDHELD
    if( Object )
    {
      setWindowTitle( tr( "Task: " ) + Object->name() );
      UI.Name->setText( Object->name() );
      UI.Description->setPlainText( Object->description() );
      UI.Priority->setCurrentIndex( UI.Priority->findData( Object->priority() ) );
      UI.Comment->setPlainText( Object->comment() );
      bool HasPlan = Object->estimation() != 0;
      if( Object->plan_start().isValid() )
      {
	HasPlan = true;
	UI.HasStart->setChecked( true );
	UI.PlanStart->setDateTime( Object->plan_start() );
      }
      if( Object->plan_finish().isValid() )
      {
	HasPlan = true;
	UI.HasFinish->setChecked( true );
	UI.PlanFinish->setDateTime( Object->plan_finish() );
      }
      UI.PlanGroup->setChecked( HasPlan );
      UI.Complete->setValue( Object->completed() * 100 );
      UI.Estimated->setValue( Object->estimation() );
      if( HasPlan ) UI.EstimatedUnits->setCurrentIndex( UI.EstimatedUnits->findData( Object->estimation_units() ) );
      if( Object->supertask() ) supertask( Object->supertask() );
      else
      {
	PlanStartActions.action( TimeActions::FromSupertask )->setEnabled( false );
	PlanFinishActions.action( TimeActions::FromSupertask )->setEnabled( false );
      }
    }
    else // No Object
      UI.PlanGroup->setChecked( false );
#ifndef PLANSPLANT_NO_BLOCKERS_SPLITTER
    if( MainWindow::desktop_view() )
    {
#if 0 // def Q_WS_MAEMO_5 // If we disable kinetic scroller all controls inside the scrollarea will not get input.
      disable_kinetic_scroller( UI.scrollArea );
#endif // Q_WS_MAEMO_5
      Blockers = new BlockersEditor( Model, Object, UI.scrollArea );
      connect( Blockers, SIGNAL( blockers_changed() ), SLOT( blockers_changed() ) );
      if( QBoxLayout* Layout = qobject_cast<QBoxLayout*>( UI.scrollAreaWidgetContents->layout() ) )
	Layout->insertWidget( 3, Blockers );
    }
#endif // PLANSPLANT_NO_BLOCKERS_SPLITTER
    PlanStartActions.action( TimeActions::FromSubtasks )->setEnabled( subitems_time( false, true, true ).isValid() );
    PlanFinishActions.action( TimeActions::FromSubtasks )->setEnabled( subitems_time( false, true, false ).isValid() );
    UI.Name->setFocus();
    connect( UI.HasParent, SIGNAL( toggled( bool ) ), SLOT( supertask_toggled( bool ) ) );
    connect( UI.SelectParent, SIGNAL( clicked( bool ) ), SLOT( select_supertask() ) );
    connect( PlanStartActions.now(), SIGNAL( triggered( bool ) ), SLOT( plan_start_now() ) );
    connect( PlanStartActions.start_of_day(), SIGNAL( triggered( bool ) ), SLOT( plan_start_start_of_day() ) );
    connect( PlanStartActions.end_of_day(), SIGNAL( triggered( bool ) ), SLOT( plan_start_end_of_day() ) );

    connect( PlanStartActions.action( TimeActions::Estimated ), SIGNAL( triggered( bool ) ), SLOT( plan_start_estimated() ) );
    connect( PlanStartActions.action( TimeActions::FromSubItems ), SIGNAL( triggered( bool ) ), SLOT( plan_start_from_subitems() ) );
    connect( PlanStartActions.action( TimeActions::FromSubtasks ), SIGNAL( triggered( bool ) ), SLOT( plan_start_from_subtasks() ) );
    connect( PlanStartActions.action( TimeActions::BlockersStart ), SIGNAL( triggered( bool ) ), SLOT( plan_start_blockers_start() ) );
    connect( PlanStartActions.action( TimeActions::BlockersFinish ), SIGNAL( triggered( bool ) ), SLOT( plan_start_blockers_finish() ) );
    connect( PlanStartActions.action( TimeActions::FromSupertask ), SIGNAL( triggered( bool ) ), SLOT( plan_start_from_supertask() ) );

    connect( PlanFinishActions.now(), SIGNAL( triggered( bool ) ), SLOT( plan_finish_now() ) );
    connect( PlanFinishActions.start_of_day(), SIGNAL( triggered( bool ) ), SLOT( plan_finish_start_of_day() ) );
    connect( PlanFinishActions.end_of_day(), SIGNAL( triggered( bool ) ), SLOT( plan_finish_end_of_day() ) );

    connect( PlanFinishActions.action( TimeActions::Estimated ), SIGNAL( triggered( bool ) ), SLOT( plan_finish_estimated() ) );
    connect( PlanFinishActions.action( TimeActions::FromSubItems ), SIGNAL( triggered( bool ) ), SLOT( plan_finish_from_subitems() ) );
    connect( PlanFinishActions.action( TimeActions::FromSubtasks ), SIGNAL( triggered( bool ) ), SLOT( plan_finish_from_subtasks() ) );
    connect( PlanFinishActions.action( TimeActions::BlockersStart ), SIGNAL( triggered( bool ) ), SLOT( plan_finish_blockers_start() ) );
    PlanFinishActions.action( TimeActions::BlockersStart )->setVisible( false );
    connect( PlanFinishActions.action( TimeActions::BlockersFinish ), SIGNAL( triggered( bool ) ), SLOT( plan_finish_blockers_finish() ) );
    connect( PlanFinishActions.action( TimeActions::FromSupertask ), SIGNAL( triggered( bool ) ), SLOT( plan_finish_from_supertask() ) );
    blockers_changed();
  } // TaskDialog
  Task::TimeUnits TaskDialog::time_units()
  {
    int CurrUnits = UI.EstimatedUnits->currentIndex();
    if( CurrUnits >= 0 )
    {
      CurrUnits = UI.EstimatedUnits->itemData( CurrUnits ).toInt();
      if( UI.Estimated->value() > 0 ) QSettings().setValue( "Status/RecentUnits", CurrUnits );
    }
    else
    {
      qDebug() << "Wrong estimation time units:" << CurrUnits;
      CurrUnits = 1;
    }
    return Task::TimeUnits( CurrUnits );
  } // time_units()
  void TaskDialog::accept()
  {
    bool TimesOk = true;
    Time Est = 0;
    QDateTime PlanStart;
    QDateTime PlanFinish;
    Task::TimeUnits CurrUnits = time_units();
    if( UI.PlanGroup->isChecked() )
    {
      Est = UI.Estimated->value();
      if( UI.HasStart->isChecked() )
	PlanStart = UI.PlanStart->dateTime();
      if( UI.HasFinish->isChecked() )
      {
	PlanFinish = UI.PlanFinish->dateTime();
	if( UI.HasStart->isChecked() )
	{
	  if( Est > 0 )
	    TimesOk = PlanFinish >= Task::add_time( PlanStart, Est, CurrUnits );
	  else
	    TimesOk = PlanFinish >= PlanStart;
	  if( !TimesOk ) //! \todo Add auto-adjust options.
	    TimesOk = QMessageBox::question( this, tr( "Plans Plant" ), tr( "Finish time is too small.\nContinue?" ),
					     QMessageBox::Yes | QMessageBox::No ) == QMessageBox::Yes;
	}
      }
    } // Has plan
    int Index = UI.Priority->currentIndex();
    if( TimesOk )
    {
      if( !Object )
      {
	Object = new Task( UI.Name->text(), UI.HasParent->isChecked() ? SuperTask : 0 );
	Object->description( UI.Description->toPlainText() );
	if( Index >= 0 ) Object->priority( UI.Priority->itemData( Index ).toInt() );
	Object->comment( UI.Comment->toPlainText() );
	Object->plan_start( PlanStart );
	Object->plan_finish( PlanFinish );
	Object->completed( UI.Complete->value() / 100.0 );
	Object->estimation( Est );
	Object->estimation_units( CurrUnits );
	if( Blockers )
	  foreach( Task* Blocker, Blockers->blockers() )
	    Object->add_blocker( *Blocker );
	Model.file().add_task( *Object );
      }
      else
      {
	Task::ChangesList Changes( *Object );
	if( Object->name() != UI.Name->text() )			{ Changes.add( *new Task::Changes::Name( UI.Name->text() ) ); }
	if( Object->description() != UI.Description->toPlainText() )	{ Changes.add( *new Task::Changes::Description( UI.Description->toPlainText() ) ); }
	if( Object->comment() != UI.Comment->toPlainText() )	   	{ Changes.add( *new Task::Changes::Comment( UI.Comment->toPlainText() ) ); }
	if( Object->plan_start() != PlanStart )	   		{ Changes.add( *new Task::Changes::PlanStart( PlanStart ) ); }
	if( Object->plan_finish() != PlanFinish )	   		{ Changes.add( *new Task::Changes::PlanFinish( PlanFinish ) ); }
	double Comp = UI.Complete->value() / 100.0;
	if( Index >= 0 && Object->priority() != UI.Priority->itemData( Index ).toInt() ) { Changes.add( *new Task::Changes::Priority( UI.Priority->itemData( Index ).toInt() ) ); }
	if( Object->completed() != Comp )				{ Changes.add( *new Task::Changes::Completed( Comp ) ); }
	if( Object->estimation() != Est )				{ Changes.add( *new Task::Changes::Estimation( Est ) ); }
	if( Object->estimation_units() != CurrUnits )			{ Changes.add( *new Task::Changes::EstimationUnits( Task::TimeUnits( CurrUnits ) ) ); }
	if( !UI.HasParent->isChecked() ) SuperTask = 0;
	if( Blockers && Blockers->blockers() != Object->blockers() )	{ Changes.add( *new Task::Changes::Blockers( Blockers->blockers() ) ); }
	if( Object->supertask() != SuperTask )
	{
	  Changes.add( *new Task::Changes::SuperTask( SuperTask ) );
	  if( SuperTask && SuperTask->blockers().contains( Object )
	      && QMessageBox::question( this, tr( "Change task" ),
					tr( "Remove \"" ) + Object->name() + tr( "\"\nfrom blockers of \"" ) + SuperTask->name() + "\"?", QMessageBox::Yes | QMessageBox::No,
					QMessageBox::Yes )  == QMessageBox::Yes )
	    Model.file().remove_blocker( *SuperTask, *Object );
	}
	if( !Changes.changes().empty() ) Model.file().change_task( Changes );
      }
      QDialog::accept();
    }
  } // accept()
  void TaskDialog::supertask( Task* NewSuperTask )
  {
    if( NewSuperTask != SuperTask )
    {
      SuperTask = NewSuperTask;
      if( SuperTask )
      {
	UI.SelectParent->setText( SuperTask->name() );
	UI.HasParent->setChecked( true );
      }
      else
      {
	UI.SelectParent->setText( tr( "none" ) );
	UI.HasParent->setChecked( false );
      }
      supertask_changed();
    }
  } // supertask( Task* )
  void TaskDialog::select_supertask()
  {
    ParentSelectDialog Dlg( Model, this, Blockers ? Blockers->blockers() : ( Object ? Object->blockers() : Task::List() ),
			    Object, SuperTask );
    if( Dlg.exec() )
      supertask( Dlg.current_task() );
  } // select_supertask()
  void TaskDialog::supertask_toggled( bool On )
  {
    if( On )
    {
      if( SuperTask )
      {
	if( SuperTask->check_loop( Blockers ? Blockers->blockers() : ( Object ? Object->blockers() : Task::List() ), Object ) )
	{
	  QMessageBox::warning( this, tr( "Plans Plant" ),
				tr( "This will create a loop in the tree.\nSelect another task as parent." ) );
	  UI.SelectParent->setText( tr( "none" ) );
	  SuperTask = 0;
	}
	else
	  supertask_changed();
      }
      if( !SuperTask )
      {
	select_supertask();
	if( !SuperTask )
	  UI.HasParent->setChecked( false );
      }
    }
    else
      supertask_changed();
  } // supertask_toggled( bool )
  void TaskDialog::supertask_changed()
  {
    if( Blockers )
      Blockers->supertask( UI.HasParent->isChecked() ? SuperTask : 0 );
    PlanStartActions.action( TimeActions::FromSupertask )->setEnabled( SuperTask && SuperTask->plan_start().isValid() );
    PlanFinishActions.action( TimeActions::FromSupertask )->setEnabled( SuperTask && SuperTask->plan_finish().isValid() );
  } // supertask_changed()
  void TaskDialog::blockers_changed()
  {
    PlanStartActions.action( TimeActions::FromSubItems )->setEnabled( subitems_time( true, true, true ).isValid() );
    PlanStartActions.action( TimeActions::BlockersStart )->setEnabled( subitems_time( true, false, true ).isValid() );
    PlanStartActions.action( TimeActions::BlockersFinish )->setEnabled( subitems_time( true, false, false ).isValid() );
    PlanFinishActions.action( TimeActions::FromSubItems )->setEnabled( subitems_time( true, true, false ).isValid() );
    // PlanFinishActions.action( TimeActions::BlockersStart )->setEnabled( subitems_time( true, false, true ).isValid() );
    PlanFinishActions.action( TimeActions::BlockersFinish )->setEnabled( subitems_time( true, false, false ).isValid() );
  } // blockers_changed()
  void TaskDialog::plan_start_now()
  {
    UI.PlanStart->setDateTime( QDateTime::currentDateTime() );
    PlanStartActions.default_action( TimeActions::Now );
  } // plan_start_now()
  void TaskDialog::plan_start_start_of_day()
  {
    UI.PlanStart->setTime( QTime( 0, 0 ) );
    PlanStartActions.default_action( TimeActions::StartOfDay );
  } // plan_start_start_of_day()
  void TaskDialog::plan_start_end_of_day()
  {
    UI.PlanStart->setTime( QTime( 23, 59 ) );
    PlanStartActions.default_action( TimeActions::EndOfDay );
  } // plan_start_end_of_day()
  void TaskDialog::plan_start_estimated()
  {
    if( UI.HasStart->isChecked() )
      UI.PlanStart->setDateTime( Task::add_time( UI.PlanFinish->dateTime(), -UI.Estimated->value(), time_units() ) );
    PlanStartActions.default_action( TimeActions::Estimated );
  } // plan_start_from_estimated()
  void TaskDialog::plan_start_from_subitems()
  {
    QDateTime Date = subitems_time( true, true, true );
    if( Date.isValid() )
      UI.PlanStart->setDateTime( Date );
    PlanStartActions.default_action( TimeActions::FromSubItems );
  } // plan_start_from_subitems()
  void TaskDialog::plan_start_from_subtasks()
  {
    QDateTime Date = subitems_time( false, true, true );
    if( Date.isValid() )
      UI.PlanStart->setDateTime( Date );
    PlanStartActions.default_action( TimeActions::FromSubtasks );
  } // plan_start_from_subtasks()
  void TaskDialog::plan_start_blockers_start()
  {
    QDateTime Date = subitems_time( true, false, true );
    if( Date.isValid() )
      UI.PlanStart->setDateTime( Date );
    PlanStartActions.default_action( TimeActions::BlockersStart );
  } // plan_start_blockers_start()
  void TaskDialog::plan_start_blockers_finish()
  {
    QDateTime Date = subitems_time( true, false, false );
    if( Date.isValid() )
      UI.PlanStart->setDateTime( Date );
    PlanStartActions.default_action( TimeActions::BlockersFinish );
  } // plan_start_blockers_finish()
  void TaskDialog::plan_start_from_supertask()
  {
    if( SuperTask && SuperTask->plan_start().isValid() )
      UI.PlanStart->setDateTime( SuperTask->plan_start() );
    PlanStartActions.default_action( TimeActions::FromSupertask );
  } // plan_start_from_supertask()
  void TaskDialog::plan_finish_now()
  {
    UI.PlanFinish->setDateTime( QDateTime::currentDateTime() );
    PlanFinishActions.default_action( TimeActions::Now );
  } // plan_finish_now()
  void TaskDialog::plan_finish_start_of_day()
  {
    UI.PlanFinish->setTime( QTime( 0, 0 ) );
    PlanFinishActions.default_action( TimeActions::StartOfDay );
  } // plan_finish_start_of_day()
  void TaskDialog::plan_finish_end_of_day()
  {
    UI.PlanFinish->setTime( QTime( 23, 59 ) );
    PlanFinishActions.default_action( TimeActions::EndOfDay );
  } // plan_finish_end_of_day()
  void TaskDialog::plan_finish_estimated()
  {
    if( UI.HasFinish->isChecked() )
      UI.PlanFinish->setDateTime( Task::add_time( UI.PlanStart->dateTime(), UI.Estimated->value(), time_units() ) );
    PlanFinishActions.default_action( TimeActions::Estimated );
  } // plan_finish_from_estimated()
  void TaskDialog::plan_finish_from_subitems()
  {
    QDateTime Date = subitems_time( true, true, false );
    if( Date.isValid() )
      UI.PlanFinish->setDateTime( Date );
    PlanFinishActions.default_action( TimeActions::FromSubItems );
  } // plan_finish_from_subitems()
  void TaskDialog::plan_finish_from_subtasks()
  {
    QDateTime Date = subitems_time( false, true, false );
    if( Date.isValid() )
      UI.PlanFinish->setDateTime( Date );
    PlanFinishActions.default_action( TimeActions::FromSubtasks );
  } // plan_finish_from_subtasks()
  void TaskDialog::plan_finish_blockers_start()
  {
    QDateTime Date = subitems_time( true, false, true );
    if( Date.isValid() )
      UI.PlanFinish->setDateTime( Date );
    PlanFinishActions.default_action( TimeActions::BlockersStart );
  } // plan_finish_blockers_start()
  void TaskDialog::plan_finish_blockers_finish()
  {
    QDateTime Date = subitems_time( true, false, false );
    if( Date.isValid() )
      UI.PlanFinish->setDateTime( Date );
    PlanFinishActions.default_action( TimeActions::BlockersFinish );
  } // plan_finish_blockers_finish()
  void TaskDialog::plan_finish_from_supertask()
  {
    if( SuperTask && SuperTask->plan_finish().isValid() )
      UI.PlanFinish->setDateTime( SuperTask->plan_finish() );
    PlanFinishActions.default_action( TimeActions::FromSupertask );
  } // plan_finish_from_supertask()
  void TaskDialog::check_time( Task* Tsk, bool Start, QDateTime& Result )
  {
    if( Tsk )
    {
      if( Start )
      {
	QDateTime Try = Tsk->plan_start().isValid() ? Tsk->plan_start() : Tsk->plan_finish();
	if( Try.isValid() && ( !Result.isValid() || Result > Try ) )
	  Result = Try;
      }
      else
      {
	QDateTime Try = Tsk->plan_finish().isValid() ? Tsk->plan_finish() : Tsk->plan_start();
	if( Try.isValid() && ( !Result.isValid() || Result < Try ) )
	  Result = Try;
      }
    }
  } // check_time( Task*, bool, QDateTime& )
  QDateTime TaskDialog::subitems_time( bool UseBlockers, bool UseSubtasks, bool Start )
  {
    QDateTime Result;
    if( UseBlockers )
    {
      if( Blockers )
	foreach( Task* B, Blockers->blockers() )
	  check_time( B, Start, Result );
      else if( Object )
	foreach( Task* B, Object->blockers() )
	  check_time( B, Start, Result );
    }
    if( UseSubtasks && Object )
      foreach( Task* T, Object->subtasks() )
	check_time( T, Start, Result );
    return Result;
  } // subitems_time( bool, bool, bool )

  class TaskItemFinder
  {
  public:
    enum { Name = 0x1, Description = 0x2, Comment = 0x4, Everywhere = Name | Description | Comment };
    TaskItemFinder( TasksModel& Model0, TasksModel::Item& Start0, const QRegExp& Pattern0, unsigned Where0 = TaskItemFinder::Everywhere, bool DescendTasks0 = false );
    virtual ~TaskItemFinder() {}
    TasksModel* model() const { return &Model; }
    TasksModel::Item* find_next();
  protected:
    TasksModel::Item* next( TasksModel::Item* ); // Get the next item in the tree (no matter if it has search pattern or not).
    virtual bool descendable( TasksModel::Item& It ) const;
    virtual bool compare( TasksModel::Item& It ) const;
    virtual bool compare( Task& T ) const;
    TasksModel& Model;
    TasksModel::Item& Start;
    QRegExp Pattern;
    unsigned Where;
    bool DescendTasks;
    TasksModel::Item* Current;
  }; // TaskFinder
  TaskItemFinder::TaskItemFinder( TasksModel& Model0, TasksModel::Item& Start0, const QRegExp& Pattern0, unsigned Where0, bool DescendTasks0 )
    : Model( Model0 ), Start( Start0 ), Pattern( Pattern0 ), Where( Where0 ), DescendTasks( DescendTasks0 ), Current( &Start0 ) {}
  TasksModel::Item* TaskItemFinder::find_next()
  {
    TasksModel::Item* Result = next( Current );
    while( Result && !compare( *Result ) )
      Result = next( Result );
    Current = Result;
    return Result;
  } // find_next()
  TasksModel::Item* TaskItemFinder::next( TasksModel::Item* FromItem )
  {
    TasksModel::Item* Result = 0;
    if( FromItem )
    {
      if( descendable( *FromItem ) )
	Result = FromItem->subitem( 0 );
      else
      { // Try to find first sibling.
	TasksModel::Item* TestItem = FromItem;
	for( TasksModel::Item* Parent = TestItem->parent(); Parent && !Result; Parent = TestItem->parent() )
	{
	  int ItemIndex = Parent->subitems().indexOf( TestItem );
	  if( ItemIndex >= 0 && ItemIndex < Parent->subitems().size()-1 )
	    Result = Parent->subitem( ItemIndex+1 );
	  if( Result && Result->relation() == TasksModel::Item::Blocker )
	    Result = 0;
	  TestItem = Parent;
	}
      }
    }
    return Result;
  } // next( TasksModel::Item* )
  bool TaskItemFinder::descendable( TasksModel::Item& It ) const
  {
    bool Result = !It.is_leaf();
    if( Task* T = It.task() )
      if( !DescendTasks || T->subtasks().empty() )
	Result = false;
    if( Result )
    {
      It.populate();
      Result = !It.subitems().empty();
    }
    return Result;
  } // descendable( TasksModel::Item& ) const
  bool TaskItemFinder::compare( TasksModel::Item& It ) const { return It.task() && compare( *It.task() ); }
  bool TaskItemFinder::compare( Task& T ) const
  {
    qDebug() << "Test" << T.name() << "pattern" << Pattern;
    return ( ( (Where&Name) && T.name().contains( Pattern ) )
	     || ( (Where&Description) && T.description().contains( Pattern ) )
	     || ( (Where&Comment) && T.comment().contains( Pattern ) ) );
  }

  // SearchDialog
  class SearchDialog : public QDialog
  {
  public:
    SearchDialog( QWidget* Parent = 0 );
    const QRegExp& pattern() const { return Pattern; }
    unsigned where() const { return Where; }
    void accept();
  protected:
    QRegExp Pattern;
    unsigned Where;
    Ui::SearchDialog UI;
  }; // SearchDialog
  SearchDialog::SearchDialog( QWidget* Parent ) : QDialog( Parent )
  {
    QSettings Set;
    Set.beginGroup( "Status/Recent/Search" );
    Pattern.setPattern( Set.value( "Pattern" ).toString() );
    Pattern.setCaseSensitivity( Qt::CaseSensitivity( Set.value( "CaseSensitive", Qt::CaseInsensitive ).toInt() ) );
    Pattern.setPatternSyntax( QRegExp::PatternSyntax( Set.value( "PatternSyntax", QRegExp::FixedString ).toInt() ) );
    Where = Set.value( "SearchArea", TaskItemFinder::Everywhere ).toInt();
    UI.setupUi( this );
    UI.Pattern->setText( Pattern.pattern() );
    if( Pattern.caseSensitivity() == Qt::CaseSensitive ) UI.CaseSensitive->setChecked( true );
    if( Pattern.patternSyntax() == QRegExp::RegExp ) UI.RegularExpressions->setChecked( true );
    if( Where & TaskItemFinder::Name ) UI.SearchInName->setChecked( true );
    if( Where & TaskItemFinder::Description ) UI.SearchInDescription->setChecked( true );
    if( Where & TaskItemFinder::Comment ) UI.SearchInComment->setChecked( true );
  } // SearchDialog( QWidget* )
  void SearchDialog::accept()
  {
    Pattern.setPattern( UI.Pattern->text() );
    Pattern.setCaseSensitivity( UI.CaseSensitive->isChecked() ? Qt::CaseSensitive : Qt::CaseInsensitive );
    Pattern.setPatternSyntax( UI.RegularExpressions->isChecked() ? QRegExp::RegExp : QRegExp::FixedString );
    Where = 0;
    if( UI.SearchInName->isChecked() ) Where |= TaskItemFinder::Name;
    if( UI.SearchInDescription->isChecked() ) Where |= TaskItemFinder::Description;
    if( UI.SearchInComment->isChecked() ) Where |= TaskItemFinder::Comment;
    if( Pattern.isEmpty() )
      QMessageBox::warning( this, tr( "Search" ), tr( "Please, specify search pattern." ) );
    else if( !Where )
      QMessageBox::warning( this, tr( "Search" ), tr( "Select one or more search area." ) );
    else
    {
      QSettings Set;
      Set.beginGroup( "Status/Recent/Search" );
      Set.setValue( "Pattern", Pattern.pattern() );
      Set.setValue( "CaseSensitive", Pattern.caseSensitivity() );
      Set.setValue( "PatternSyntax", Pattern.patternSyntax() );
      Set.setValue( "SearchArea", Where );
      QDialog::accept();
    }
  } // accept()

  // MainWindow
  QString MainWindow::basename( const QString& FileName )
  {
    int DirIndex = FileName.lastIndexOf( '/' );
    if( DirIndex >= 0 ) DirIndex++;
    else DirIndex = 0;
    QString Result = FileName.mid( DirIndex );
    if( Result.right( 11 ) == ".plansplant" )
      Result = Result.left( Result.size()-11 );
    return Result;
  } // basename( const QString& )
  void MainWindow::recent_file( const QString& FileName )
  {
    QSettings().setValue( "Status/RecentFile", FileName );
  } // recent_file( const QString& )
  class MenuHolder
  {
  public:
    MenuHolder( MainWindow* Parent ) : Bar( new QMenuBar( Parent ) ), Menu( 0 ) {}
    QAction* add_menu( const QString& Name )
    {
      Menu = new QMenu( Name );
      return Bar->addMenu( Menu );
    } // add_menu( const QString& )
    void add_action( QAction* Act )
    {
      if( Menu ) Menu->addAction( Act );
      else Bar->addAction( Act );
    } // add_action( QAction* )
    void add_separator()
    {
      if( Menu ) Menu->addSeparator();
    } // add_separator()
    operator QMenuBar*() { return Bar; }
  protected:
    QMenuBar* Bar;
    QMenu* Menu;
  }; // MenuHolder
  MainWindow::MainWindow( QWidget* Parent, Qt::WindowFlags Flags )
    : QMainWindow( Parent, Flags ), Tasks( 0 ), Tree( 0 ), TimeList( 0 ), TreeMenuAction( 0 ), TimeListMenuAction( 0 ), TreeTools( 0 ), TimeListTools( 0 ), Finder( 0 )
  {
    setWindowTitle( tr( "Plans Plant" ) );
    setWindowIcon( load_icon( "plansplant" ) );
    QWidget* Container = new QWidget( this );
    QLayout* Layout = new QHBoxLayout( Container );
    Layout->setContentsMargins( 0, 0, 0, 0 );
    Tree = new TasksTreeWidget( Container );
    Layout->addWidget( Tree );
    TimeList = new TasksTimeListWidget( Container );
    TimeList->setAlternatingRowColors( true );
    TimeList->setUniformRowHeights( true );
    Layout->addWidget( TimeList );
    TimeList->hide();
    setCentralWidget( Container );
#ifdef PLANSPLANT_HANDHELD
#ifdef Q_WS_MAEMO_5
    if( desktop_view() )
      XcursorSetTheme( QX11Info::display(), "DMZ-White" );
    else
      setAttribute( Qt::WA_Maemo5StackedWindow );
#endif // Q_WS_MAEMO_5
    Tree->setContextMenuPolicy( Qt::ActionsContextMenu );
#endif // PLANSPLANT_HANDHELD
    create_menu();
  } // MainWindow( QWidget*, Qt::WindowFlags )
  MainWindow::~MainWindow()
  {
    close_file( true );
  } // ~MainWindow()
  void MainWindow::create_menu()
  {
    QToolBar* Tools = new QToolBar( tr( "Standard" ), this );
    TreeTools = new QToolBar( tr( "Tasks tree" ), this );
    TimeListTools = new QToolBar( tr( "Time list" ), this );
    // Build menu
    MenuHolder Menu( this );
    if( desktop_view() ) Menu.add_menu( tr( "&File" ) );
    QAction* Act = new QAction( load_icon( "document-new" ), tr( "New file" ), this );
    Act->setShortcut( QKeySequence::New );
    connect( Act, SIGNAL( triggered() ), SLOT( new_file() ) );
    Tools->addAction( Act );
    Menu.add_action( Act );
    Act = new QAction( load_icon( "document-open" ), tr( "Open file..." ), this );
    Act->setShortcut( QKeySequence::Open );
    connect( Act, SIGNAL( triggered() ), SLOT( open_file() ) );
    Tools->addAction( Act );
    Menu.add_action( Act );
    if( desktop_view() ) Menu.add_separator();
    Act = new QAction( load_icon( "document-save" ), tr( "Save file" ), this );
    Act->setShortcut( QKeySequence::Save );
    connect( Act, SIGNAL( triggered() ), SLOT( save_file() ) );
    Tools->addAction( Act );
    Menu.add_action( Act );
    Act = new QAction( load_icon( "document-save-as" ), tr( "Save file as..." ), this );
    Act->setShortcut( QKeySequence::SaveAs );
    connect( Act, SIGNAL( triggered() ), SLOT( save_file_as() ) );
    Menu.add_action( Act );
    if( desktop_view() ) Menu.add_separator();
    Act = new QAction( tr( "Export to HTML..." ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( export_to_html() ) );
    Menu.add_action( Act );
    if( desktop_view() )
    {
      Menu.add_separator();
      Act = new QAction( load_icon( "application-exit" ), tr( "E&xit" ), this );
      Act->setShortcut( QKeySequence::Quit );
      Act->setMenuRole( QAction::QuitRole );
      connect( Act, SIGNAL( triggered() ), SLOT( close() ) );
      Menu.add_action( Act );
    }
    TreeMenuAction = Menu.add_menu( tr( "&Edit" ) );
    QAction* FindAction = new QAction( load_icon( "edit-find" ), tr( "&Find" ), this );
    FindAction->setShortcut( QKeySequence::Find );
    connect( FindAction, SIGNAL( triggered() ), SLOT( search() ) );
    QAction* FindNextAction = new QAction( tr( "&Find Next" ), this );
    FindNextAction->setShortcut( QKeySequence::FindNext );
    connect( FindNextAction, SIGNAL( triggered() ), SLOT( search_next() ) );
    Menu.add_action( FindAction );
    Menu.add_action( FindNextAction );
    Menu.add_action( Tree->task_add_action() );
    Menu.add_action( Tree->task_edit_action() );
    if( desktop_view() ) Menu.add_separator();
    Menu.add_action( Tree->dependency_add_action() );
    Menu.add_action( Tree->dependencies_edit_action() );
    Menu.add_action( Tree->dependency_remove_action() );
    if( desktop_view() ) Menu.add_separator();
    Menu.add_action( Tree->move_up_action() );
    Menu.add_action( Tree->move_down_action() );

    TreeTools->addAction( Tree->task_add_action() );
    TreeTools->addAction( Tree->task_edit_action() );
    TreeTools->addAction( Tree->dependency_add_action() );
    TreeTools->addSeparator();
    TreeTools->addAction( Tree->move_up_action() );
    TreeTools->addAction( Tree->move_down_action() );

    TimeListMenuAction = Menu.add_menu( tr( "&Edit" ) );
    Menu.add_action( FindAction );
    Menu.add_action( FindNextAction );
    Menu.add_action( TimeList->task_add_action() );
    Menu.add_action( TimeList->task_edit_action() );
    if( desktop_view() ) Menu.add_separator();
    Menu.add_action( TimeList->dependency_add_action() );
    Menu.add_action( TimeList->dependencies_edit_action() );
    Menu.add_action( TimeList->dependency_remove_action() );
    if( desktop_view() ) Menu.add_separator();
    Menu.add_action( TimeList->move_up_action() );
    Menu.add_action( TimeList->move_down_action() );
    
    TimeListTools->addAction( TimeList->task_add_action() );
    TimeListTools->addAction( TimeList->task_edit_action() );
    TimeListTools->addAction( TimeList->dependency_add_action() );
    TimeListTools->addSeparator();
    TimeListTools->addAction( TimeList->move_up_action() );
    TimeListTools->addAction( TimeList->move_down_action() );
    
    //if( desktop_view() )
    Menu.add_menu( tr( "&View" ) );
    QActionGroup* Gr = new QActionGroup( this );
    Act = new QAction( tr( "Tasks &tree" ), Gr );
    Act->setCheckable( true );
    Act->setChecked( true );
    connect( Act, SIGNAL( triggered() ), SLOT( show_tasks_tree() ) );
    Menu.add_action( Act );
    Act = new QAction( tr( "Times &list" ), Gr );
    Act->setCheckable( true );
    connect( Act, SIGNAL( triggered() ), SLOT( show_time_list() ) );
    Menu.add_action( Act );
#if 0
    if( desktop_view() ) Menu.add_separator();
    Act = new QAction( tr( "Tool&bar" ), this );
    Act->setCheckable( true );
    Act->setChecked( true );
    Tools->connect( Act, SIGNAL( toggled( bool ) ), SLOT( setVisible( bool ) ) );
    Menu.add_action( Act );
#endif

    Menu.add_menu( tr( "&Tools" ) );
    Act = new QAction( tr( "&Move times" ), this );
    Menu.add_action( Act );
    connect( Act, SIGNAL( triggered() ), SLOT( move_times() ) );
    // if( desktop_view() )
    Menu.add_menu( tr( "&Help" ) );
    Act = new QAction( load_icon( "help-about" ), tr( "About..." ), this );
    Act->setMenuRole( QAction::AboutRole );
    connect( Act, SIGNAL( triggered() ), SLOT( about() ) );
    Menu.add_action( Act );
    Act = new QAction( tr( "About Qt..." ), this );
    Act->setMenuRole( QAction::AboutQtRole );
    connect( Act, SIGNAL( triggered() ), qApp, SLOT( aboutQt() ) );
    Menu.add_action( Act );

    StartStopAction = new QAction( load_icon( "media-playback-start" ), tr( "Start" ), this );
    connect( StartStopAction, SIGNAL( triggered() ), SLOT( start_stop() ) );
    TreeTools->addAction( StartStopAction );
    TimeListTools->addAction( StartStopAction );

    setMenuBar( Menu );
#ifdef PLANSPLANT_HANDHELD
    if( desktop_view() )
    {
      if( menuWidget() ) menuWidget()->show();
      // Force toolbar placement
      Tools->setMovable( true );
      addToolBar( Qt::TopToolBarArea, Tools );
      TreeTools->setMovable( true );
      addToolBar( Qt::TopToolBarArea, TreeTools );
      TimeListTools->setMovable( true );
      addToolBar( Qt::TopToolBarArea, TimeListTools );
    }
    else
    {
      addToolBar( Tools );
      addToolBar( TreeTools );
      addToolBar( TimeListTools );
    }
#else
    addToolBar( Tools );
    addToolBar( TreeTools );
    addToolBar( TimeListTools );
#endif // PLANSPLANT_HANDHELD
    show_tasks_tree();
  } // create_menu()
  QSize MainWindow::sizeHint() const { return QSize( 800, 480 ); }
  bool MainWindow::close_file( bool Force )
  {
    if( Tasks )
    {
      if( Tasks->is_modified() )
      {
	QMessageBox::StandardButtons Buttons = QMessageBox::Save | QMessageBox::Discard;
	if( !Force ) Buttons |= QMessageBox::Cancel;
	switch( QMessageBox::question( this, QString(), tr( "The file is modified. Do you want to save it?" ), Buttons ) )
	{
	case QMessageBox::Save:
	  if( !save_file() && !Force ) break;
	case QMessageBox::Discard:
 	  // We assume that models will get notification.
	  delete Tasks;
	  Tasks = 0;
	default:
	  break;
	}
      }
      else
      {
	// See above.
	delete Tasks;
	Tasks = 0;
      }
    }
    update_buttons();
    return !Tasks;
  } // close_file( bool )
  bool MainWindow::open_file()
  {
    bool Result = false;
    QFileDialog::Options Opt = 0;
#ifdef PLANSPLANT_USE_QT_FILE_DIALOG
    Opt = QFileDialog::DontUseNativeDialog;
#endif
    QString Filter;
    if( desktop_view() )
      Filter = tr( "Plans Plant files" ) + " (*.plansplant);;" + tr( "All files" ) + " (*)";
    else
      Filter = "Plan (*.plansplant)";
    QString NewName = QFileDialog::getOpenFileName( this, tr( "Open file:" ), Tasks ? Tasks->file_name() : QString(), Filter, 0, Opt );
    if( !NewName.isEmpty() )
      Result = open_file( NewName );
    return Result;
  } // open_file()
  bool MainWindow::open_file( const QString& NewFileName, bool AddToRecent )
  {
    bool Result = false;
    TasksFile* NewFile = new TasksFile( NewFileName );
    if( NewFile->good() && close_file() )
    {
      Tasks = NewFile;
      if( Tree )
	Tree->tasks( new TasksTreeModel( *Tasks ) );
      if( TimeList )
	TimeList->tasks( new TasksTimeListModel( *Tasks ) );
      connect_actions();
      setWindowTitle( tr( "Plans Plant: " ) + basename( NewFileName ) );
      if( AddToRecent ) recent_file( NewFileName );
      Result = true;
    }
    else
    {
      if( !NewFile->good() )
	QMessageBox::warning( this, tr( "Plans Plant" ), tr( "Can't open file\n\"" ) + NewFileName + ( "\"." ) );
      delete NewFile;
    }
    return Result;
  } // open_file( const QString&, bool )
  bool MainWindow::save_file()
  {
    bool Result = false;
    if( Tasks )
    {
      if( Tasks->file_name().isEmpty() )
	Result = save_file_as();
      else
	Result = Tasks->save();
    }
    return Result;
  } // save_file()
  bool MainWindow::save_file_as()
  {
    bool Result = false;
    if( Tasks )
    {
      QFileDialog::Options Opt = 0;
#ifdef PLANSPLANT_USE_QT_FILE_DIALOG
      Opt = QFileDialog::DontUseNativeDialog;
#endif
      QString Filter;
      QString Extension = "plansplant";
      if( desktop_view() ) 
	Filter = tr( "Plans Plant files" ) + " (*." + Extension + ");;" + tr( "All files" ) + " (*)";
      QString NewName = QFileDialog::getSaveFileName( this, tr( "Select file for saving:" ), Tasks->file_name(), Filter, 0, Opt );
      if( !NewName.isEmpty() )
      {
	if( !desktop_view() )
	{
	  if( !NewName.endsWith( '.'+Extension ) ) // Force .plansplant extension
	  {
	    if( !NewName.endsWith( '.' ) ) NewName += '.';
	    NewName += Extension;
	  }
	}
	Result = Tasks->save( NewName );
      }
      if( Result )
      {
	setWindowTitle( tr( "Plans Plant: " ) + basename( NewName ) );
	recent_file( NewName );
      }
    }
    return Result;
  } // save_file_as()
  bool MainWindow::export_to_html()
  {
    bool Result = false;
    if( Tasks )
    {
      QString ExportName = QSettings().value( "Status/HTMLpath" ).toString();
      QFileDialog::Options Opt = 0;
#ifdef PLANSPLANT_USE_QT_FILE_DIALOG
      Opt = QFileDialog::DontUseNativeDialog;
#endif
#ifdef PLANSPLANT_HANDHELD
      if( ExportName.isEmpty() )
	ExportName = "/home/user/MyDocs/";
#endif // PLANSPLANT_HANDHELD
      QString BaseName = basename( Tasks->file_name() );
      if( BaseName.isEmpty() ) BaseName = "plans";
      ExportName += BaseName + ".html";
      QString Filter;
      if( desktop_view() )
	Filter = tr( "HTML files" ) + " (*.html *.htm);;" + tr( "All files" ) + " (*)";
      ExportName = QFileDialog::getSaveFileName( this, tr( "Export to HTML. Select file:" ), ExportName, Filter, 0, Opt );
      if( !ExportName.isEmpty() )
      {
	HTMLExportFilter Exp( ExportName );
	Result = Exp.export_tasks( Tasks->roots() );
	if( Result )
	{ //! \todo move save/restore html path to export object
	  int DirIndex = ExportName.lastIndexOf( '/' );
	  if( DirIndex < 0 ) DirIndex = ExportName.lastIndexOf( '\\' );
	  QSettings().setValue( "Status/HTMLpath", DirIndex < 0 ? QString() : ExportName.left( DirIndex+1 ) );
	}
      }
    }
    return Result;
  } // export_to_html()
  bool MainWindow::new_file()
  {
    bool Result = false;
    if( close_file() )
    {
      Tasks = new TasksFile;
      if( Tree )
	Tree->tasks( new TasksTreeModel( *Tasks ) );
      if( TimeList )
	TimeList->tasks( new TasksTimeListModel( *Tasks ) );
      connect_actions();
      Result = true;
      setWindowTitle( tr( "Plans Plant (new document)" ) );
    }
    return Result;
  } // new_file()
  void MainWindow::about()
  {
    QMessageBox::about( this, tr( "About Plans Plant..." ),
			tr( "Plans Plant. Version " ) + QApplication::applicationVersion() +
			trUtf8( "\n© Copyright 2010 Nick Slobodsky (Николай Слободской) <plansplant@slobodsky.ru>\nWeb: http://plansplant.garage.maemo.org"
				"\n\nThis is a simple planning application: tasks tree with planned start/due times."
				"\nSome icons and icons' parts are from the GNOME environment, © 2007–2009 Jakub \'jimmac\' Steiner" ) );
  } // about()
  void MainWindow::show_tasks_tree()
  {
    Tree->setVisible( true ); 
    addToolBar( TreeTools );
    TreeTools->show();
    TreeMenuAction->setVisible( true );

    TimeList->setVisible( false );
    removeToolBar( TimeListTools );
    TimeListMenuAction->setVisible( false );
    update_buttons();
  } // show_tasks_tree()
  void MainWindow::show_time_list()
  {
    TimeList->setVisible( true );
    addToolBar( TimeListTools );
    TimeListTools->setVisible( true );
    TimeListMenuAction->setVisible( true );

    Tree->setVisible( false );
    removeToolBar( TreeTools );
    TreeMenuAction->setVisible( false );
    update_buttons();
  } // show_time_list()
  void MainWindow::start_stop()
  {
    if( Tasks )
    {
      if( Tasks->active_task() )
      {
	qDebug() << "Stop task" << Tasks->active_task()->name();
	Tasks->stop_work();
	update_buttons();
      }
      else
      {
	Task* NewTask = 0;
	if( Tree->isVisible() )
	  NewTask = Tree->selected_task();
	else if( TimeList->isVisible() )
	  NewTask = TimeList->selected_task();
	if( NewTask )
	{
	  qDebug() << "Start task" << NewTask->name();
	  Tasks->start_work( *NewTask );
	  update_buttons();
	}
      }
    }
  } // start_stop()
  void MainWindow::update_buttons()
  {
    bool Enable = true;
    if( Tasks )
    {
      if( Task* Active = Tasks->active_task() )
      {
	StartStopAction->setIcon( load_icon( "media-playback-stop" ) );
	StartStopAction->setText( "Stop " + Active->name() );
	StartStopAction->setToolTip( "Stop \"" + Active->name() + '\"' );
      }
      else
      {
	Task* NewTask = 0;
	if( Tree->isVisible() )
	  NewTask = Tree->selected_task();
	else if( TimeList->isVisible() )
	  NewTask = TimeList->selected_task();
	if( NewTask )
	{
	  StartStopAction->setIcon( load_icon( "media-playback-start" ) );
	  StartStopAction->setText( "Start" );
	  StartStopAction->setToolTip( "Start \"" + NewTask->name() + '\"' );
	}
	else Enable = false;
      }
    }
    else Enable = false;
    if( !Enable )
    {
      StartStopAction->setIcon( load_icon( "media-playback-start" ) );
      StartStopAction->setText( "Start" );
      StartStopAction->setToolTip( "Nothing to start" );
    }
    StartStopAction->setEnabled( Enable );
  } // update_buttons()
  void MainWindow::search()
  {
    TasksTreeWidget* Tr = 0;
    bool DescendTasks = false;
    if( Tree->isVisible() )
    {
      Tr = Tree;
      DescendTasks = true;
    }
    else if( TimeList->isVisible() )
      Tr = TimeList;
    if( Tr )
      if( TasksModel* Model = Tr->tasks() )
	if( TasksModel::Item* Start = Tr->selected_item() )
	{
	  SearchDialog Dlg( this );
	  if( Dlg.exec() )
	  {
	    if( Finder ) delete Finder;
	    Finder = new TaskItemFinder( *Model, *Start, Dlg.pattern(), Dlg.where(), DescendTasks );
	    if( TasksModel::Item* Found = Finder->find_next() )
	      Tr->setCurrentIndex( Model->index( *Found ) );
	    else
	      QMessageBox::information( this, tr( "Search" ), tr( "Task not found" ) );
	  }
	}
  } // search()
  void MainWindow::search_next()
  {
    bool FallBack = true;
    if( Finder )
    {
      TasksTreeWidget* Tr = 0;
      if( Tree->isVisible() )
	Tr = Tree;
      else if( TimeList->isVisible() )
	Tr = TimeList;
      if( Tr )
	if( TasksModel* Model = Tr->tasks() )
	  if( Model == Finder->model() )
	  {
	    if( TasksModel::Item* Found = Finder->find_next() )
	      Tr->setCurrentIndex( Model->index( *Found ) );
	    else
	      QMessageBox::information( this, tr( "Search" ), tr( "Task not found" ) );
	    FallBack = false;
	  }
    }
    if( FallBack ) search();
  } // search_next()
  void MainWindow::move_times()
  {
    if( Tasks )
    {
      MoveTimesDialog Dlg( this );
      if( Dlg.exec() )
      {
	for( TasksIterator It( *Tasks ); *It; ++It )
	  if( It->plan_start().isValid() || It->plan_finish().isValid() )
	    if( ( !It->plan_start().isValid() || ( It->plan_start() >= Dlg.from() && It->plan_start() <= Dlg.to() ) )
		&& ( !It->plan_finish().isValid() || ( It->plan_finish() >= Dlg.from() && It->plan_finish() <= Dlg.to() ) ) )
	    {
	      Task::ChangesList Changes( **It );
	      if( It->plan_start().isValid() )  { Changes.add( *new Task::Changes::PlanStart( It->plan_start().addSecs( Dlg.move() ) ) ); }
	      if( It->plan_finish().isValid() ) { Changes.add( *new Task::Changes::PlanFinish( It->plan_finish().addSecs( Dlg.move() ) ) ); }
	      if( !Changes.changes().empty() )	Tasks->change_task( Changes );
	    }
      }
    }
  } // move_times()
  void MainWindow::closeEvent( QCloseEvent* Close )
  {
    if( close_file() )
      QMainWindow::closeEvent( Close );
    else
      Close->ignore();
  } // closeEvent( QCloseEvent* )
  void MainWindow::task_changed( Task& /*Object*/, Task::Change::FieldID Field )
  {
    if( Field == Task::Change::Times )
      update_buttons();
  } // task_changed( Task&, Task::Change::FieldID )
  void MainWindow::connect_actions()
  {
    if( Tasks ) add_informer( *Tasks );
    connect( Tree->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ), SLOT( update_buttons() ) );
    connect( TimeList->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ), SLOT( update_buttons() ) );
    update_buttons();
  } // connect_actions()
#ifdef PLANSPLANT_HANDHELD
  bool MainWindow::DesktopView = false;
#else
  bool MainWindow::DesktopView = true;
#endif // PLANSPLANT_HANDHELD
} // PlansPlant
