// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <timeshop.hpp>
#include "plansplant/tasks_changes.hpp"
#include "plansplant/events.hpp"
#include "plansplant/syncserver.hpp"
#include <QDebug>
#include <QApplication>

namespace PlansPlant
{
  QDateTime read_time( QXmlStreamReader& Stream ) // no "fromUTCString" method in QDateTime.
  {
    QDateTime Result;
    QString TimeStr = Stream.readElementText();
    if( !TimeStr.isEmpty() && TimeStr != "none" )
    {
      if( TimeStr.contains( '.' ) )
	Result = QDateTime::fromString( TimeStr, "yyyy-MM-dd hh:mm:ss.zzz" );
      else
	Result = QDateTime::fromString( TimeStr, "yyyy-MM-dd hh:mm:ss" );
      Result.setTimeSpec( Qt::UTC );
      Result = Result.toLocalTime();
    }
    return Result;
  } // read_time( QXmlStreamReader& Stream )
  QString time_string( const QDateTime& TimeVal ) { return TimeVal.isValid() ? TimeVal.toUTC().toString( "yyyy-MM-dd hh:mm:ss.zzz" ) : "none"; }

  Task::ID::ID( quint32 Value0, int Origin0 ) : Origin( Origin0 ), Value( Value0 ) {}
  Task::ID::ID( const QString& Str ) : Origin( 0 ), Value( 0 ) { operator=( Str ); } // ID( const QString& )
  Task::ID Task::ID::operator=( const QString& Str )
  {
    if( !Str.isEmpty() )
    {
      int AtPos = Str.indexOf( '@' );
      if( AtPos < 0 )
      {
	Origin = 0; //!< \todo Maybe leave as is?
	Value = Str.toUInt();
      }
      else
      {
	Origin = Str.mid( AtPos+1 ).toUInt();
	Value = Str.left( AtPos ).toUInt();
      }
    }
    return *this;
  } // ID = const QString&
  QString Task::ID::str() const { return QString::number( Value ) + ( Origin == 0 ? QString() : ( '@' + QString::number( Origin ) ) ); }
  //! \todo Move relations and fields to separate library (SUIT-like).
  // Task::Change members are in the task_changes.cpp
  void Task::ChangesList::inform( Watcher& Dest ) const
  {
    foreach( Task::Change* Ch, Changes )
      Ch->inform( Dest, Object );
  } // inform( Watcher& ) const
  bool Task::ChangesList::load( QXmlStreamReader& Stream, TasksFile& Tasks )
  {
    while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
      if( Stream.name() == "change" )
      {
	int Field = 0;
	if( Timeshop::Persistent::Loader::attribute( Stream.attributes(), "field", Field ) )
	{
	  Change* NewChange = 0;
	  switch( Field )
	  { // No Dependents list manipulations yet.
	  case Change::Name: NewChange = Task::Changes::Name::load( Stream, Tasks ); break;
	  case Change::Description: NewChange = Task::Changes::Description::load( Stream, Tasks ); break;	
	  case Change::Priority: NewChange = Task::Changes::Priority::load( Stream, Tasks ); break;
	  case Change::Completed: NewChange = Task::Changes::Completed::load( Stream, Tasks ); break;
	  case Change::PlanStart: NewChange = Task::Changes::PlanStart::load( Stream, Tasks ); break;
	  case Change::PlanFinish: NewChange = Task::Changes::PlanFinish::load( Stream, Tasks ); break;
	  case Change::Estimation: NewChange = Task::Changes::Estimation::load( Stream, Tasks ); break;
	  case Change::EstimationUnits: NewChange = Task::Changes::EstimationUnits::load( Stream, Tasks ); break;
	  case Change::Comment: NewChange = Task::Changes::Comment::load( Stream, Tasks ); break;
	  case Change::SuperTask: NewChange = Task::Changes::SuperTask::load( Stream, Tasks ); break;
	  case Change::SubTasks: NewChange = Task::Changes::SubTasks::load( Stream, Tasks ); break;
	  case Change::Blockers: NewChange = Task::Changes::Blockers::load( Stream, Tasks ); break;
	  case Change::Times: NewChange = Task::Changes::Times::load( Stream, Tasks ); break;
	  default:
	    qDebug() << "Don't know how to load field" << Field;
	    Timeshop::Persistent::Loader::skip( Stream );
	    break;
	  }
	  if( NewChange ) Changes.push_back( NewChange );
	}
	else
	{
	  qDebug() << "!!! Change element has no field attribute.";
	  Timeshop::Persistent::Loader::skip( Stream );
	}
      }
      else Timeshop::Persistent::Loader::skip( Stream );
    return true;
  } // load( QXmlStreamReader&, TasksFile& )
  void Task::ChangesList::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeStartElement( "changes_list" );
    Stream.writeAttribute( "task_id", Object.id() );
    foreach( Task::Change* Ch, Changes )
      Ch->write( Stream );
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const

  Task::Watcher::Event::Type Task::Watcher::Event::type() const { return None; }
  void Task::Watcher::Event::apply( TasksFile& /*Tasks*/ ) const { qDebug() << "??? Empty event" << id().str() << "applied."; }
  void Task::Watcher::Event::inform( Watcher& /*Dest*/ ) const { /* Nothing here. */ }
  void Task::Watcher::Event::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeStartElement( "event" );
    Stream.writeAttribute( "id", Ident );
    Stream.writeAttribute( "type", QString::number( type() ) );
    write_fields( Stream );
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const
  void Task::Watcher::Event::write_fields( QXmlStreamWriter& /*Stream*/ ) const { /* Nothing here: no fields. */ }

  Task::Watcher::~Watcher()
  {
    while( !Informers.empty() )
      remove_informer( *Informers.back() );
  } // ~Watcher()
  void Task::Watcher::add_informer( TasksFile& NewInformer )
  {
    if( !Informers.contains( &NewInformer ) )
    {
      Informers.push_back( &NewInformer );
      NewInformer.add_watcher( *this );
    }
  } // add_informer( TaskFile& )
  void Task::Watcher::remove_informer( TasksFile& OldInformer )
  {
    if( Informers.contains( &OldInformer ) )
    {
      Informers.removeAll( &OldInformer );
      OldInformer.remove_watcher( *this );
    }
  } // remove_informer( TaskFile& )
  void Task::Watcher::event( const Event& Ev ) { Ev.inform( *this ); }
  void Task::Watcher::task_added( Task& /*NewTask*/ ) {}
  void Task::Watcher::task_removed( Task::ID /*TaskID*/ ) {}
  void Task::Watcher::task_changed( const ChangesList& Changes ) { Changes.inform( *this ); }
  void Task::Watcher::task_changed( Task& /*Object*/, Task::Change::FieldID /*Field*/ ) {}
  void Task::Watcher::task_moved( const Task& /*Object*/, int /*OldIndex*/, int /*NewIndex*/ ) {}
  void Task::Watcher::blocker_added( const Task& /*Object*/, Task& /*NewBlocker*/ ) {}
  void Task::Watcher::blocker_removed( const Task& /*Object*/, const Task& /*OldBlocker*/ ) {}
  void Task::Watcher::blocker_moved( const Task& /*Object*/, const Task& /*Blocker*/, int /*OldIndex*/, int /*NewIndex*/ ) {}
  void Task::Watcher::blockers_replaced( const Task& /*Object*/ ) {}

  Task::ChangesList::ChangesList( Task& Object0, Change& FirstChange ) : Object( Object0 ) { add( FirstChange ); }
  Task::ChangesList::~ChangesList()
  {
    while( !Changes.empty() )
    {
      delete Changes.back();
      Changes.pop_back();
    }
  } // ~ChangesList()

  void Task::TimeSlice::start( const QDateTime& NewStart ) { Start = NewStart; }
  void Task::TimeSlice::start_now() { start( QDateTime::currentDateTime() ); }
  void Task::TimeSlice::finish( const QDateTime& NewFinish ) { Finish = NewFinish; }
  void Task::TimeSlice::finish_now() { finish( QDateTime::currentDateTime() ); }
  void Task::TimeSlice::load( QXmlStreamReader& Stream )
  {
    while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
      if( Stream.name() == "start" )
	Start = read_time( Stream );
      else if( Stream.name() == "finish" )
	Finish = read_time( Stream );
      else
	Timeshop::Persistent::Loader::skip( Stream );
  } // load( QXmlStreamReader& )
  void Task::TimeSlice::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeStartElement( "time_slice" );
    if( Start.isValid() ) Stream.writeTextElement( "start", time_string( Start ) );
    if( Finish.isValid() ) Stream.writeTextElement( "finish", time_string( Finish ) );
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& )

  QString Task::units_name( TimeUnits Units )
  {
    QString Result = QObject::tr( "unknown" );
    switch( Units )
    {
    case Seconds:   Result = QObject::tr( "Seconds" ); break;
    case Minutes:   Result = QObject::tr( "Minutes" ); break;
    case Hours:     Result = QObject::tr( "Hours" ); break;
    case WorkDays:  Result = QObject::tr( "WorkDays" ); break;
    case Days:      Result = QObject::tr( "Days" ); break;
    case WorkWeeks: Result = QObject::tr( "WorkWeeks" ); break;
    case Weeks:     Result = QObject::tr( "Weeks" ); break;
    case Months:    Result = QObject::tr( "Months" ); break;
    case Quarters:  Result = QObject::tr( "Quarters" ); break;
    case Years:     Result = QObject::tr( "Years" ); break;
    }
    return Result;
  } // units_name( TimeUnits )
  QString Task::units_short_name( TimeUnits Units )
  {
    QString Result = QObject::tr( "unknown" );
    switch( Units )
    {
    case Seconds:   Result = QObject::tr( "s" ); break;
    case Minutes:   Result = QObject::tr( "min" ); break;
    case Hours:     Result = QObject::tr( "h" ); break;
    case WorkDays:  Result = QObject::tr( "wd" ); break;
    case Days:      Result = QObject::tr( "d" ); break;
    case WorkWeeks: Result = QObject::tr( "ww" ); break;
    case Weeks:     Result = QObject::tr( "w" ); break;
    case Months:    Result = QObject::tr( "M" ); break;
    case Quarters:  Result = QObject::tr( "Q" ); break;
    case Years:     Result = QObject::tr( "Y" ); break;
    }
    return Result;
  } // units_short_name( TimeUnits )
  QString Task::priority_name( int Pri )
  {
    QString Result = QApplication::translate( "PlansPlant::Task", "Custom" );
    switch( Pri )
    {
    case -2: Result = QApplication::translate( "PlansPlant::Task", "Lowest" ); break;
    case -1: Result = QApplication::translate( "PlansPlant::Task", "Low" ); break;
    case 0: Result = QApplication::translate( "PlansPlant::Task", "Normal" ); break;
    case 1: Result = QApplication::translate( "PlansPlant::Task", "High" ); break;
    case 2: Result = QApplication::translate( "PlansPlant::Task", "Highest" ); break;
    default: break;
    }
    return Result;
  } // priority_name( int )
  QDateTime Task::add_time( const QDateTime& Start, Time Amount, TimeUnits Units )
  {
    QDateTime Result = Start;
    if( Amount != 0 )
      switch( Units )
      {
      case Years:
	Result = Start.addYears( Amount );
	break;
      case Quarters:
	Result = Start.addMonths( Amount*3 );
	break;
      case Months:
	Result = Start.addMonths( Amount );
	break;
      case WorkWeeks:
	Result = Start.addDays( Amount*7 );
	break;
      case WorkDays:
	Result = Start.addDays( ( Amount / 5 ) * 7 + Amount % 5 );
	break;
      default:
	Result = Start.addSecs( Amount * Units );
	break;
      }
    return Result;
  } // add_time( const QDateTime&, Time, TimeUnits )

  Task::Task( const QString& Name0, Pointer SuperTask0, ID ID0 )
  : Ident( ID0 ), SuperTask( SuperTask0 ), Name( Name0 ), Priority( 0 ), Completed( 0 ), Estimation( 0 ), EstimationUnits( Seconds )
  {
#if 0
    if( LastID < Ident ) LastID = Ident;
#endif
    if( SuperTask && SuperTask->SubTasks.indexOf( this ) < 0 )
      SuperTask->SubTasks.push_back( this );
  } // Task( const QString&, Pointer* )
  Task::~Task()
  {
    while( !Blockers.empty() )
      remove_blocker( *Blockers.back() );
    while( !Dependents.empty() )
      Dependents.back()->remove_blocker( *this ); //!< \todo Use relation objects.
    while( !SubTasks.empty() )
      delete SubTasks.back(); // Will be removed from this list in the subtask's destructor.
    supertask( 0 );
  } // ~Task()

  void Task::supertask( Task* NewTask )
  {
    if( NewTask != SuperTask )
    {
      if( SuperTask )
	SuperTask->SubTasks.removeAll( this );
      SuperTask = NewTask;
      if( SuperTask )
	SuperTask->SubTasks.push_back( this );
    }
  } // supertask( Task* )

  Task* Task::find_task( Task::ID ForID ) const
  {
    Task* Result = 0;
    if( ForID.valid() )
      for( int I = 0; I < SubTasks.size() && !Result; I++ )
	if( SubTasks[ I ]->id() == ForID )
	  Result = SubTasks[ I ];
	else
	  Result = SubTasks[ I ]->find_task( ForID );
    return Result;
  } // find_task( Task::ID ) const
  bool Task::check_loop( Task& NewChild ) const
  {
    return check_loop( NewChild.blockers(), &NewChild );
  } // check_loop( Task& )
  bool Task::blockers_list_helper( Task::List& List, const Task::List& Source ) const
  {
    bool Result = false;
    for( Task::List::const_iterator It = Source.begin(); It != Source.end() && !Result; It++ )
      if( Task* Dep = *It )
      {
	if( Dep == this ) Result = true;
	else
	  if( !List.contains( Dep ) )
	    List.push_back( Dep );
      }
    return Result;
  } // blockers_list_helper( Task::List&, const Task::List& )
  bool Task::check_loop( const Task::List& Blockers, Task* NewChild ) const
  {
    bool Result = this == NewChild;
    if( !Result ) // Search for it in the tasks that depends on this
    { //! \todo Optimize the search
      Task::List Deps;
      int I = 0;
      if( NewChild )
      {
	Deps.push_back( NewChild );
	Result = blockers_list_helper( Deps, NewChild->subtasks() );
	I++;
      }
      Result = Result || blockers_list_helper( Deps, Blockers );
      while( I < Deps.size() && !Result )
      {
	if( Task* Dep = Deps[ I ] )
	{
	  if( Dep == this ) Result = true; // No blocker for this, because we check it when add data to the Deps list.
	  else Result = Result || blockers_list_helper( Deps, Dep->subtasks() ) || blockers_list_helper( Deps, Dep->blockers() );
	}
	I++;
      } // for each Dep.
    }
    return Result;
  } // check_loop( const Task::List&, Task* ) const
  bool Task::move_subtask( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < SubTasks.size() && To >= 0 && To < SubTasks.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	{
	  qDebug() << "Swap tasks" << Index << "and" << Index+Inc;
	  SubTasks.swap( Index, Index+Inc );
	}
	Result = true;
      }
    }
    return Result;
  } // move_subtask( int, int )
  bool Task::move_blocker( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < Blockers.size() && To >= 0 && To < Blockers.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	{
	  qDebug() << "Swap tasks" << Index << "and" << Index+Inc;
	  Blockers.swap( Index, Index+Inc );
	}
	Result = true;
      }
    }
    return Result;
  } // move_blocker( int, int )
  bool Task::blocked() const
  {
    bool Result = false;
    foreach( Task* SubTask, subtasks() )
      if( SubTask && SubTask->completed() < 1 )
	Result = true;
    if( !Result )
      foreach( Task* Dep, blockers() )
	if( Dep && Dep->completed() < 1 )
	  Result = true;
    return Result;
  } // blocked() const
  void Task::add_to_map( Map& TasksMap )
  {
    if( Ident )
    {
      TasksMap[ Ident ] = this;
      foreach( Task* SubTask, SubTasks )
	if( SubTask )
	  SubTask->add_to_map( TasksMap );
    }
    else qDebug() << "!!! Task::add_to_map: task without ID"; 
  } // add_to_map( Map& )
  bool Task::active() const { return !Times.empty() && Times.back().active(); }
  QDateTime Task::start_time() const { return active() ? Times.back().start() : QDateTime(); }
  int Task::secs( const QDateTime& ToTime ) const
  {
    int Result = 0;
    if( ToTime.isValid() )
    {
      TimeSlice::List::const_iterator It = Times.begin();
      for( ; It != Times.end() && It->finish().isValid() && It->finish() <= ToTime; ++It )
	Result += It->secs();
      if( It != Times.end() && It->start().isValid() && It->start() <= ToTime )
	Result += It->start().secsTo( ToTime );
    }
    else
      for( TimeSlice::List::const_iterator It = Times.begin(); It != Times.end(); ++It )
	Result += It->secs();
    return Result;
  } // seconds( const QDateTime& )
  void Task::start( const QDateTime& StartTime )
  {
    if( !active() && StartTime.isValid() )
      Times.push_back( TimeSlice( StartTime ) );
  } // start( const QDateTime& )
  void Task::stop( const QDateTime& StopTime )
  {
    if( active() && StopTime.isValid() )
      Times.back().finish( StopTime );
  } // stop( const QDateTime& )
  void Task::insert_time( const TimeSlice& Slice, int Index )
  {
    if( Index >= 0 && Index <= Times.size() )
      Times.insert( Index, Slice );
    else
      Times.append( Slice );
  } // insert_time( const TimeSlice&, int )
  void Task::remove_time( int Index )
  {
    if( Index >= 0 && Index < Times.size() )
      Times.removeAt( Index );
  } // remove_time( int )
  void Task::move_time( int From, int To )
  {
    if( From >= 0 && From < Times.size() && To >= 0 && To < Times.size() )
      Times.move( From, To );
  } // move_time( int From, int To )
  void Task::write( QXmlStreamWriter& Stream, bool Standalone )
  {
    if( Ident )
    {
      Stream.writeStartElement( "task" );
      Stream.writeAttribute( "id", Ident );
      // We don't  write a supertask (parent), if we save tasks in parent's subtasks list, but we have to write it in the TaskAdded event.
      if( Standalone ) qDebug() << "Write standalone task," << ( supertask() ? "supertask is:" + supertask()->name() : QString( "no supertask" ) );
      if( Standalone && supertask() ) Stream.writeTextElement( "supertask", supertask()->id() );
      if( !Name.isEmpty() ) Stream.writeTextElement( "name", Name );
      if( !Description.isEmpty() ) Stream.writeTextElement( "description", Description );
      if( Priority ) Stream.writeTextElement( "priority", QString::number( Priority ) );
      if( !Comment.isEmpty() ) Stream.writeTextElement( "comment", Comment );
      if( PlanStart.isValid() ) Stream.writeTextElement( "plan_start", time_string( PlanStart ) );
      if( PlanFinish.isValid() ) Stream.writeTextElement( "plan_finish", time_string( PlanFinish ) );
      if( Completed > 0 ) Stream.writeTextElement( "completed", QString::number( Completed ) );
      if( Estimation > 0 )
      {
	Stream.writeStartElement( "estimation" );
	Stream.writeAttribute( "units", QString::number( EstimationUnits ) );
	Stream.writeCharacters( QString::number( Estimation ) );
	Stream.writeEndElement();
      }
      if( !Standalone && !SubTasks.isEmpty() )
      {
	Stream.writeStartElement( "subtasks" );
	foreach( Task* SubTask, SubTasks )
	  if( SubTask )
	    SubTask->write( Stream );
	Stream.writeEndElement(); // "subtasks"
      }
      if( !Times.isEmpty() )
      {
	Stream.writeStartElement( "time_slices" );
	foreach( TimeSlice Slice, Times )
	  Slice.write( Stream );
	Stream.writeEndElement(); // "time_slices"
      }
      Stream.writeEndElement(); // "task"
    }
    else qDebug() << "!!! Task::write: task without ID"; 
  } // write( QXmlStreamWriter& ) const
  void Task::write_blockers( QXmlStreamWriter& Stream )
  {
    if( !Blockers.empty() )
    {
      Stream.writeStartElement( "blockers" );
      Stream.writeAttribute( "task_id", Ident );
      foreach( Task* Blocker, Blockers )
	if( Blocker )
	{
	  if( Blocker->Ident )
	    Stream.writeTextElement( "blocker", Blocker->id() );
	}
      Stream.writeEndElement();
    }
    foreach( Task* SubTask, SubTasks )
      if( SubTask )
	SubTask->write_blockers( Stream );
  } // write_blockers( QXmlStreamWriter& )
  void Task::load( QXmlStreamReader& Stream, TasksFile& Tasks )
  {
    if( Stream.isStartElement() && Stream.name() == "task" )
    {
      Ident = Stream.attributes().value( "id" ).toString();
#ifdef PLANSPLANT_LASTID_IN_TASK
      if( LastID < Ident ) LastID = Ident;
#endif
      while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
      {
	QStringRef Tag = Stream.name();
	if( Tag == "name" ) Name = Stream.readElementText();
	else if( Tag == "supertask" )
	{
	  if( Task::ID SuperID = Stream.readElementText() )
	  {
	    Task* NewSuperTask = Tasks.find_task( SuperID );
	    if( !NewSuperTask )
	      qDebug() << "Supertask" << SuperID.str() << "not found!";
	    else
	      supertask( NewSuperTask );
	  }
	  else
	    qDebug() << "Invalid supertask ID for task" << Name;
	}
	else if( Tag == "priority" ) Priority = Stream.readElementText().toInt();
	else if( Tag == "description" ) Description = Stream.readElementText();
	else if( Tag == "comment" ) Comment = Stream.readElementText();
	else if( Tag == "plan_start" ) PlanStart = read_time( Stream );
	else if( Tag == "plan_finish" ) PlanFinish = read_time( Stream );
	else if( Tag == "completed" ) Completed = Stream.readElementText().toDouble();
	else if( Tag == "estimation" )
	{
	  int Units = 1;
	  if( Timeshop::Persistent::Loader::attribute( Stream.attributes(), "units", Units ) )
	    EstimationUnits = TimeUnits( Units );
	  Estimation = Stream.readElementText().toInt();
	}
	else if( Tag == "subtasks" )
	{
	  while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
	    if( Stream.isStartElement() && Stream.name() == "task" )
	      ( new Task( QString(), this ) )->load( Stream, Tasks );
	    else
	      Timeshop::Persistent::Loader::skip( Stream );
	}
	else if( Tag == "time_slices" )
	{
	  while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
	    if( Stream.isStartElement() && Stream.name() == "time_slice" )
	    {
	      TimeSlice Slice;
	      Slice.load( Stream );
	      Times.append( Slice );
	    }
	    else
	      Timeshop::Persistent::Loader::skip( Stream );
	}
	else
	  Timeshop::Persistent::Loader::skip( Stream );
      }
    }
  } // load( QXmlStreamReader& )

  void Task::add_blocker( Task& Upon )
  {
    if( !Blockers.contains( &Upon ) && !Upon.Dependents.contains( this ) )
    {
      Blockers.push_back( &Upon );
      Upon.Dependents.push_back( this );
    }
  } // add_blocker( Task&, Task& )
  void Task::remove_blocker( Task& Upon )
  {
    Blockers.removeAll( &Upon );
    Upon.Dependents.removeAll( this );
  } // remove_blocker( Task&, Task& ); 

  int random( int MaxVal )
  {
    int Result = rand() / ( double( RAND_MAX )+1 ) * ( MaxVal+1 );
    if( Result >= MaxVal )
      Result = MaxVal;
    return Result;
  } // random( int )
  TasksFile::TasksFile( const QString& FileName0 ) : ActiveTask( 0 ), FileName( FileName0 ), Modified( false ), Good( true ), LastID( 0 ), LastEventID( 0 ), Ident( 0 ), Sync( 0 )
  {
    if( !FileName.isEmpty() )
    {
      QFile File( FileName );
      if( File.open( QIODevice::ReadOnly ) )
      {
	qDebug() << "Loading tasks";
	Task::Map TasksMap;
	QXmlStreamReader Stream( &File );
	if( Timeshop::Persistent::Loader::next_subelement( Stream ) && Stream.name() == "plansplant" )
	{
	  while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
	  {
	    if( Stream.isStartElement() )
	    {
	      if( Stream.name() == "task" )
	      {
		Task* NewTask = new Task;
		Roots.push_back( NewTask );
		NewTask->load( Stream, *this );
		NewTask->add_to_map( TasksMap );
		qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
	      }
	      else if( Stream.name() == "blockers" || Stream.name() == "dependencies" )
	      {	
		qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
		if( Task::ID ForID = Stream.attributes().value( "task_id" ).toString() )
		{
		  if( Task* For = TasksMap[ ForID ] )
		    while( Timeshop::Persistent::Loader::next_subelement( Stream ) && Stream.name() == "blocker" )
		    {
		      qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
		      Task::ID UponID = Stream.readElementText();
		      if( Task* Upon = TasksMap[ UponID ] ) For->add_blocker( *Upon );
		      else qDebug() << "Task with ID" << UponID.str() << "(UponID) not found.";
		    }
		  else qDebug() << "Task with ID" << ForID.str() << "(ForID) not found.";
		}
		else qDebug() << "Found blockers tag without task id";
	      }
	      else if( Stream.name() == "id" )
		Ident = Stream.readElementText().toInt();
	      else if( Stream.name() == "uuid" )
		UUID = Stream.readElementText();
	      else if( Stream.name() == "synchronization" )
	      {
		QString Type = Stream.attributes().value( "type" ).toString();
		qDebug() << "Found" << Type << Stream.name() << "section for file with id" << Ident;
		if( Type == "server" )
		  Sync = new Server( *this );
		else if( Type == "client" )
		  Sync = new Node( *this );
		else qDebug() << "Found unknown synchronization type:" << Type << "!";
		if( Sync ) Sync->load( Stream );
		else Timeshop::Persistent::Loader::skip( Stream );
	      }
	      else if( Stream.name() == "last_event_id" )
		LastEventID = Stream.readElementText().toInt();
	      else Timeshop::Persistent::Loader::skip( Stream );
	    }
	    else Timeshop::Persistent::Loader::skip( Stream );
	  } // file elements loop
	  qDebug() << "after while" << Stream.tokenType() << Stream.name() << Stream.tokenString() << Stream.errorString();
	  Good = ( File.error() == QFile::NoError ) && !Stream.hasError();
	}
	else
	{
	  qDebug() << "No plansplant tag in the file.";
	  Good = false;
	}
	foreach( Task* T, Roots )
	  scan_tasks( *T );
      }
      else // Can't open file
	Good = false;
      if( Good )
      {	if( Sync ) Sync->start(); }
      else
	qDebug() << "Can't open file for reading.";
    }
#ifdef PLANSPLANT_RANDOM_TREE_GENERATOR
#warning Move sample generation to separate method (or function).
    else // FileName is empty.
    {
      // Prepare test model
      const int Total = 4*1024*1024;
      const int MaxRoots = 16;
      const int MaxChildren = MaxRoots;
      const int MaxBlockers = 16;
      Task::List All;
      qDebug() << "Start generating test file.";
      srand( time( 0 ) );
      int Children = random( MaxRoots );
      for( int I = 0; I < Children; I++ )
      {
	All.push_back( new Task( QString::number( I+1 ), 0, new_id() ) );
	Roots.push_back( All.back() );
      }
      for( int Index = 0; Index < All.size() && All.size() < Total; Index++ )
      {
	Children = random( MaxChildren );
	if( Children > Total-All.size() ) Children = Total-All.size();
	for( int I = 0; I < Children; I++ )
	  All.push_back( new Task( All[ Index ]->name() + "." + QString::number( I+1 ), All[ Index ], new_id() ) );
      }
      qDebug() << Total << "tasks generated." << "\nGenerating blockers (no checks for loops).";
      int TotalBlockers = 0;
      for( int Index = 0; Index < Total; Index++ )
      {
	int Blockers = random( MaxBlockers );
	Task& Obj = *All[ Index ];
	while( Obj.blockers().size() < Blockers )
	{
	  int Blocker = random( Total-1 );
	  if( Blocker != Index )
	    Obj.add_blocker( *All[ Blocker ] );
	}
	TotalBlockers += Obj.blockers().size();
      }
      qDebug() << TotalBlockers << "blockers. Avg:" << TotalBlockers / double( Total );
    } 
#endif // PLANSPLANT_RANDOM_TREE_GENERATOR
  } // TasksFile( const QString& )
  TasksFile::~TasksFile()
  {
    if( Sync )
    {
      delete( Sync );
      Sync = 0;
    }
    while( !Roots.empty() )
    {
      delete Roots.back();
      Roots.pop_back();
    }
    foreach( Task::Watcher* W, Watchers )
      W->file_closed( *this );
  } // ~TasksFile()
  Task* TasksFile::find_task( Task::ID ForID ) const
  {
    Task* Result = 0;
    if( ForID.valid() )
      for( int I = 0; I < Roots.size() && !Result; I++ )
	if( Roots[ I ]->id() == ForID )
	  Result = Roots[ I ];
	else
	  Result = Roots[ I ]->find_task( ForID );
    return Result;
  } // find_task( Task::ID ) const
  bool TasksFile::update_root_status( Task& NewRootTask )
  {
    bool Result = false;
    if( NewRootTask.supertask() )
      Result = Roots.removeAll( &NewRootTask ) > 0;
    else
      if( !Roots.contains( &NewRootTask ) )
      {
	Roots.push_back( &NewRootTask );
	Result = true;
      }
    return Result;
  } // task_to_roots( Task& )
  bool TasksFile::save( const QString& NewFileName )
  {
    bool Result = false;
    if( !NewFileName.isEmpty() ) FileName = NewFileName;
    if( !FileName.isEmpty() )
    {
      qDebug() << "Save tasks";
      QFile File( FileName );
      if( File.open( QIODevice::WriteOnly ) )
      {
	QXmlStreamWriter Stream( &File );
	Stream.setAutoFormatting( true );
	Stream.setAutoFormattingIndent( 2 );
	Stream.writeStartDocument();
	Stream.writeStartElement( "plansplant" );
	Stream.writeTextElement( "id", QString::number( id() ) );
	if( !uuid().isEmpty() ) Stream.writeTextElement( "uuid", uuid() );
	Stream.writeTextElement( "last_event_id", QString::number( LastEventID ) );
	if( Sync )
	{
	  Stream.writeStartElement( "synchronization" );
	  Sync->write( Stream );
	  Stream.writeEndElement();
	}
	foreach( Task* CurrTask,  Roots )
	  if( CurrTask )
	    CurrTask->write( Stream );
	foreach( Task* CurrTask,  Roots )
	  if( CurrTask )
	    CurrTask->write_blockers( Stream );
	Stream.writeEndElement();
	Stream.writeEndDocument();
      }
      Result = ( File.error() == QFile::NoError );
    }
    if( Result ) modified( false );
    return Result;
  } // save( const QString )
  void TasksFile::add_task( Task& NewTask, const Task::Watcher::Event* Ev )
  {
    if( !NewTask.id() )
      NewTask.id( new_id() );
    if( !NewTask.supertask() && !Roots.contains( &NewTask ) )
      Roots.push_back( &NewTask );
    if( Ev ) broadcast( *Ev );
    else broadcast( Task::Watcher::Events::TaskAdded( NewTask, new_event_id() ) );
    modified();
  } // add_task( Task& )
  void TasksFile::delete_task( Task& OldTask, const Task::Watcher::Event* Ev )
  { //! \todo Optimize subtasks remove, because on every delete we will search for the task in every model's tree. Maybe it will be faster to remove all subtasks in a list.
    if( !OldTask.supertask() )
      Roots.removeAll( &OldTask );
    while( !OldTask.subtasks().empty() )
      delete_task( *OldTask.subtasks().back() );
    if( Ev ) broadcast( *Ev );
    else broadcast( Task::Watcher::Events::TaskRemoved( OldTask, new_event_id() ) );
    modified();
    delete &OldTask; //! \todo Consider moving it to some sort of recycle bin.
  } // delete_task( Task& )
  void TasksFile::start_work( Task& ToStart, const QDateTime& When )
  {
    if( When.isValid() )
    {
      if( ActiveTask ) stop_work( When );
      ActiveTask = &ToStart;
      change_task( ToStart, new Task::Changes::Times( new Task::Changes::Times::Operations::Insert( Task::TimeSlice( When ) ) ) );
    }
  } // start_work( Task&, const QDateTime& )
  void TasksFile::stop_work( const QDateTime& When )
  {
    if( When.isValid() )
      if( ActiveTask )
      {
	change_task( *ActiveTask, new Task::Changes::Times( new Task::Changes::Times::Operations::Change( When, Task::Changes::Times::Operations::Change::Finish ) ) );
	ActiveTask = 0;
      }
  } // stop_work( const QDateTime& )
  void TasksFile::change_task( Task::ChangesList& Changes, const Task::Watcher::Event* Ev )
  {
    Task& Object = Changes.task();
    foreach( Task::Change* Ch, Changes.changes() )
      Ch->apply( Object, *this );
    if( Object.active() )
      ActiveTask = &Object;
    else if( ActiveTask == &Object )
      ActiveTask = 0;
    if( Ev ) broadcast( *Ev );
    else broadcast( Task::Watcher::Events::TaskChanged( &Changes, new_event_id() ) );
    modified();
  } // change_task( Task::ChangesList& )
  void TasksFile::change_task( Task& Object, Task::Change* Change, const Task::Watcher::Event* Ev )
  {
    if( Change )
    {
      Task::ChangesList Changes( Object, *Change );
      change_task( Changes, Ev );
    }
  } // change_task( Task& Object, Task::Change* Change )
  // Usual changes to a task.
  bool TasksFile::move_task( Task& Object, int From, int To, const Task::Watcher::Event* Ev )
  {
    bool Result = false;
    if( Task* Parent = Object.supertask() )
    {
      change_task( *Parent, new Task::Changes::SubTasks( *new Task::Changes::SubTasks::Move( Object, From, To ) ), Ev );
      Result = true;
    }
    else
    {
      if( From != To && From >= 0 && From < Roots.size() && To >= 0 && To < Roots.size() )
      {
	int Step = 1;
	if( To < From ) Step = -1;
	for( int I = From; I != To; I+=Step )
	  Roots.swap( I, I+Step );
	modified();
	Result = true;
	if( Ev ) broadcast( *Ev );
	else broadcast( Task::Watcher::Events::RootTaskMoved( Object, From, To, new_event_id() ) );
      }
    }
    return Result;
  } // move_task( Task&, int, int, const Task::Watcher::Event* )
  bool TasksFile::add_blocker( Task& To, Task& Block )
  {
    bool Result = false;
    if( !To.blockers().contains( &Block ) && !To.check_loop( Block ) )
    {
      change_task( To, new Task::Changes::Blockers( *new Task::Changes::Blockers::Operations::Add( Block ) ) );
      Result = true;
    }
    return Result;
  } // add_blocker( Task&, Task& )
  void TasksFile::remove_blocker( Task& From, Task& Block )
  { change_task( From, new Task::Changes::Blockers( *new Task::Changes::Blockers::Operations::Remove( Block ) ) ); }
  bool TasksFile::move_blocker( Task& For, Task& Blocker, int From, int To )
  {
    bool Result = false;
    if( From != To && For.blocker( From ) == &Blocker && To >= 0 && To < For.blockers().size() )
    {
      change_task( For, new Task::Changes::Blockers( *new Task::Changes::Blockers::Operations::Move( Blocker, From, To ) ) );
      Result = true;
    }
    return Result;
  } // move_blocker( Task&, Task&, int, int )
  bool TasksFile::change_blockers( Task& Object, const Task::List& NewBlockers, Task* SuperTask )
  {
    bool Result = false;
    if( !( ( SuperTask && SuperTask->check_loop( NewBlockers, &Object ) ) || ( !SuperTask && Object.check_loop( NewBlockers ) ) ) )
    {
      change_task( Object, new Task::Changes::Blockers( NewBlockers ) );
      Result = true;
    }
    return Result;
  } // change_blockers( Task&, const Task::List&, Task* )
  void TasksFile::add_watcher( Task::Watcher& NewWatcher )
  {
    if( !Watchers.contains( &NewWatcher ) )
    {
      Watchers.push_back( &NewWatcher );
      NewWatcher.add_informer( *this );
    }
  } // add_watcher( Task::Watcher& )
  void TasksFile::remove_watcher( Task::Watcher& OldWatcher )
  {
    if( Watchers.contains( &OldWatcher ) )
    {
      Watchers.removeAll( &OldWatcher );
      OldWatcher.remove_informer( *this );
    }
  } // remove_watcher( Task::Watcher& )
  void TasksFile::scan_tasks( Task& Obj )
  {
    if( Obj.id().origin() == id() && Obj.id().seq() > LastID ) LastID = Obj.id().seq();
    if( Obj.active() )
      ActiveTask = &Obj;
    foreach( Task* T, Obj.subtasks() )
      scan_tasks( *T );
  } // scan_tasks( Task& )
  void TasksFile::broadcast( const Task::Watcher::Event& Ev ) const
  {
    qDebug() << "Broadcating" << Ev.type() << Ev.id().str();
    foreach( Task::Watcher* W, Watchers )
      W->event( Ev );
  } // broadcast( const Task::Watcher::Event& ) const
} // PlansPlant
