// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QFile>
#include <QTextStream>
#include <QTextCodec>
#include <QTextDocument>
#include <QMessageBox>
#include <QApplication>
#include "plansplant/export.hpp"
namespace PlansPlant
{
  ExportFilter::~ExportFilter()
  {
    if( Out ) // Is it right to destroy this?
      Out->deleteLater();
  } // ~ExportFilter()
  bool ExportFilter::export_tasks( const Task::List& /*Tasks*/ ) { return false; } // export_tasks( const Task::List& )
  HTMLExportFilter::HTMLExportFilter( const QString& FileName ) : ExportFilter( new QFile( FileName ) )
  {
  } // HTMLExportFilter( const QString& )
  bool HTMLExportFilter::export_tasks( const Task::List& Tasks )
  {
    bool Result = false;
    if( Out && Out->open( QFile::WriteOnly ) )
    {
      QTextStream Str( Out );
      Str.setCodec( QTextCodec::codecForName( "UTF-8" ) );
      Str << "<HTML><HEAD><META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; Charset=utf-8\">"
	  << "<STYLE TYPE=\"text/css\">\n"
	  << ".toc_task_completed, .toc_plan_start, .toc_plan_time, .toc_estimate { padding-left: 1em; }\n"
	  << ".field_name { padding-right: 1em; font-weight: bold; }\n"
	  << "TABLE { width: 100%; }\n"
	  << ".even_task_row { background: silver; }"
	  << "</STYLE></HEAD>\n<BODY>\n"
	  << "<H1>" << QObject::tr( "Tasks tree" ) << "</H1>\n<OL>";
      foreach( Task* Cur, Tasks )
      {
	Str << "\n\t<LI>";
	toc( Str, *Cur );
      }
      Str << "</OL>\n<H1>" << QObject::tr( "Tasks details" ) << "</H1>";
      int I = 1;
      foreach( Task* Cur, Tasks )
	details( Str, *Cur, QString::number( I++ ) );
      Str << "\n</BODY></HTML>";
      Result = true;
    }
    else // Move this diagnostics out of here.
      QMessageBox::warning( 0, QApplication::applicationName(),
			    QObject::tr( "Can't open file for writing." ) );
    return Result;
  } // export_tasks( const Task::List& )
  void HTMLExportFilter::toc( QTextStream& Str, const Task& T )
  {
    Str << "<SPAN CLASS=\"toc_task_name\">" << Qt::escape( T.name() ) << "</SPAN>"
	<< "<SPAN CLASS=\"toc_task_completed\">" << T.completed()*100 << "%</SPAN>";
    if( T.plan_start().isValid() || T.plan_finish().isValid() )
    {
      Str << "<SPAN CLASS=\"toc_plan_time\">";
      if( T.plan_start().isValid() )
	Str << T.plan_start().toString( Qt::SystemLocaleShortDate );
      Str << "&nbsp;-&nbsp;";
      if( T.plan_finish().isValid() )
	Str << T.plan_finish().toString( Qt::SystemLocaleShortDate );
      Str << "</SPAN>";
    }
    if( T.estimation() > 0 )
      Str << "<SPAN CLASS=\"toc_estimate\">" << T.estimation() << " " << Qt::escape( Task::units_short_name( T.estimation_units() ) ) << "</SPAN>";
    if( !T.subtasks().empty() ) Str << "\n<OL>";
    foreach( const Task* SubTask, T.subtasks() )
    {
      Str << "\n\t<LI>";
      toc( Str, *SubTask );
    }
    if( !T.subtasks().empty() ) Str << "\n</OL>";
  } // toc( QTextStream&, const Task& )
  void HTMLExportFilter::details( QTextStream& Str, const Task& T, const QString& Prefix )
  {
    Str << "<H2>" << Prefix << ": " << Qt::escape( T.name() ) << "</H2>\n";
    if( T.completed() > 0 )
      Str << "<DIV><SPAN CLASS=\"field_name\">" << QObject::tr( "Complete" ) << ":</SPAN><SPAN>" << QString::number( T.completed()*100 ) << "%</SPAN></DIV>\n";
    if( T.plan_start().isValid() )
      Str << "<DIV><SPAN CLASS=\"field_name\">" << QObject::tr( "Start" ) << ":</SPAN><SPAN>" << T.plan_start().toString( Qt::SystemLocaleShortDate ) << "</SPAN></DIV>\n";
    if( T.plan_finish().isValid() )
      Str << "<DIV><SPAN CLASS=\"field_name\">" << QObject::tr( "Finish" ) << ":</SPAN><SPAN>" << T.plan_finish().toString( Qt::SystemLocaleShortDate ) << "</SPAN></DIV>\n";
    if( T.estimation() > 0 )
      Str << "<DIV><SPAN CLASS=\"field_name\">" << QObject::tr( "Estimated" ) << ":</SPAN><SPAN>" << T.estimation() << ' ' << Qt::escape( Task::units_short_name( T.estimation_units() ) ) << "</SPAN></DIV>\n";
    if( !T.description().isEmpty() )
      Str << "<DIV><SPAN CLASS=\"field_name\">" << QObject::tr( "Description" ) << ":</SPAN><SPAN>" << Qt::escape( T.description() ) << "</SPAN></DIV></DIV>\n";
    if( !T.comment().isEmpty() )
      Str << "<DIV><SPAN CLASS=\"field_name\">" << QObject::tr( "Comment" ) << ":</SPAN><SPAN>" << Qt::escape( T.comment() ) << "</SPAN></DIV></DIV>\n";
    if( !T.subtasks().empty() )
    {
      Str << "<SPAN CLASS=\"field_name\">" << QObject::tr( "Subtasks" ) << ":</SPAN><TABLE>";
      int I = 1;
      foreach( const Task* SubTask, T.subtasks() )
      {
	I++;
	table_row( Str, *SubTask, QString::number( I ), I%2 );
      }
      Str << "</TABLE>";
    }
    if( !T.blockers().empty() )
    {
      Str << "<SPAN CLASS=\"field_name\">" << QObject::tr( "Depends on" ) << ":</SPAN><TABLE>";
      int I = 1;
      foreach( const Task* Dep, T.blockers() )
      {
	I++;
	table_row( Str, *Dep, QString::number( I ), I%2 );
      }
      Str << "</TABLE>";
    }
    int I = 1;
    QString FixPref = Prefix;
    if( !FixPref.isEmpty() )
      FixPref += '.';
    foreach( const Task* SubTask, T.subtasks() )
      details( Str, *SubTask, FixPref + QString::number( I++ ) );
  } // details( QTextStream&, const Task& )
  void HTMLExportFilter::table_row( QTextStream& Str, const Task& T, const QString& Prefix, bool EvenRow )
  {
    Str << "<TR";
    if( EvenRow ) Str << " CLASS=even_task_row";
    Str << "><TD>" << Prefix << "<TD>" << Qt::escape( T.name() )  << "<TD>";
    if( T.completed() > 0 )
      Str << QString::number( T.completed()*100 ) << '%';
    Str << "<TD>";
    if( T.plan_start().isValid() )
      Str << T.plan_start().toString( Qt::SystemLocaleShortDate );
    Str << "<TD>";
    if( T.plan_finish().isValid() )
      Str << T.plan_finish().toString( Qt::SystemLocaleShortDate );
    Str << "<TD>";
    if( T.estimation() > 0 )
      Str << QString::number( T.estimation() ) + ' ' + Qt::escape( Task::units_short_name( T.estimation_units() ) );
    Str << '\n';
  } // table_row( QTextStream&, const Task&, const QString& );
} // PlansPlant
