// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QFile>
#include <QMap>
#include <QDebug>
#include <QMessageBox>
#include <QFileDialog>
#include <QToolBar>
#include <QMenuBar>
#include <QCloseEvent>
#include <timeshop.hpp>
#include "plansplant/widgets.hpp"

namespace PlansPlant
{
  TasksTreeModel::Carrier* TasksTreeModel::carrier_from_index( const QModelIndex& Index )
  { return static_cast<Carrier*>( Index.internalPointer() ); } // carrier_from_index( const QModelIndex& )
  Task* TasksTreeModel::task_from_index( const QModelIndex& Index )
  {
    Task* Result = 0;
    if( Index.isValid() )
      if( Carrier* Item = carrier_from_index( Index ) )
	Result = &Item->task();
    return Result;
  } // task_from_index( const QModelIndex& )
  TasksTreeModel::Carrier::Carrier( Task& Object0, Pointer Parent0, Relation Rel0 ) : Object( Object0 ), Rel( Rel0 ), Parent( 0 ), Populated( false )
  {
    parent( Parent0 );
    if( !has_subitems() )
      Populated = true; // Because nothing to populate.
  } // Carrier( Task&, Pointer, Relation )
  TasksTreeModel::Carrier::~Carrier()
  {
    parent( 0 ); // Remove from parent
    while( !SubItems.empty() )
      delete SubItems.back();
  } // ~Carrier()
  void TasksTreeModel::Carrier::parent( Pointer NewParent )
  {
    if( NewParent != Parent )
    {
      if( Parent )
	Parent->SubItems.removeAll( this );
      Parent = NewParent;
      if( Parent && !Parent->SubItems.contains( this ) )
      {
	if( Rel == Subtask && Parent->SubItems.size() >= Parent->task().subtasks().size() )
	  Parent->SubItems.insert( Parent->task().subtasks().size()-1, this );
	else
	  Parent->SubItems.push_back( this );
      }
    }
  } // parent( Pointer )
  bool TasksTreeModel::Carrier::has_subitems() const { return !( Object.subtasks().empty() && Object.needs().empty() ); }
  const TasksTreeModel::Carrier::List& TasksTreeModel::Carrier::populate()
  {
    if( !Populated )
    {
      Populated = true;
      foreach( Task* SubTask, Object.subtasks() )
	if( SubTask )
	  new Carrier( *SubTask, this );
      foreach( Task* Block, Object.needs() )
	if( Block )
	  new Carrier( *Block, this, Blocker );
    }
    return SubItems;
  } // populate()
  TasksTreeModel::Carrier::Pointer TasksTreeModel::Carrier::subitem( int Index ) const
  {
    Pointer Result = 0;
    if( Index >= 0 && Index < SubItems.size() )
      Result = SubItems[ Index ];
    return Result;
  } // subitem( int ) const
  bool TasksTreeModel::Carrier::move_subitem( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < SubItems.size() && To >= 0 && To < SubItems.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	{
	  qDebug() << "Swap items" << Index << "and" << Index+Inc;
	  SubItems.swap( Index, Index+Inc );
	}
	Result = true;
      }
    }
    return Result;
  } // move_subitem( int, int )

  TasksTreeModel::TasksTreeModel( const QString& FileName0, QObject* Parent ) : QAbstractItemModel( Parent ), FileName( FileName0 ), Modified( false ), Good( true )
  {
    if( !FileName.isEmpty() )
    {
      QFile File( FileName );
      if( File.open( QIODevice::ReadOnly ) )
      {
	qDebug() << "Loading tasks";
	Task::Map TasksMap;
	QXmlStreamReader Stream( &File );
	if( Timeshop::Persistent::Loader::next_subelement( Stream ) && Stream.name() == "plansplant" )
	  while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
	  {
	    if( Stream.isStartElement() )
	    {
	      if( Stream.name() == "task" )
	      {
		Task* NewTask = new Task;
		Roots.push_back( NewTask );
		NewTask->load( Stream );
		NewTask->add_to_map( TasksMap );
		qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
	      }
	      else if( Stream.name() == "dependencies" )
	      {	
		qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
		Task::ID ForID = 0;
		if( Timeshop::Persistent::Loader::attribute( Stream.attributes(), "task_id", ForID ) )
		{
		  if( Task* For = TasksMap[ ForID ] )
		    while( Timeshop::Persistent::Loader::next_subelement( Stream ) && Stream.name() == "blocker" )
		    {
		      qDebug() << Stream.tokenType() << Stream.name() << Stream.tokenString();
		      Task::ID UponID = Stream.readElementText().toInt();
		      if( Task* Upon = TasksMap[ UponID ] )
			For->add_dependency( *Upon );
		      else qDebug() << "Task with ID" << UponID << "(UponID) not found.";
		    }
		  else qDebug() << "Task with ID" << ForID << "(ForID) not found.";
		}
		else qDebug() << "Found dependencies tag without task id";
	      }
	      else Timeshop::Persistent::Loader::skip( Stream );
	    }
	    else Timeshop::Persistent::Loader::skip( Stream );
	  }
	qDebug() << "after while" << Stream.tokenType() << Stream.name() << Stream.tokenString() << Stream.errorString();
      }
#if 0
      else
      {
	// Prepare test model
	Task* NewTask = new Task( "1" );
	Roots.push_back( NewTask );
	Task* SubTask = new Task( "1.1", NewTask );
	new Task( "1.1.1", SubTask );
	new Task( "1.1.2", SubTask );
	SubTask = new Task( "1.2", NewTask );
	NewTask = new Task( "2" );
	Roots.push_back( NewTask );
	SubTask->add_dependency( *NewTask );
      }
#endif
      Good = ( File.error() == QFile::NoError );
    }
    foreach( Task* Root, Roots )
      if( Root )
	Carriers.push_back( new Carrier( *Root ) );
  } // TasksTreeModel( QObject* )
  TasksTreeModel::~TasksTreeModel()
  {
    while( !Carriers.empty() )
    {
      delete Carriers.back();
      Carriers.pop_back();
    }
    while( !Roots.empty() )
    {
      delete Roots.back();
      Roots.pop_back();
    }
  } // ~TasksTreeModel()
  bool TasksTreeModel::save( const QString& NewFileName )
  {
    bool Result = false;
    if( !NewFileName.isEmpty() ) FileName = NewFileName;
    if( !FileName.isEmpty() )
    {
      qDebug() << "Save tasks";
      QFile File( FileName );
      if( File.open( QIODevice::WriteOnly ) )
      {
	QXmlStreamWriter Stream( &File );
	Stream.setAutoFormatting( true );
	Stream.setAutoFormattingIndent( 2 );
	Stream.writeStartDocument();
	Stream.writeStartElement( "plansplant" );
	foreach( Task* CurrTask,  Roots )
	  if( CurrTask )
	    CurrTask->write( Stream );
	foreach( Task* CurrTask,  Roots )
	  if( CurrTask )
	    CurrTask->write_dependencies( Stream );
	Stream.writeEndElement();
	Stream.writeEndDocument();
      }
      Result = ( File.error() == QFile::NoError );
    }
    if( Result ) modified( false );
    return Result;
  } // save( const QString )
  int TasksTreeModel::columnCount( const QModelIndex& /*Parent*/ ) const { return TotalCols; }
  int TasksTreeModel::rowCount( const QModelIndex& Parent ) const
  {
    int Result = 0;
    if( Parent.isValid() )
    {
      if( Carrier* CurrItem = carrier_from_index( Parent ) ) //! \todo else report error
	Result = CurrItem->populate().size();
    }
    else
      Result = Carriers.size();
    return Result;
  } // rowCount( const QModelIndex& ) const
  QModelIndex TasksTreeModel::index( int Row, int Column, const QModelIndex& Parent ) const
  {
    QModelIndex Result;
    if( Parent.isValid() )
    {
      if( Carrier* ParItem = carrier_from_index( Parent ) ) //! \todo else report error
      {
	ParItem->populate();
	if( Carrier* Item = ParItem->subitem( Row ) )
	  Result = createIndex( Row, Column, Item );
      }
    }
    else
    {
      if( Row >= 0 && Row < Carriers.size() )
	Result = createIndex( Row, Column, Carriers[ Row ] );
    }
    return Result;
  } // index( int Row, int Column, const QModelIndex& Parent ) const
  QModelIndex TasksTreeModel::index( Carrier& ForItem ) const
  {
    QModelIndex Result;
    if( Carrier* Parent = ForItem.parent() )
      Result = createIndex( Parent->subitems().indexOf( &ForItem ), 0, &ForItem );
    else
      Result = createIndex( Carriers.indexOf( &ForItem ), 0, &ForItem );
    return Result;
  } // index( Carrier& ) 
 QModelIndex TasksTreeModel::parent( const QModelIndex& Index ) const
  {
    QModelIndex Result;
    if( Index.isValid() )
      if( Carrier* Item = carrier_from_index( Index ) ) //! \todo else report error
	if( Carrier* Parent = Item->parent() )
	  Result = index( *Parent );
    return Result;
  } // parent( const QModelIndex& ) const
  QVariant TasksTreeModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    if( Index.isValid() )
      if( Carrier* CurrItem = carrier_from_index( Index ) ) //! \todo else report error
      {
	Task& CurrTask = CurrItem->task();
	if( Role == Qt::DisplayRole )
	{
	  switch( Index.column() )
	  {
	  case NameCol: Result = CurrTask.name(); break;
	  case CompletedCol: Result = QString::number( CurrTask.completed() * 100 ) + '%'; break;
	  case PlanStartCol:
	    if( CurrTask.plan_start().isValid() )
	      Result = CurrTask.plan_start();
	    break;
	  case PlanFinishCol:
	    if( CurrTask.plan_finish().isValid() )
	      Result = CurrTask.plan_finish();
	    break;
	  case EstimationCol:
	    if( CurrTask.estimation() > 0 )
	    {
	      QString EstStr = QString::number( CurrTask.estimation() ) + ' ';
	      switch( CurrTask.estimation_units() )
	      {
	      case Task::Seconds: EstStr += tr( "s" ); break;
	      case Task::Minutes: EstStr += tr( "m" ); break;
	      case Task::Hours: EstStr += tr( "h" ); break;
	      case Task::WorkDays: EstStr += tr( "wd" ); break;
	      case Task::Days: EstStr += tr( "d" ); break;
	      case Task::WorkWeeks: EstStr += tr( "ww" ); break;
	      case Task::Weeks: EstStr += tr( "w" ); break;
	      case Task::Months: EstStr += tr( "M" ); break;
	      case Task::Quarters: EstStr += tr( "Q" ); break;
	      case Task::Years: EstStr += tr( "Y" ); break;
	      default: break;
	      }
	      Result = EstStr;
	    }
	    break;
	  default: break;
	  }
	}
	else if( Role == Qt::TextAlignmentRole )
	{ if( Index.column() == CompletedCol ) Result = int( Qt::AlignRight | Qt::AlignVCenter ); }
	else if( Role == Qt::DecorationRole )
	{ if( Index.column() == NameCol ) Result = QIcon( is_blocker( Index ) ? ":/images/dependency.svg" : ":/images/task.svg" ); } //! \todo Select from the list of preloaded icons
#ifdef Q_WS_MAEMO_5
	else if( Role == Qt::BackgroundRole )
	{ //! \todo This don't change the background color, but it seems that it speeds up scrolling.
	  if( CurrItem->relation() == Carrier::Blocker )
	  {
	    if( CurrTask.completed() < 1 )
	      Result = QBrush( QColor( 160, 64, 0 ) );
	    else
	      Result = QBrush( QColor( 0, 128, 0 ) );
	  }
	  else
	    Result = QBrush( QColor( 0, 0, 0 ) );
	}
#else
	else if( Role == Qt::ForegroundRole ) //! \todo On Maemo this cause very high load when scrolling.
	{
	  if( CurrItem->relation() == Carrier::Blocker )
	  {
	    if( CurrTask.completed() < 1 )
	      Result = QBrush( QColor( 160, 120, 0 ) );
	    else
	      Result = QBrush( QColor( 0, 200, 0 ) );
	  }
	}
#endif // !Q_WS_MAEMO_5
	else if( Role == Qt::CheckStateRole )
	{
	  if( Index.column() == CompletedMarkCol )
	  {
	    if( CurrTask.completed() >= 1 )
	      Result = Qt::Checked;
	    else
	    { if( !CurrTask.blocked() ) Result = Qt::Unchecked; }
	  }
	}
#ifdef Q_WS_MAEMO_5
	else if( Role == Qt::SizeHintRole )
	  Result = QSize( 300, 42 );
#endif
      }
    return Result;
  } // data( const QModelIndex&, int ) const
  QVariant TasksTreeModel::headerData( int Section, Qt::Orientation Orient, int Role ) const
  {
    QVariant Result;
    if( Orient == Qt::Horizontal && Role == Qt::DisplayRole )
    {
      switch( Section )
      {
      case NameCol: Result = tr( "Name" ); break;
      case CompletedCol: Result = tr( "Completed" ); break;
      case PlanStartCol: Result = tr( "Start" ); break;
      case PlanFinishCol: Result = tr( "Finish" ); break;
      case EstimationCol: Result = tr( "Estimation" ); break;
      default: break;
      }
    }
    return( Result );
  } // headerData( int, Qt::Orientation, int )
  void TasksTreeModel::add_root_task( Task& NewRoot )
  {
    if( !Roots.contains( &NewRoot ) )
    {
      int NextRow = rowCount( QModelIndex() );
      beginInsertRows( QModelIndex(), NextRow, NextRow );
      Roots.push_back( &NewRoot );
      modified();
      Carriers.push_back( new Carrier( NewRoot ) );
      endInsertRows();
    }
  } // add_root_task( Task& )
  void TasksTreeModel::remove_from_root_tasks( Task& OldRoot )
  { //! \todo Maybe check that OldRoot has supertask.
    for( int Index = Roots.indexOf( &OldRoot ); Index >= 0; Index = Roots.indexOf( &OldRoot, Index ) )
    {
      beginRemoveRows( QModelIndex(), Index, Index );
      Carriers.removeAt( Index );
      Roots.removeAt( Index );
      modified();
      endRemoveRows();
    }
  } // remove_from_root_tasks( Task& )
  class TreeIterator
  {
  public:
    TreeIterator( TasksTreeModel& Model0 ) : Model( Model0 ), Current( Model0.index( 0, 0 ) ) {}
    TreeIterator( TasksTreeModel& Model0, const QModelIndex& Start ) : Model( Model0 ), Current( Start ) {}
    TreeIterator& operator++();
    TreeIterator& operator--();
    const QModelIndex& index() const { return Current; }
    const QModelIndex& operator*() const { return Current; }
    operator bool() const { return Current.isValid(); }
  protected:
    TasksTreeModel& Model;
    QModelIndex Current;
  }; // TreeIterator
  TreeIterator& TreeIterator::operator++()
  {
    QModelIndex Next = Current.child( 0, 0 ); // Search the model in-depth.
    if( !Next.isValid() )
    {
      Next = Current.sibling( Current.row()+1, 0 );
      while( !Next.isValid() && Current.isValid() ) // Try to find next sibling on any level.
      {
	Current = Current.parent();
	if( Current.isValid() )
	  Next = Current.sibling( Current.row()+1, 0 );
      }
    }
    Current = Next;
    return *this;
  } // operator++()
  TreeIterator& TreeIterator::operator--()
  {
    QModelIndex Prev;
    if( Current.row() > 0 ) Prev = Current.sibling( Current.row()-1, 0 ); // Try to step backward.
    if( !Prev.isValid() ) Prev = Current.parent(); // Try to step up.
    Current = Prev;
    return *this;
  } // operator--()
  bool TasksTreeModel::move_task( const QModelIndex& Index, int From, int To )
  {
    bool Result = false;
    qDebug() << "Move task from" << From << "to" << To;
    int Rows = rowCount( Index );
    if( From >= 0 && From < Rows && To >= 0 && To < Rows && From != To )
    {
      int FixedTo = (To > From) ? To+1 : To;
      if( beginMoveRows( Index, From, From, Index, FixedTo ) )
      {
	if( Carrier* Car = carrier_from_index( Index ) )
	{
	  Task& T = Car->task();
	  int SubTasksNum = T.subtasks().size();
	  
	  if( From > 0 && From < SubTasksNum && To >= 0 && To < SubTasksNum )
	    Result = T.move_subtask( From, To );
	  else
	    Result = T.move_dependency( From-SubTasksNum, To-SubTasksNum );
	  modified();
	  Car->move_subitem( From, To );
	  endMoveRows();
	  qDebug() << "Rows moved";
	  for( TreeIterator It( *this ); It.index().isValid(); ++It )
	    if( Carrier* TestCar = carrier_from_index( It.index() ) )
	      if( TestCar != Car && &TestCar->task() == &T && beginMoveRows( It.index(), From, From, It.index(), FixedTo ) )
	      {
		TestCar->move_subitem( From, To );
		endMoveRows();
	      }
	}
	else if( !Index.isValid() ) // It's a root item
	{
	  Task* TmpTask = Roots[ From ];
	  Carrier* TmpCar = Carriers[ From ];
	  int Step = To > From ? 1 : -1;
	  for( int I = From; I != To; I += Step )
	  {
	    Roots[ I ] = Roots[ I+Step ];
	    Carriers[ I ] = Carriers[ I+Step ];
	  }
	  Roots[ To ] = TmpTask;
	  Carriers[ To ] = TmpCar;
	  endMoveRows();
	  Result = true;
	  modified();
	}
      }
      else
	qDebug() << "Can\'t move" << From << '-' << From << "to" << To;
    }
    return Result;
  } // move_task( const QModelIndex&, int, int )
  void TasksTreeModel::add_task( Task& NewTask )
  {
    modified();
    Task* SuperTask = NewTask.supertask();
    if( !SuperTask )
      add_root_task( NewTask );
    else
      for( TreeIterator It( *this ); It.index().isValid(); ++It )
	if( Carrier* Car = carrier_from_index( It.index() ) )
	  if( &Car->task() == SuperTask )
	  {
	    int Row = SuperTask->subtasks().size()-1; // We've updated it before.
	    beginInsertRows( It.index(), Row, Row );
	    new Carrier( NewTask, Car );
	    endInsertRows();
	  }
  } // add_task( Task& )
  void TasksTreeModel::delete_task( Task& OldTask )
  {
    Task* SuperTask = OldTask.supertask();
    if( !SuperTask )
      remove_from_root_tasks( OldTask );
    for( TreeIterator It( *this ); It; ++It )
      if( Carrier* Car = carrier_from_index( *It ) )
	if( &Car->task() == &OldTask )
	{
	  int Row = It.index().row();
	  beginRemoveRows( parent( *It ), Row, Row );
	  --It;
	  delete Car;
	  endRemoveRows();
	}
    delete &OldTask;
    modified();
  } // delete_task( Task& )
  bool TasksTreeModel::change_parent( Task* ForTask, Task* NewParent )
  {
    bool Result = false;
    if( ForTask )
    {
      Task* OldParent = ForTask->supertask();
      if( OldParent != NewParent )
      {
	if( NewParent && NewParent->check_loop( *ForTask ) )
	  QMessageBox::warning( 0, tr( "PlansPlant" ),
				tr( "Can't set \"" ) + NewParent->name() + tr( "\"\nas a supertask for \"" ) + ForTask->name() + tr( "\":\nit will create a loop." ) );
	else
	{
	  ForTask->supertask( NewParent );
	  modified();
	  if( !OldParent ) remove_from_root_tasks( *ForTask );
	  else if( !NewParent ) add_root_task( *ForTask );
	  for( TreeIterator It( *this ); It.index().isValid(); ++It )
	    if( Carrier* Car = carrier_from_index( It.index() ) )
	    {
	      if( &Car->task() == NewParent && NewParent )
	      {
		int Row = NewParent->subtasks().size()-1; // We've updated it before.
		beginInsertRows( It.index(), Row, Row );
		new Carrier( *ForTask, Car );
		endInsertRows();
	      }
	      else if( &Car->task() == ForTask && Car->relation() == Carrier::Subtask && Car->parent() && &Car->parent()->task() == OldParent )
	      {
		beginRemoveRows( parent( It.index() ), It.index().row(), It.index().row() );
		--It;
		delete Car;
		endRemoveRows();
	      }
	    }
	  Result = true;
	}
      }
    }
    return Result;
  } // change_parent( Task*, Task* )
  bool TasksTreeModel::is_subtask( const QModelIndex& Index ) const
  {
    bool Result = false;
    if( Carrier* Item = carrier_from_index( Index ) )
      Result = ( Item->relation() == Carrier::Subtask );
    return Result;
  } // is_subtask( const QModelIndex& ) const
  void TasksTreeModel::add_blocker( Task& To, Task& Block )
  {
    if( !To.needs().contains( &Block ) )
    {
      if( To.check_loop( Block ) )
	QMessageBox::warning( 0, tr( "PlansPlant" ), tr( "Can't add \"" ) + Block.name() + tr( "\"\nas a dependency for \"" ) + To.name() + tr( "\":\nit will create a loop." ) );
      else
      {
	To.add_dependency( Block );
	modified();
	for( TreeIterator It( *this ); It.index().isValid(); ++It )
	  if( Carrier* Car = carrier_from_index( It.index() ) )
	  {
	    if( &Car->task() == &To )
	    {
	      qDebug() << "Add blocker" << Block.name() << "to" << To.name();
	      int Row = Car->subitems().size();
	      beginInsertRows( It.index(), Row, Row );
	      new Carrier( Block, Car, Carrier::Blocker );
	      endInsertRows();
	    }
	  }
      }
    }
  } // add_blocker( Task&, Task& )
  bool TasksTreeModel::is_blocker( const QModelIndex& Index ) const
  {
    bool Result = false;
    if( Carrier* Item = carrier_from_index( Index ) )
      Result = ( Item->relation() == Carrier::Blocker );
    return Result;
  } // is_blocker( const QModelIndex& ) const
  void TasksTreeModel::remove_blocker( const QModelIndex& Index )
  {
    if( Carrier* Item = carrier_from_index( Index ) )
      if( Carrier* Parent = Item->parent() )
	if( Item->relation() == Carrier::Blocker )
	  remove_blocker( Parent->task(), Item->task() );
  } // remove_blocker( const QModelIndex& )
  void TasksTreeModel::remove_blocker( Task& From, Task& Block )
  {
    if( From.needs().contains( &Block ) )
    {
      for( TreeIterator It( *this ); It.index().isValid(); ++It )
	if( Carrier* Car = carrier_from_index( It.index() ) )
	  if( &Car->task() == &Block && Car->relation() == Carrier::Blocker && Car->parent() && &Car->parent()->task() == &From )
	  {
	    int Row = It.index().row();
	    beginRemoveRows( parent( It.index() ), Row, Row );
	    --It;
	    delete Car;
	    endRemoveRows();
	  }
      From.remove_dependency( Block );
      modified();
    }
    else
      qDebug() << "Blocker not found in the task.";
  } // remove_blocker( Task&, Task& );

  TaskSelectDialog::TaskSelectDialog( TasksTreeModel& Model0, QWidget* Parent, Task* CurrentTask0, const QString& Title )
    : QDialog( Parent ), CurrentTask( CurrentTask0 ), TasksTree( 0 ), Buttons( 0 )
  {
    setWindowTitle( Title );
    TasksTree = new TasksTreeWidget( this, &Model0 );
    QVBoxLayout* Layout = new QVBoxLayout( this );
    Layout->addWidget( TasksTree );
    Buttons = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( CurrentTask );
    connect( Buttons, SIGNAL( accepted() ), SLOT( accept() ) );
    connect( Buttons, SIGNAL( rejected() ), SLOT( reject() ) );
    Layout->addWidget( Buttons );
    setLayout( Layout );
    connect( TasksTree->selectionModel(), SIGNAL( currentChanged( const QModelIndex&, const QModelIndex& ) ), SLOT( current_changed( const QModelIndex& ) ) );
    TasksTree->select_task( CurrentTask );
  } // TaskSelectDialog( TasksTreeModel&, QWidget*, Task* )
  void TaskSelectDialog::accept()
  {
    CurrentTask = TasksTree->selected_task();
    if( CurrentTask ) QDialog::accept();
    else QMessageBox::warning( this, tr( "PlansPlant" ), tr( "Select a task before." ) );
  } // accept()
  void TaskSelectDialog::current_changed( const QModelIndex& New )
  {
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( TasksTree->tasks() && TasksTreeModel::task_from_index( New ) );
  } // current_changed( const QModelIndex&, const QModelIndex& )

  ParentSelectDialog::ParentSelectDialog( TasksTreeModel& Model0, QWidget* Parent, Task* ForTask0, Task* SuperTask0 )
    : TaskSelectDialog( Model0, Parent, SuperTask0, tr( "Select new supertask for " ) + ( ForTask0 ? ( '\"' + ForTask0->name() + '\"' ) : tr( "new task" ) ) + '\"' ),
      ForTask( ForTask0 )
  {
    current_changed( TasksTree->currentIndex() );
  } // ParentSelectDialog( TasksTreeModel&, Task&, QWidget*, Task* )
  void ParentSelectDialog::current_changed( const QModelIndex& New )
  {
    bool Enable = false;
    if( Task* NewParent = TasksTreeModel::task_from_index( New ) )
      Enable = !ForTask || !NewParent->check_loop( *ForTask );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( Enable );
  } // current_changed( const QModelIndex& )
  BlockerSelectDialog::BlockerSelectDialog( TasksTreeModel& Model0, QWidget* Parent, Task* ForTask0, Task* Blocker0 )
    : TaskSelectDialog( Model0, Parent, Blocker0, tr( "Select new dependency for " ) + ( ForTask0 ? ( '\"' + ForTask0->name() + '\"' ) : tr( "new task" ) ) + '\"' ),
      ForTask( ForTask0 )
  {
    current_changed( TasksTree->currentIndex() );
  } // BlockerSelectDialog( TasksTreeModel&, Task&, QWidget*, Task* )
  void BlockerSelectDialog::current_changed( const QModelIndex& New )
  {
    bool Enable = false;
    if( Task* NewBlocker = TasksTreeModel::task_from_index( New ) )
      Enable = !ForTask || !ForTask->check_loop( *NewBlocker );
    Buttons->button( QDialogButtonBox::Ok )->setEnabled( Enable );
  } // current_changed( const QModelIndex& )

  TaskDialog::TaskDialog( TasksTreeModel& Model0, Task* Object0, QWidget* Parent ) : QDialog( Parent ), Model( Model0 ), Object( Object0 ), SuperTask( 0 )
  {
    UI.setupUi( this );
    UI.BlockersLabel->hide();
    UI.BlockersList->hide();
    UI.BlockerAddButton->hide();
    UI.BlockerRemoveButton->hide();
    UI.BlockerEditButton->hide();
    UI.EstimatedUnits->addItem( tr( "seconds" ), Task::Seconds );
    UI.EstimatedUnits->addItem( tr( "minutes" ), Task::Minutes );
    UI.EstimatedUnits->addItem( tr( "hours" ), Task::Hours );
    UI.EstimatedUnits->addItem( tr( "workdays" ), Task::WorkDays );
    UI.EstimatedUnits->addItem( tr( "days" ), Task::Days );
    UI.EstimatedUnits->addItem( tr( "workweeks" ), Task::WorkWeeks );
    UI.EstimatedUnits->addItem( tr( "weeks" ), Task::Weeks );
    UI.EstimatedUnits->addItem( tr( "months" ), Task::Months );
    UI.EstimatedUnits->addItem( tr( "quarters" ), Task::Quarters );
    UI.EstimatedUnits->addItem( tr( "years" ), Task::Years );
#ifdef Q_WS_MAEMO_5
    UI.Description->setMinimumHeight( 128 );
    UI.Comment->setMinimumHeight( 128 );
    QMargins Margins = UI.PlanGroup->layout()->contentsMargins();
    qDebug() << "Top margin:" << Margins;
    Margins.setTop( 20 );
    UI.PlanGroup->layout()->setContentsMargins( Margins );
#endif
    
    if( Object )
    {
      UI.Name->setText( Object->name() );
      UI.Description->setPlainText( Object->description() );
      UI.Comment->setPlainText( Object->comment() );
      if( Object->plan_start().isValid() )
      {
	UI.HasStart->setChecked( true );
	UI.PlanStart->setDateTime( Object->plan_start() );
      }
      if( Object->plan_finish().isValid() )
      {
	UI.HasFinish->setChecked( true );
	UI.PlanFinish->setDateTime( Object->plan_finish() );
      }
      UI.Complete->setValue( Object->completed() * 100 );
      UI.Estimated->setText( QString::number( Object->estimation() ) );
      UI.EstimatedUnits->setCurrentIndex( UI.EstimatedUnits->findData( Object->estimation_units() ) );
#if 0
      UI.BlockersList->setModel( &Model );
      UI.BlockersList->setRootIndex( Model.index( *Object ) );
      connect( UI.BlockerAddButton, SIGNAL( clicked() ), SLOT( add_blocker() ) );
      connect( UI.BlockerRemoveButton, SIGNAL( clicked() ), SLOT( remove_blocker() ) );
#endif
      if( Object->supertask() ) supertask( Object->supertask() );
    }
    connect( UI.SelectParent, SIGNAL( clicked( bool ) ), SLOT( select_supertask() ) );
    connect( UI.PlanStartNow, SIGNAL( clicked( bool ) ), SLOT( plan_start_now() ) );
    connect( UI.PlanFinishNow, SIGNAL( clicked( bool ) ), SLOT( plan_finish_now() ) );
  } // TaskDialog
  void TaskDialog::accept()
  {
    int CurrUnits = UI.EstimatedUnits->currentIndex();
    if( CurrUnits >= 0 ) CurrUnits = UI.EstimatedUnits->itemData( CurrUnits ).toInt();
    else
    {
      qDebug() << "Wrong estimation time units:" << CurrUnits;
      CurrUnits = 1;
    }
    QDateTime PlanStart;
    if( UI.HasStart->isChecked() )
      PlanStart = UI.PlanStart->dateTime();
    QDateTime PlanFinish;
    if( UI.HasFinish->isChecked() )
      PlanFinish = UI.PlanFinish->dateTime();
    if( !Object )
    {
      Object = new Task( UI.Name->text(), UI.HasParent->isChecked() ? SuperTask : 0 );
      Object->description( UI.Description->toPlainText() );
      Object->comment( UI.Comment->toPlainText() );
      Object->plan_start( PlanStart );
      Object->plan_finish( PlanFinish );
      Object->completed( UI.Complete->value() / 100.0 );
      Object->estimation( UI.Estimated->text().toInt() );
      Object->estimation_units( Task::TimeUnits( CurrUnits ) );
      Model.add_task( *Object );
    }
    else
    {
      bool Modified = false;
      if( Object->name() != UI.Name->text() )			   { Object->name( UI.Name->text() ); Modified = true; }
      if( Object->description() != UI.Description->toPlainText() ) { Object->description( UI.Description->toPlainText() ); Modified = true; }
      if( Object->comment() != UI.Comment->toPlainText() )	   { Object->comment( UI.Comment->toPlainText() ); Modified = true; }
      if( Object->plan_start() != PlanStart )	   		   { Object->plan_start( PlanStart ); Modified = true; }
      if( Object->plan_finish() != PlanFinish )	   		   { Object->plan_finish( PlanFinish ); Modified = true; }
      double Comp = UI.Complete->value() / 100.0;
      if( Object->completed() != Comp )				   { Object->completed( Comp ); Modified = true; }
      Time Est = UI.Estimated->text().toInt();
      if( Object->estimation() != Est )				   { Object->estimation( Est ); Modified = true; }
      if( Object->estimation_units() != CurrUnits )
      { Object->estimation_units( Task::TimeUnits( CurrUnits ) ); Modified = true; }
      if( !UI.HasParent->isChecked() ) SuperTask = 0;
      if( Object->supertask() != SuperTask )
	Model.change_parent( Object, SuperTask );
      if( Modified ) Model.modified();
    }
    QDialog::accept();
  } // accept()
  void TaskDialog::supertask( Task* NewSuperTask )
  {
    if( NewSuperTask != SuperTask )
    {
      SuperTask = NewSuperTask;
      if( SuperTask )
      {
	UI.SelectParent->setText( SuperTask->name() );
	UI.HasParent->setChecked( true );
      }
      else
      {
	UI.SelectParent->setText( tr( "none" ) );
	UI.HasParent->setChecked( false );
      }
    }
  } // supertask( Task* )
  void TaskDialog::select_supertask()
  {
    ParentSelectDialog Dlg( Model, this, Object, SuperTask );
    if( Dlg.exec() )
      supertask( Dlg.current_task() );
  } // select_supertask()
  void TaskDialog::add_blocker()
  {
#if 0
    BlockerSelectDialog Dlg( Model, this, Object );
    if( Dlg.exec() )
    {
      if( Object )
      {
	if( Dlg.current_task() )
	{
	  Object->add_dependency( *Dlg.current_task() );
	  qDebug() << "Add blocker" << Dlg.current_task()->name() << "to" << Object->name();
	}
	else
	  qDebug() << "No task selected. Don't add anything.";
      }
      else
	QMessageBox::warning( this, "PlansPlant", "Save current task before adding dependencies." );
    }
#endif
  } // add_blocker()
  void TaskDialog::remove_blocker()
  {
    QMessageBox::warning( this, "PlansPlant", "Remove blocker isn't implemented yet." );
  } // remove_blocker()
  void TaskDialog::plan_start_now() { UI.PlanStart->setDateTime( QDateTime::currentDateTime() ); }
  void TaskDialog::plan_finish_now() { UI.PlanFinish->setDateTime( QDateTime::currentDateTime() ); }

  TasksTreeWidget::TasksTreeWidget( QWidget* Parent, TasksTreeModel* Model0 ) : QTreeView( Parent )
  {
    if( Model0 ) tasks( Model0 );
    setUniformRowHeights( true );
    setContextMenuPolicy( Qt::ActionsContextMenu );
    QAction* Act = new QAction( tr( "Add &task" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( add_task() ) );
    addAction( Act );
    Act = new QAction( tr( "&Open task" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( open_task() ) );
    addAction( Act );
    Act = new QAction( tr( "&Delete task" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( delete_task() ) );
    addAction( Act );
    Act = new QAction( this );
    Act->setSeparator( true );
    addAction( Act );
    Act = new QAction( tr( "Add de&pendency" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( add_blocker() ) );
    addAction( Act );
    Act = new QAction( tr( "Remo&ve dependency" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( remove_blocker() ) );
    addAction( Act );
    Act = new QAction( this );
    Act->setSeparator( true );
    addAction( Act );
    Act = new QAction( tr( "Move &up" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( move_up() ) );
    addAction( Act );
    Act = new QAction( tr( "Move do&wn" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( move_down() ) );
    addAction( Act );
    connect( this, SIGNAL( clicked( const QModelIndex& ) ), SLOT( item_clicked( const QModelIndex& ) ) );
  } // TasksTreeWidget( TasksTreeModel&, QWidget* )
  QSize TasksTreeWidget::sizeHint() const { return QSize( 800, 480 ); }
  void TasksTreeWidget::tasks( TasksTreeModel* Tasks )
  {
    setModel( Tasks );
    setColumnWidth( TasksTreeModel::NameCol, 400 );
#ifdef Q_WS_MAEMO_5
    setColumnWidth( TasksTreeModel::CompletedMarkCol, 55 );
    setColumnWidth( TasksTreeModel::CompletedCol, 70 );
    setColumnWidth( TasksTreeModel::PlanStartCol, 128 );
    setColumnWidth( TasksTreeModel::PlanFinishCol, 128 );
#else
    setColumnWidth( TasksTreeModel::CompletedMarkCol, 24 );
    setColumnWidth( TasksTreeModel::CompletedCol, 43 );
    setAlternatingRowColors( true );
#endif
  } // tasks( TasksTreeModel* )
  bool TasksTreeWidget::select_task( const Task* NewCurrent )
  {
    bool Result = false;
    QModelIndex Index;
    if( NewCurrent )
    {
      if( TasksTreeModel* Tasks = tasks() )
	for( TreeIterator It( *Tasks ); It && !Index.isValid(); ++It )
	  if( Tasks->is_subtask( *It ) && TasksTreeModel::task_from_index( *It ) == NewCurrent )
	    Index = *It;
    }
    else
      Result = true;
    setCurrentIndex( Index );
    return Result;
  } // select_task( const Task* )
  void TasksTreeWidget::add_task()
  {
    if( TasksTreeModel* Tasks = tasks() )
    {
      TaskDialog Dlg( *Tasks, 0, this );
      if( Task* Sel = selected_task() )
	Dlg.supertask( Sel );
      Dlg.exec();
    }
  } // add_task()
  void TasksTreeWidget::open_task()
  {
    if( Task* Sel = selected_task() )
      if( TasksTreeModel* Tasks = tasks() )
      {
	TaskDialog Dlg( *Tasks, Sel, this );
	Dlg.exec();
      }
  } // open_task()
  void TasksTreeWidget::delete_task()
  {
    if( Task* Sel = selected_task() )
      if( QMessageBox::question( this, tr( "PlansPlant" ), tr( "Are you shure that you want to delete task\n\"" ) + Sel->name() + tr( "\"?\nThere's NO WAY to restore it." ),
				 QMessageBox::Yes | QMessageBox::No ) == QMessageBox::Yes )
	if( TasksTreeModel* Tasks = tasks() )
	  Tasks->delete_task( *Sel ); //! \todo else Error message
  } // delete_task()
  void TasksTreeWidget::add_blocker()
  {
    if( Task* Object = selected_task() )
      if( TasksTreeModel* Tasks = tasks() )
      {
	BlockerSelectDialog Dlg( *Tasks, this, Object, Object->needs().empty() ? 0 : Object->needs().back() );
	if( Dlg.exec() )
	{
	  if( Dlg.current_task() )
	  {
	    Tasks->add_blocker( *Object, *Dlg.current_task() );
	    qDebug() << "Add blocker" << Dlg.current_task()->name() << "to" << Object->name();
	  }
	  else
	    qDebug() << "No task selected. Don't add anything.";
	}
      }
  } // add_blocker()
  void TasksTreeWidget::remove_blocker()
  {
    QModelIndex Current = currentIndex();
    if( Current.isValid() )
    {
      if( TasksTreeModel* Tasks = tasks() )
      {
	if( Tasks->is_blocker( Current ) )
	  Tasks->remove_blocker( Current );
	else
	  qDebug() << "Can't remove blocker: it's not a blocker.";
      }
    }
    else
      qDebug() << "Can't remove blocker: nothing selected.";
  } // remove_blocker()
  void TasksTreeWidget::move_up()
  {
    QModelIndex Index = currentIndex();
    if( Index.isValid() )
      if( TasksTreeModel* Tasks = tasks() )
	Tasks->move_task( Index.parent(), Index.row(), Index.row()-1 );
  } // move_up()
  void TasksTreeWidget::move_down()
  {
    QModelIndex Index = currentIndex();
    if( Index.isValid() )
      if( TasksTreeModel* Tasks = tasks() )
	Tasks->move_task( Index.parent(), Index.row(), Index.row()+1 );
  } // move_down()
  void TasksTreeWidget::item_clicked( const QModelIndex& Index )
  {
    if( Index.column() == TasksTreeModel::CompletedMarkCol )
      if( Task* Cur = TasksTreeModel::task_from_index( Index ) )
	if( !Cur->blocked() && Cur->completed() < 1
	    && QMessageBox::question( this, tr( "Plans Plant" ),
				      tr( "Complete task\n\"" ) + Cur->name() + tr( "\"?" ),
				      QMessageBox::Yes | QMessageBox::No ) == QMessageBox::Yes )
	{
	  Cur->completed( 1 ); //! \todo Do this not directly but through the model.
	  if( TasksTreeModel* Tasks = tasks() )
	    Tasks->modified();
	}
  } // item_clicked( const QModelIndex& )

  MainWindow::MainWindow( QWidget* Parent, Qt::WindowFlags Flags ) : QMainWindow( Parent, Flags ), Model( 0 ), Tree( 0 )
  {
    setWindowTitle( tr( "Plans Plant" ) );
    setWindowIcon( QIcon( ":/images/task.svg" ) );
    Tree = new TasksTreeWidget();
    setCentralWidget( Tree );
    QToolBar* Tools = new QToolBar( tr( "Standard" ), this );
    QMenuBar* MainMenu = new QMenuBar( this );
#ifdef Q_WS_MAEMO_5
    QMenuBar* Menu = MainMenu;
#else
    QMenu* Menu = new QMenu( tr( "&File" ) );
    MainMenu->addMenu( Menu );
#endif
    QAction* Act = new QAction( QIcon::fromTheme( "document-new", QIcon( ":/images/document-new.svg" ) ), tr( "New file" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( new_file() ) );
    Tools->addAction( Act );
    Menu->addAction( Act );
    Act = new QAction( QIcon::fromTheme( "document-open", QIcon( ":/images/document-open.svg" ) ), tr( "Open file..." ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( open_file() ) );
    Tools->addAction( Act );
    Menu->addAction( Act );
#ifndef Q_WS_MAEMO_5
    Menu->addSeparator();
#endif
    Act = new QAction( QIcon::fromTheme( "document-save", QIcon( ":/images/document-save.svg" ) ), tr( "Save file" ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( save_file() ) );
    Tools->addAction( Act );
    Menu->addAction( Act );
    Act = new QAction( tr( "Save file as..." ), this );
    connect( Act, SIGNAL( triggered() ), SLOT( save_file_as() ) );
    Menu->addAction( Act );
    Tools->addSeparator();
#ifndef Q_WS_MAEMO_5
    Menu->addSeparator();
    Act = new QAction( tr( "E&xit" ), this );
    Act->setMenuRole( QAction::AboutRole );
    connect( Act, SIGNAL( triggered() ), SLOT( close() ) );
    Menu->addAction( Act );
    Menu = new QMenu( tr( "&Edit" ) );
    MainMenu->addMenu( Menu );
#endif
    Act = new QAction( QIcon( ":/images/task-new.svg" ), tr( "New task..." ), this );
    Tree->connect( Act, SIGNAL( triggered() ), SLOT( add_task() ) );
    Tools->addAction( Act );
    Menu->addAction( Act );
    Act = new QAction( QIcon( ":/images/task-edit.svg" ), tr( "Open task..." ), this );
    Tree->connect( Act, SIGNAL( triggered() ), SLOT( open_task() ) );
    Tools->addAction( Act );
    Menu->addAction( Act );
#ifndef Q_WS_MAEMO_5
    Menu->addSeparator();
#endif
    Act = new QAction( QIcon( ":/images/dependency-new.svg" ), tr( "Add dependency..." ), this );
    Tree->connect( Act, SIGNAL( triggered() ), SLOT( add_blocker() ) );
    Tools->addAction( Act );
    Menu->addAction( Act );
    Act = new QAction( tr( "Remove dependency..." ), this );
    Tree->connect( Act, SIGNAL( triggered() ), SLOT( remove_blocker() ) );
    Menu->addAction( Act );
    Tools->addSeparator();
#ifndef Q_WS_MAEMO_5
    Menu->addSeparator();
#endif
    Act = new QAction( QIcon::fromTheme( "go-up", QIcon( ":/images/go-up.svg" ) ), tr( "Move up" ), this );
    Tree->connect( Act, SIGNAL( triggered() ), SLOT( move_up() ) );
    Tools->addAction( Act );
    Menu->addAction( Act );
    Act = new QAction( QIcon::fromTheme( "go-down", QIcon( ":/images/go-down.svg" ) ), tr( "Move down" ), this );
    Tree->connect( Act, SIGNAL( triggered() ), SLOT( move_down() ) );
    Tools->addAction( Act );
    Menu->addAction( Act );
#ifndef Q_WS_MAEMO_5
    Menu = new QMenu( tr( "&Help" ) );
    MainMenu->addMenu( Menu );
#endif
    Act = new QAction( tr( "About..." ), this );
    Act->setMenuRole( QAction::AboutRole );
    connect( Act, SIGNAL( triggered() ), SLOT( about() ) );
    Menu->addAction( Act );
    Act = new QAction( tr( "About Qt..." ), this );
    Act->setMenuRole( QAction::AboutQtRole );
    connect( Act, SIGNAL( triggered() ), qApp, SLOT( aboutQt() ) );
    Menu->addAction( Act );
    addToolBar( Tools );
    setMenuBar( MainMenu );
  } // MainWindow( QWidget*, Qt::WindowFlags )
  MainWindow::~MainWindow()
  {
    close_file( true );
  } // ~MainWindow()
  bool MainWindow::close_file( bool Force )
  {
    if( Model )
    {
      if( Model->is_modified() )
      {
	QMessageBox::StandardButtons Buttons = QMessageBox::Save | QMessageBox::Discard;
	if( !Force ) Buttons |= QMessageBox::Cancel;
	switch( QMessageBox::question( this, QString(), tr( "The file is modified. Do you want to save it?" ), Buttons ) )
	{
	case QMessageBox::Save:
	  if( !save_file() && !Force ) break;
	case QMessageBox::Discard:
	  Model->deleteLater();
	  Model = 0;
	default:
	  break;
	}
      }
      else
      {
	Model->deleteLater();
	Model = 0;
      }
    }
    return !Model;
  } // close_file( bool )
  bool MainWindow::open_file()
  {
    bool Result = false;
    QString NewName = QFileDialog::getOpenFileName( this, tr( "Open file:" ), Model->file_name(),
						    tr( "Plans Plant files" ) + " (*.plansplant);;" + tr( "All files" ) + " (*)" );
    if( !NewName.isEmpty() )
      Result = open_file( NewName );
    return Result;
  } // open_file()
  bool MainWindow::open_file( const QString& NewFileName )
  {
    bool Result = false;
    TasksTreeModel* NewModel = new TasksTreeModel( NewFileName );
    if( !Model || ( NewModel->good() && close_file() ) )
    {
      Model = NewModel;
      Tree->tasks( Model );
    }
    else
      delete NewModel;
    return Result;
  } // open_file( const QString& )
  bool MainWindow::save_file()
  {
    bool Result = false;
    if( Model )
    {
      if( Model->file_name().isEmpty() )
	Result = save_file_as();
      else
	Result = Model->save();
    }
    return Result;
  } // save_file()
  bool MainWindow::save_file_as()
  {
    bool Result = false;
    if( Model )
    {
      QString NewName = QFileDialog::getSaveFileName( this, tr( "Select file for saving:" ), Model->file_name(),
						      tr( "Plans Plant files" ) + " (*.plansplant);;" + tr( "All files" ) + " (*)" );
      if( !NewName.isEmpty() )
	Result = Model->save( NewName );
    }
    return Result;
  } // save_file_as()
  bool MainWindow::new_file()
  {
    bool Result = false;
    if( close_file() )
    {
      Model = new TasksTreeModel;
      Tree->tasks( Model );
      Result = true;
    }
    return Result;
  } // new_file()
  void MainWindow::about()
  {
    QMessageBox::about( this, tr( "About Plans Plant..." ),
			tr( "Plans Plant. Version " ) + QApplication::applicationVersion() +
			trUtf8( "\n© Copyright 2010 Nick Slobodsky (Николай Слободской)\nWeb: http://plansplant.garage.maemo.org"
				"\n\nThis is a simple planning application: tasks tree with planned start/due times."
				"\nSome icons and icons' parts are from the GNOME environment, © 2007–2009 Jakub \'jimmac\' Steiner" ) );
  } // about()
  void MainWindow::closeEvent( QCloseEvent* Close )
  {
    if( close_file() )
      QMainWindow::closeEvent( Close );
    else
      Close->ignore();
  } // closeEvent( QCloseEvent* )
} // PlansPlant
