// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <timeshop.hpp>
#include "plansplant/tasks.hpp"
#include <QDebug>

namespace PlansPlant
{
  Task::Task( const QString& Name0, Pointer SuperTask0, ID ID0 ) : Ident( ID0 ), SuperTask( SuperTask0 ), Name( Name0 ), Completed( 0 ), Estimation( 0 ), EstimationUnits( Seconds )
  {
    if( LastID < Ident ) LastID = Ident;
    if( SuperTask && SuperTask->SubTasks.indexOf( this ) < 0 )
      SuperTask->SubTasks.push_back( this );
  } // Task( const QString&, Pointer* )
  Task::~Task()
  {
    while( !SubTasks.empty() )
      delete SubTasks.back(); // Will be removed from this list in the subtask's destructor.
    supertask( 0 );
  } // ~Task()
  
  void Task::supertask( Task* NewTask )
  {
    if( NewTask != SuperTask )
    {
      if( SuperTask )
	SuperTask->SubTasks.removeAll( this );
      SuperTask = NewTask;
      if( SuperTask )
	SuperTask->SubTasks.push_back( this );
    }
  } // supertask( Task* )

  bool Task::check_loop( Task& NewChild )
  {
    bool Result = this == &NewChild;
#if 0
    // First try to find new child in parents (we'll do it in the main loop)
    for( Task* Parent = this; Parent && !Result; Parent = Parent->supertask() )
      if( Parent == &NewChild )
	Result = true;
#endif
    if( !Result ) // Search for it in the tasks that depends on this
    { //! \todo Optimize the search
      Task::List Deps;
      Deps.push_back( this );
      for( int I = 0; I < Deps.size() && !Result; I++ )
	if( Task* Dep = Deps[ I ] )
	{
	  if( Task* Parent = Dep->supertask() )
	  { //! \todo Move to the main "for"
	    if( Parent == &NewChild )
	      Result = true;
	    else if( !Deps.contains( Parent ) )
	      Deps.push_back( Parent );
	  }
	  for( int J = 0; J < Dep->dependents().size() && !Result; J++ )
	    if( Task* NewDep = Dep->dependent( J ) )
	    {
	      if( NewDep == &NewChild )
		Result = true;
	      else if( !Deps.contains( NewDep ) )
		Deps.push_back( NewDep );
	    }
	}
    }
    return Result;
  } // check_loop( Task& )
  bool Task::move_subtask( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < SubTasks.size() && To >= 0 && To < SubTasks.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	{
	  qDebug() << "Swap tasks" << Index << "and" << Index+Inc;
	  SubTasks.swap( Index, Index+Inc );
	}
	Result = true;
      }
    }
    return Result;
  } // move_subtask( int, int )
  bool Task::move_dependency( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < Needs.size() && To >= 0 && To < Needs.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	{
	  qDebug() << "Swap tasks" << Index << "and" << Index+Inc;
	  Needs.swap( Index, Index+Inc );
	}
	Result = true;
      }
    }
    return Result;
  } // move_dependency( int, int )
  bool Task::blocked() const
  {
    bool Result = false;
    foreach( Task* SubTask, subtasks() )
      if( SubTask && SubTask->completed() < 1 )
	Result = true;
    if( !Result )
      foreach( Task* Dep, needs() )
	if( Dep && Dep->completed() < 1 )
	  Result = true;
    return Result;
  } // blocked() const
  void Task::add_to_map( Map& TasksMap )
  {
    if( !Ident ) Ident = new_id();
    TasksMap[ Ident ] = this;
    foreach( Task* SubTask, SubTasks )
      if( SubTask )
	SubTask->add_to_map( TasksMap );
  } // add_to_map( Map& )
  void Task::write( QXmlStreamWriter& Stream )
  {
    Stream.writeStartElement( "task" );
    if( !Ident ) Ident = new_id();
    Stream.writeAttribute( "id", QString::number( Ident ) );
    // We don't write a supertask (parent), because we save tasks in parent's subtasks list.
    if( !Name.isEmpty() ) Stream.writeTextElement( "name", Name );
    if( !Description.isEmpty() ) Stream.writeTextElement( "description", Description );
    if( !Comment.isEmpty() ) Stream.writeTextElement( "comment", Comment );
    if( PlanStart.isValid() ) Stream.writeTextElement( "plan_start", PlanStart.toUTC().toString( "yyyy-MM-dd hh:mm:ss" ) );
    if( PlanFinish.isValid() ) Stream.writeTextElement( "plan_finish", PlanFinish.toUTC().toString( "yyyy-MM-dd hh:mm:ss" ) );
    if( Completed > 0 ) Stream.writeTextElement( "completed", QString::number( Completed ) );
    if( Estimation > 0 )
    {
      Stream.writeStartElement( "estimation" );
      Stream.writeAttribute( "units", QString::number( EstimationUnits ) );
      Stream.writeCharacters( QString::number( Estimation ) );
      Stream.writeEndElement();
    }
    if( !SubTasks.isEmpty() )
    {
      Stream.writeStartElement( "subtasks" );
      foreach( Task* SubTask, SubTasks )
	if( SubTask )
	  SubTask->write( Stream );
      Stream.writeEndElement();
    }
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const
  void Task::write_dependencies( QXmlStreamWriter& Stream )
  {
    if( !Needs.empty() )
    {
      Stream.writeStartElement( "dependencies" );
      Stream.writeAttribute( "task_id", QString::number( Ident ) );
      foreach( Task* Need, Needs )
	if( Need )
	{
	  if( !Need->Ident ) Need->Ident = new_id();
	  Stream.writeTextElement( "blocker", QString::number( Need->id() ) );
	}
      Stream.writeEndElement();
    }
    foreach( Task* SubTask, SubTasks )
      if( SubTask )
	SubTask->write_dependencies( Stream );
  } // write_dependencies( QXmlStreamWriter& )
  QDateTime read_time( QXmlStreamReader& Stream ) // no "fromUTCString" method in QDateTime.
  {
    QDateTime Result = QDateTime::fromString( Stream.readElementText(), "yyyy-MM-dd hh:mm:ss" );
    Result.setTimeSpec( Qt::UTC );
    return Result.toLocalTime();
  } // read_time( QXmlStreamReader& Stream )
  void Task::load( QXmlStreamReader& Stream )
  {
    if( Stream.isStartElement() && Stream.name() == "task" )
    {
      Timeshop::Persistent::Loader::attribute( Stream.attributes(), "id", Ident );
      if( LastID < Ident ) LastID = Ident;
      while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
      {
	QStringRef Tag = Stream.name();
	if( Tag == "name" ) Name = Stream.readElementText();
	else if( Tag == "description" ) Description = Stream.readElementText();
	else if( Tag == "comment" ) Comment = Stream.readElementText();
	else if( Tag == "plan_start" ) PlanStart = read_time( Stream );
	else if( Tag == "plan_finish" ) PlanFinish = read_time( Stream );
	else if( Tag == "completed" ) Completed = Stream.readElementText().toDouble();
	else if( Tag == "estimation" )
	{
	  int Units = 1;
	  if( Timeshop::Persistent::Loader::attribute( Stream.attributes(), "units", Units ) )
	    EstimationUnits = TimeUnits( Units );
	  Estimation = Stream.readElementText().toInt();
	}
	else if( Tag == "subtasks" )
	{
	  while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
	    if( Stream.isStartElement() && Stream.name() == "task" )
	      ( new Task( QString(), this ) )->load( Stream );
	    else
	      Timeshop::Persistent::Loader::skip( Stream );
	}
	else
	  Timeshop::Persistent::Loader::skip( Stream );
      }
    }
  } // load( QXmlStreamReader& )

  void Task::add_dependency( Task& Upon )
  {
    if( !Needs.contains( &Upon ) && !Upon.Dependents.contains( this ) )
    {
      Needs.push_back( &Upon );
      Upon.Dependents.push_back( this );
    }
  } // add_dependency( Task&, Task& )
  void Task::remove_dependency( Task& Upon )
  {
    Needs.removeAll( &Upon );
    Upon.Dependents.removeAll( this );
  } // remove_dependency( Task&, Task& ); 
  Task::ID Task::LastID = 0;
} // PlansPlant
