//
// blaupunktprotocol.cpp
//
// Copyright 2016 by John Pietrzak (jpietrzak8@gmail.com)
//
// This file is part of Pierogi.
//
// Pierogi is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// Pierogi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//

#include "blaupunktprotocol.h"

#include "pirinfraredled.h"

#include <QString>
#include <QMutex>

extern bool commandInFlight;
extern QMutex commandIFMutex;

BlaupunktProtocol::BlaupunktProtocol(
  QObject *guiObject,
  unsigned int index)
  : PIRProtocol(guiObject, index, 121440, FALSE),
    biphaseUnit(528),
    buffer(0)
{
  setCarrierFrequency(30300);
}


void BlaupunktProtocol::startSendingCommand(
  unsigned int threadableID,
  PIRKeyName command)
{
  // Check if this command is meant for us:
  if (threadableID != id) return;

  clearRepeatFlag();

  KeycodeCollection::const_iterator i = keycodes.find(command);

  // Sanity check, make sure command exists first:
  if (i == keycodes.end())
  {
    QMutexLocker cifLocker(&commandIFMutex);
    commandInFlight = false;
    emit errorMessage("Key not defined in this keyset.");
    return;
  }

  // Construct the object that communicates with the device driver:
  PIRInfraredLED led(carrierFrequency, dutyCycle);

  connect(
    &led,
    SIGNAL(errorMessage(QString)),
    this,
    SIGNAL(errorMessage(QString)));

  int repeatCount = 0;
  int commandDuration = 0;
  while (repeatCount < MAX_REPEAT_COUNT)
  {
    // Now, construct a Blaupunkt protocol command string.

    // First, construct the command prefix:
    commandDuration += pushPrefixBits(led);

    // Next, the key-command portion:
    commandDuration += pushKeyCommandBits((*i).second, led);

    // Finally, the device portion:
    commandDuration += pushDeviceBits(led);

    // Clear out the buffer, if necessary:
    if (buffer)
    {
      led.addSingle(buffer);
      commandDuration += buffer;

      // probably unnecessary cleanup of buffer:
      buffer = 0;
      bufferContainsSpace = false;
      bufferContainsPulse = false;
    }

    // Now, tell the device to send the whole command:
    if (!led.sendCommandToDevice())
    {
      break;
    }

    // Sleep for an amount of time.
    sleepUntilRepeat(commandDuration);

    // Have we been told to stop yet?
    if (checkRepeatFlag())
    {
      break;
    }

    ++repeatCount;
  }

  QMutexLocker cifLocker(&commandIFMutex);
  commandInFlight = false;
}


int BlaupunktProtocol::pushPrefixBits(
  PIRInfraredLED &led)
{
  int duration = 0;

  // First, we have a single-duration on pulse, followed by a 5-duration
  // off pulse:
  led.addSingle(biphaseUnit);
  led.addSingle(biphaseUnit * 5);
  duration += biphaseUnit * 6;

  // Next, the value "1" is repeated ten times:
  int i = 0;
  while (i < 10)
  {
    pushBit(1, led);
    ++i;
  }

  // Next, 39 units of off, 1 unit on, and five units off.

  // Flush any on-pulse out of buffer:
  if (buffer && bufferContainsPulse)
  {
    led.addSingle(buffer);
    duration += buffer;
  }

  // Push the 39-unit off pulse onto the array:
  int offSize = 39;
  if (buffer && bufferContainsSpace)
  {
    offSize += buffer;
  }

  // Clean up buffer:
  buffer = 0;
  bufferContainsSpace = false;
  bufferContainsPulse = false;

  led.addSingle(biphaseUnit * offSize);

  // Push a single-unit on pulse:
  led.addSingle(biphaseUnit);

  // Push a 5-unit off pulse:
  led.addSingle(biphaseUnit * 5);

  // Finally, Push the value "1":
  pushBit(1, led);

  return duration;
}


int BlaupunktProtocol::pushKeyCommandBits(
  const PIRKeyBits &pkb,
  PIRInfraredLED &led)
{
  int duration = 0;

  // Just push all the bits:
  CommandSequence::const_iterator i = pkb.firstCode.begin();
  while (i != pkb.firstCode.end())
  {
    duration += pushBit(*i, led);
    ++i;
  }

  return duration;
}


int BlaupunktProtocol::pushDeviceBits(
  PIRInfraredLED &led)
{
  int duration = 0;

  // Only two bits, so just push them:
  CommandSequence::const_iterator i = preData.begin();

  // Should an error message be generated if data is missing?
  if (i != preData.end())
  {
    // First bit:
    duration += pushBit(*i, led);
    ++i;

    // Second bit:
    if (i != preData.end())
    {
      duration += pushBit(*i, led);
    }
  }

  return duration;
}


int BlaupunktProtocol::pushBit(
  bool bitValue,
  PIRInfraredLED &led)
{
  unsigned int duration = 0;
  // Reverse of RC5, Blaupunkt encodes a "0" by using a space followed by
  // a pulse, and a "1" by using a pulse followed by a space.

  if (!bitValue)
  {
    // We've got a "0".  First add a space, then a pulse.
    if (bufferContainsSpace)
    {
      // Merge our space with the previous space, and send them to
      // the device.
      led.addSingle(buffer + biphaseUnit);
      duration += (buffer + biphaseUnit);
      buffer = 0;
      bufferContainsSpace = false;
    }
    else
    {
      if (bufferContainsPulse)
      {
        // Flush the buffer:
        led.addSingle(buffer);
        duration += buffer;
        buffer = 0;
        bufferContainsPulse = false;
      }
      // Add a space:
      led.addSingle(biphaseUnit);
      duration += biphaseUnit;
    }

    // Put a pulse into the buffer to wait.
    buffer = biphaseUnit;
    bufferContainsPulse = true;
  }
  else
  {
    // We've got a "1".  First add a pulse, then a space.
    if (bufferContainsPulse)
    {
      // Merge our pulse with the previous one, and send them to the device:
      led.addSingle(buffer + biphaseUnit);
      duration += (buffer + biphaseUnit);
      buffer = 0;
      bufferContainsPulse = false;
    }
    else
    {
      if (bufferContainsSpace)
      {
        // Flush out the buffer:
        led.addSingle(buffer);
        duration += buffer;
        buffer = 0;
        bufferContainsSpace = false;
      }

      // Add a pulse:
      led.addSingle(biphaseUnit);
      duration += biphaseUnit;
    }

    // Put a space into the buffer to wait:
    buffer = biphaseUnit;
    bufferContainsSpace = true;
  }

  return duration;
}
