/*
  TOE - Tims own texteditor.
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "BaseDialogs.h"

#include <Lum/Base/String.h>

#include <Lum/Dlg/Msg.h>

#include <Lum/Model/String.h>

#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/Combo.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/String.h>
#include <Lum/Table.h>

#include <Lum/Panel.h>
#include <Lum/Text.h>

#include "Database.h"

#include "Data.h"

class ValueDialog : public Lum::Dialog
{
private:
  bool                  result;
  Lum::Model::ActionRef okAction;
  std::wstring          caption;
  size_t                size;

public:
  Lum::Model::StringRef value;

public:
  ValueDialog(const std::wstring& caption, size_t size, const std::wstring& value)
  : result(false),
    okAction(new Lum::Model::Action),
    caption(caption),
    size(size),
    value(new Lum::Model::String)
  {
    Observe(okAction);
    Observe(GetClosedAction());

    this->value->Set(value);
  }

  virtual void PreInit()
  {
    Lum::Text        *text;

    text=new Lum::Text;
    text->SetFlex(true,false);
    text->SetText(caption);

    SetMain(Lum::VPanel::Create(true,false)
            ->Add(text)
            ->AddSpace()
            ->Add(Lum::String::Create(value,size,true,false))
            ->AddSpace()
            ->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false)));

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      result=true;
      Exit();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else {
      Dialog::Resync(model,msg);
    }
  }

  bool GetResult() const
  {
    return result;
  }
};

bool GetStringValue(Lum::OS::Window* parent,
                    const std::wstring& caption,
                    size_t width,
                    std::wstring& value)
{
  ValueDialog *dialog;

  dialog=new ValueDialog(caption,width,value);
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  if (!dialog->GetResult()) {
    delete dialog;
    return false;
  }

  value=dialog->value->Get();
  delete dialog;
  return true;
}

class LinkNameTypeDialog : public Lum::Dialog
{
private:
  Lum::Model::StringTableRef model;
  Lum::Model::ActionRef      okAction;
  Lum::Model::StringRef      nameModel;
  Lum::Model::StringRef      typeModel;
  bool                       success;

public:
  LinkNameTypeDialog()
  : model(new Lum::Model::StringTable()),
    okAction(new Lum::Model::Action),
    nameModel(new Lum::Model::String),
    typeModel(new Lum::Model::String),
    success(false)
  {
    Observe(okAction);
    Observe(GetClosedAction());
    Observe(nameModel);
    Observe(typeModel);

    nameModel->Set(L"");

    std::list<Base::TypePtr>::const_iterator type;

    type=Base::Type::typeList.begin();
    while (type!=Base::Type::typeList.end()) {
      if (type==Base::Type::typeList.begin()) {
        typeModel->Set((*type)->GetName());
      }

      model->Append((*type)->GetName());

      ++type;
    }
  }

  virtual void PreInit()
  {
    Lum::Combo       *combo;
    Lum::Label       *label;
    Lum::Panel       *vPanel;
    Lum::Text        *text;

    vPanel=Lum::VPanel::Create(true,false);

    text=new Lum::Text;
    text->SetFlex(true,false);
    text->SetText(L"Create link entry...");

    vPanel->Add(text);

    vPanel->AddSpace();

    label=Lum::Label::Create(true,false);
    label->AddLabel(new Lum::Text(L"Name:"),Lum::String::Create(nameModel,true,false));

    combo=new Lum::TextCombo();
    combo->SetFlex(true,false);
    combo->SetModel(typeModel);
    combo->SetTableModel(model);
    label->AddLabel(new Lum::Text(L"Type:"),combo);

    vPanel->Add(label);
    vPanel->AddSpace();
    vPanel->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false));

    SetMain(vPanel);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      success=true;
      Exit();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else if (model==nameModel || model==typeModel) {
      if (!nameModel->IsNull() && !typeModel->IsNull() && !nameModel->Empty() && !typeModel->Empty()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }
    }

    Dialog::Resync(model,msg);
  }

  bool ClosedSuccess()
  {
    return success;
  }

  void GetResult(std::wstring& name, std::wstring& type) const
  {
    name=nameModel->Get();
    type=typeModel->Get();
  }
};

bool GetLinkNameType(Lum::OS::Window* parent,
                     std::wstring& linkName,
                     std::wstring& linkType)
{
  LinkNameTypeDialog *dialog;

  dialog=new LinkNameTypeDialog();
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  if (dialog->ClosedSuccess()) {
    dialog->GetResult(linkName,linkType);

    delete dialog;
    return true;
  }
  else {
    delete dialog;
    return false;
  }
}

class DataEntry
{
public:
  Base::DataPtr data;

public:
  DataEntry(Base::DataPtr data)
  : data(data)
  {
    // no code
  }
};

typedef Lum::Model::StdTable<DataEntry>      DataEntryModel;
typedef Lum::Base::Reference<DataEntryModel> DataEntryModelRef;

class DataEntryModelPainter : public Lum::StringCellPainter
{
public:
  std::wstring GetCellData() const
  {
    const DataEntry entry=dynamic_cast<const DataEntryModel*>(GetModel())->GetEntry(GetRow());

    return entry.data->GetDisplayText();
  }
};

class DataEntryComparator : public DataEntryModel::Comparator
{
public:
  bool operator()(const DataEntryModel::ElementType& a,
                  const DataEntryModel::ElementType& b,
                  size_t column, bool down) const
  {
    return !a.data->IsGreater(b.data);
  }
};

class DataLinkDialog : public Lum::Dialog
{
private:
  Lum::Model::StringTableRef         model;
  Lum::Model::ActionRef              okAction;
  Lum::Model::StringRef              nameModel;
  Lum::Model::SizeTRef               typeModel;
  DataEntryModelRef                  dataModel;
  Lum::Model::SingleLineSelectionRef selectionModel;
  bool                               success;

public:
  DataLinkDialog()
  : model(new Lum::Model::StringTable()),
    okAction(new Lum::Model::Action),
    nameModel(new Lum::Model::String),
    typeModel(new Lum::Model::SizeT),
    dataModel(new DataEntryModel(new DataEntryComparator())),
    selectionModel(new Lum::Model::SingleLineSelection()),
    success(false)
  {
    Observe(okAction);
    Observe(GetClosedAction());
    Observe(nameModel);
    Observe(typeModel);
    Observe(selectionModel);

    nameModel->Set(L"");

    std::list<Base::TypePtr>::const_iterator type;
    type=Base::Type::typeList.begin();
    while (type!=Base::Type::typeList.end()) {
      model->Append((*type)->GetName());

      ++type;
    }

    typeModel->Set(1);
  }

  virtual void PreInit()
  {
    Lum::Combo       *combo;
    Lum::Label       *label;
    Lum::Table       *dataTable;
    Lum::String      *string;
    Lum::Panel       *vPanel;
    Lum::Text        *text;

    vPanel=Lum::VPanel::Create(true,true);

    text=new Lum::Text;
    text->SetFlex(true,false);
    text->SetText(L"Select dataset to link to...");

    vPanel->Add(text);
    vPanel->AddSpace();

    label=Lum::Label::Create(true,false);

    string=new Lum::String();
    string->SetFlex(true,false);
    string->SetModel(nameModel);
    label->AddLabel(new Lum::Text(L"Link name:"),string);

    combo=new Lum::IndexCombo();
    combo->SetFlex(true,false);
    combo->SetModel(typeModel);
    combo->SetTableModel(model);
    label->AddLabel(new Lum::Text(L"Type:"),combo);

    vPanel->Add(label);
    vPanel->AddSpace();

    dataTable=new Lum::Table();
    dataTable->SetFlex(true,true);
    dataTable->SetModel(dataModel);
    dataTable->SetPainter(new DataEntryModelPainter());
    dataTable->SetSelection(selectionModel);
    vPanel->Add(dataTable);

    vPanel->AddSpace();
    vPanel->Add(Lum::ButtonRow::CreateOkCancel(okAction,GetClosedAction(),true,false));

    SetMain(vPanel);

    Dialog::PreInit();
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==okAction && okAction->IsFinished()) {
      success=true;
      Exit();
    }
    else if (model==GetClosedAction() && GetClosedAction()->IsFinished()) {
      Exit();
    }
    else if (model==nameModel || model==typeModel || model==selectionModel) {
      if (!nameModel->IsNull() && !typeModel->IsNull() && !selectionModel->IsNull() &&
          !nameModel->Empty() && selectionModel->HasSelection()) {
        okAction->Enable();
      }
      else {
        okAction->Disable();
      }

      if (model==typeModel) {
        dataModel->Off();
        dataModel->Clear();
        if (!model->IsNull()) {
          ::Base::DataConstIterator iter;
          ::Base::TypePtr           type;

          type=::Base::Type::GetTypeByIndex(typeModel->Get()-1);

          iter=database->database.begin();
          while (iter!=database->database.end()) {
            if ((*iter)->GetType()==type) {
              dataModel->Append(DataEntry((*iter)));
            }
            ++iter;
          }
        }
        dataModel->Sort(1);
        dataModel->On();
      }
    }
    else {
      Dialog::Resync(model,msg);
    }
  }

  bool ClosedSuccess()
  {
    return success;
  }

  void GetResult(std::wstring& name, ::Base::DataPtr& data) const
  {
    name=nameModel->Get();
    data=dataModel->GetEntry(selectionModel->GetLine()).data;
  }
};


bool GetDataLink(Lum::OS::Window* parent,
                 std::wstring& linkName,
                 Base::DataPtr& data)
{
  DataLinkDialog *dialog;

  dialog=new DataLinkDialog();
  dialog->SetParent(parent);

  if (dialog->Open()) {
    dialog->EventLoop();
    dialog->Close();
  }

  if (dialog->ClosedSuccess()) {
   dialog->GetResult(linkName,data);

    delete dialog;
    return true;
  }
  else {
    delete dialog;
    return false;
  }
}
