
/*
	MAEMO specific emulator code

	Uses SDL for video, audio and event processing

*/


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <fcntl.h>
#include <errno.h>
#include <SDL/SDL.h>
#include <glib.h>
#include <pthread.h>

#include "maemo.h"
#include "usbjoy.h"
#include "../common/arm_utils.h"
#include "emu.h"
#include "../common/emu.h"

#define min(a,b) (a<b?a:b)

extern osso_events( gboolean block );

static int running = 0;
static pthread_t sdl_thread;

void *md_screen;
#define SND_BUFFER_SIZE 0x10000
static unsigned char sound_buffer[SND_BUFFER_SIZE];
static volatile int soundptrr, soundptrw;

static SDL_Surface* screen;
static unsigned long keystates;
static unsigned long mousestates;
static unsigned long accelstates;

static unsigned short ts_key_layout[800*480];

static void sdl_key( int key, int press ) {
	unsigned short cmd = key_map[key];
	if( cmd < 0x8000 ) {
		if( currentConfig.enableKey )
			if( press ) keystates |= cmd; else keystates &= ~cmd;
	} else switch( cmd ) {
		case CMD_QUIT:
			running = 0;
			maemo_quit();
		break;
	}
}

static void sdl_mouse( int x, int y, int mode ) {
	if( !currentConfig.enableTS ) return;
	if( ! (mode & 0x01) ) mousestates = 0;
	if( !mode ) return;
#define set( cmd ) mousestates |= cmd
	if( x >= 0 && x <= 300 && y >= 0 && y <= 480 ) {
		if( x >= 0 && x <= 100 ) set( MDB_LEFT );
		if( x >= 200 && x <= 300 ) set( MDB_RIGHT );
		if( y >= 0 && y <= 160 ) set( MDB_UP );
		if( y >= 320 && y <= 480 ) set( MDB_DOWN );
	} else if( x >= 700 && x <= 800 && y >= 0 && y <= 480) {
		if( y <= 160 ) set( MDB_A );
		else if( y <= 320 ) set( MDB_B );
		else set( MDB_C );
	}
}

static void read_accelerometer() {
	if( !currentConfig.enableAccel ) return;
#define ACCEL_TOLERANCE currentConfig.accelSensitivity
	accelstates = 0;
	int x, y, z;
	FILE* f = fopen( "/sys/class/i2c-adapter/i2c-3/3-001d/coord", "r" );
	if( !f ) return;
	fscanf( f, "%d %d %d", &x, &y, &z );
	fclose( f );
	if( x > ACCEL_TOLERANCE ) accelstates |= MDB_LEFT;
	else if( x < -ACCEL_TOLERANCE ) accelstates |= MDB_RIGHT;
	if( y > ACCEL_TOLERANCE ) accelstates |= MDB_UP;
	else if( y < -ACCEL_TOLERANCE ) accelstates |= MDB_DOWN;
}

static void* SDL_Loop( void* obj ) {
	while( running ) {
		SDL_Event event;
		while( SDL_PollEvent( &event ) ) {
			switch( event.type ) {
			case SDL_MOUSEBUTTONDOWN:
				sdl_mouse( event.button.x, event.button.y, 1 );
				break;
			case SDL_MOUSEBUTTONUP:
				sdl_mouse( event.button.x, event.button.y, 0 );
				break;
			case SDL_MOUSEMOTION:
				sdl_mouse( event.button.x, event.button.y, 2 );
				break;
			case SDL_KEYDOWN:
				sdl_key( event.key.keysym.scancode, 1 );
				break;
			case SDL_KEYUP:
				sdl_key( event.key.keysym.scancode, 0 );
				break;
			case SDL_QUIT:
				maemo_quit();
				break;
			}
		}
		read_accelerometer();
		while( osso_events( FALSE ) );
		SDL_Delay( 50 );
	}
	return NULL;
}

static void audio_loop( void* userdata, Uint8* stream, int len ) {
	register int ptrr = soundptrr, ptrw = soundptrw;
	int total = ptrw - ptrr;
	if( total < 0 ) {
		total = min( SND_BUFFER_SIZE - ptrr, len );
		memcpy( stream, sound_buffer + ptrr, total );
		stream += total;
		len -= total;
		ptrr = ( ptrr + total ) % SND_BUFFER_SIZE;
		total = ptrw;
	}
	total = min( total, len );
	memcpy( stream, sound_buffer + ptrr, total );
	soundptrr = ( ptrr + total ) % SND_BUFFER_SIZE;
}

void maemo_quit() {
	emu_set_state( PGS_Quit );
	running = 0;
}

/* video stuff */
void maemo_video_flip(void)
{
#define clerp16(a,b) ( ( ( ( a >> 12 ) + ( b >> 12 ) ) & 0x1F ) << 11 | ( ( ( a >> 6 ) + ( b >> 6 ) ) & 0x3F ) << 5 | ( ( ( a >> 1 ) + ( b >> 1 ) ) & 0x1F ) )
	register int i, j;
	register unsigned short *buffer = md_screen;
	unsigned short *frame = screen->pixels;
	register unsigned short *scan1 = frame;
	register unsigned short *scan2 = frame;
	if( currentConfig.displayTS ) memcpy( frame, ts_key_layout, sizeof(ts_key_layout) );
	switch( currentConfig.scaling ) {
	case 1:
		for( j = 0; j < 240; j++ ) {
			scan1 = frame + j*800 + 96000 + 240;
			for( i = 0; i < 320; i++ ) {
				*scan1++ = *buffer++;
			}
		}
		break;
	case 0:
		for( j = 0; j < 240; j++ ) {
			scan1 = frame + j*800*2 + 80;
			scan2 = frame + j*800*2 + 800 + 80;
			for( i = 0; i < 320; i++ ) {
				*scan1++ = *buffer;
				*scan1++ = *buffer;
				*scan2++ = *buffer;
				*scan2++ = *buffer++;
			}
		}
		break;
	case 2:
		for( j = 0; j < 240; j++ ) {
			scan1 = frame + j*800*2;
			scan2 = frame + j*800*2 + 800;
			for( i = 0; i < 320; i++ ) {
				register unsigned short pix = *buffer++;
				if( i & 0x01 ) {
					*scan1 = clerp16( *scan1, pix );
					*scan2 = clerp16( *scan2, pix );
				}
				*scan1++ = pix;
				*scan1++ = pix;
				*scan2++ = pix;
				*scan2++ = pix;
				if( ! (i & 0x01) ) {
					*scan1++ = pix;
					*scan2++ = pix;
				}
			}
		}
		break;
	}
	SDL_Flip(screen);
}

void maemo_memset_buffer(int offset, int byte, int len)
{
	memset( md_screen + offset, byte, len );
}


void maemo_cls(void)
{
	memset( md_screen, 0, 320*240*2 );
}


unsigned long maemo_joystick_read(int allow_usb_joy)
{
	return keystates | mousestates | accelstates;
}

void maemo_start_sound(int rate, int bits, int stereo)
{
	SDL_AudioSpec as;
	memset( &as, 0, sizeof(as) );
	as.callback = audio_loop;
	as.channels = stereo ? 2 : 1;
	as.format = AUDIO_S16;
	as.freq = rate;
	as.samples = 0x400;

	memset( sound_buffer, 0, sizeof(sound_buffer) );
	soundptrw = soundptrr = 0;

	SDL_OpenAudio( &as, NULL );
	SDL_PauseAudio(0);
}


void maemo_sound_write(void *buff, int len)
{
	register int ptrr = soundptrr, ptrw = soundptrw;
	register int total = ptrr - ptrw - 1;
	if( total < 0 ) {
		total = min( SND_BUFFER_SIZE - ptrw, len );
		memcpy( sound_buffer + ptrw, buff, total );
		buff += total;
		len -= total;
		ptrw = ( ptrw + total ) % SND_BUFFER_SIZE;
	}
	total = min( len, total );
	memcpy( sound_buffer + ptrw, buff, total );
	soundptrw = ( ptrw + total ) % SND_BUFFER_SIZE;
}

void maemo_sound_sync(void)
{
	//ioctl(sounddev, SOUND_PCM_SYNC, 0);
}

void maemo_sound_volume(int l, int r)
{
 	/*l=l<0?0:l; l=l>255?255:l; r=r<0?0:r; r=r>255?255:r;
 	l<<=8; l|=r;
 	ioctl(mixerdev, SOUND_MIXER_WRITE_PCM, &l);*/ /*SOUND_MIXER_WRITE_VOLUME*/
}

static void init_ts_key_layout() {
	memset( ts_key_layout, 0, sizeof( ts_key_layout ) );
	int i, j;
	for( i = 1; i <= 3; i++ )
		for( j = 0; j < 480; j++ )
			ts_key_layout[j*800+i*100] = 0xFFFF;
	for( i = 1; i < 3; i++ )
		for( j = 0; j < 300; j++ )
			ts_key_layout[j+i*160*800] = 0xFFFF;
	for( i = 0; i < 480; i++ )
		ts_key_layout[i*800+700] = 0xFFFF;
	for( i = 1; i < 3; i++ )
		for( j = 700; j < 800; j++ )
			ts_key_layout[i*160*800+j] = 0xFFFF;
}

/* common */
void maemo_init(void)
{
	printf("entering init()\n"); fflush(stdout);

	SDL_Init( SDL_INIT_VIDEO | SDL_INIT_AUDIO | SDL_INIT_EVENTTHREAD );
	SDL_WM_SetCaption( "PicoDrive for Maemo 1.0", NULL );
	SDL_ShowCursor( SDL_DISABLE );
	SDL_WM_GrabInput( SDL_GRAB_OFF );

	screen = SDL_SetVideoMode( 800, 480, 16, SDL_HWSURFACE | SDL_DOUBLEBUF );
	SDL_WM_ToggleFullScreen( screen );
	SDL_SetCursor( SDL_DISABLE );

	md_screen = malloc( 320*240*2 );
	init_ts_key_layout();

	keystates = mousestates = accelstates = 0;
	/* init usb joys -GnoStiC */
	maemo_usbjoy_init();

	running = 1;
	pthread_create( &sdl_thread, NULL, SDL_Loop, NULL );

	printf("exitting init()\n"); fflush(stdout);
}

char *ext_menu = 0, *ext_state = 0;

void maemo_deinit(void)
{
	running = 0;
	pthread_join( sdl_thread, NULL );

	maemo_usbjoy_deinit();

	SDL_PauseAudio(1);
	SDL_CloseAudio();

	free( md_screen );

	SDL_FreeSurface( screen );
	SDL_Quit();

	printf("all done, quitting\n");
}

/* lprintf */
void lprintf(const char *fmt, ...)
{
	va_list vl;

	va_start(vl, fmt);
	vprintf(fmt, vl);
	va_end(vl);
}

