//-----------------------------------------------------------------------------
#include "ImageWidget.h"
//-----------------------------------------------------------------------------

#include "Definitions.h"
#include "common/Utils.h"

#include <QPainter>
#include <QMenu>
#include <QMouseEvent>
#include <QMessageBox>

#include <QDebug>

//-----------------------------------------------------------------------------

#define CROP_MIN_WIDTH    40.0
#define CROP_MIN_HEIGHT   30.0
#define CROP_RESIZE_AREA  35.0
#define MAX_ZOOM_FACTOR   2

#define DEFAULT_CROP_LEFT   0.2
#define DEFAULT_CROP_TOP    0.3
#define DEFAULT_CROP_RIGHT  0.8
#define DEFAULT_CROP_BOTTOM 0.7

//-----------------------------------------------------------------------------
ImageWidget::ImageWidget(QWidget* a_parent):
  QWidget(a_parent)
{
  setContextMenuPolicy(Qt::NoContextMenu);

  m_hasImage = false;
  m_zoomFactor = 1.0;

  m_cropRotation = 0;

  resetState();
}

//-----------------------------------------------------------------------------
ImageWidget::~ImageWidget()
{
}

//-----------------------------------------------------------------------------
void ImageWidget::setMaximumViewportSize(const QSize& a_size)
{
  qDebug() << "ImageWidget::setMaximumViewportSize(" << a_size << ")";

  m_maxSize = a_size;
}

//-----------------------------------------------------------------------------
void ImageWidget::resetState()
{
  m_mouseState = MouseState_None;
}

//-----------------------------------------------------------------------------
void ImageWidget::loadFile(const QString& a_path)
{
  qDebug() << "ImageWidget::loadFile(" << a_path << ")";

  resetState();

  m_imageOriginal.load(a_path);

  m_imageDrawing = QPixmap();

  m_hasImage = true;

  m_rectCrop = QRectF(QPointF(DEFAULT_CROP_LEFT, DEFAULT_CROP_TOP), QPointF(DEFAULT_CROP_RIGHT, DEFAULT_CROP_BOTTOM));
  zoomTo(1.0);

  update();
}

//-----------------------------------------------------------------------------
void ImageWidget::zoomIn()
{
  emit toggleBusy(true);

  zoomTo(m_zoomFactor * 1.25);

  emit toggleBusy(false);
}

//-----------------------------------------------------------------------------
void ImageWidget::zoomOut()
{
  emit toggleBusy(true);

  zoomTo(m_zoomFactor / 1.25);

  emit toggleBusy(false);
}

//-----------------------------------------------------------------------------
void ImageWidget::zoomTo(qreal a_zoom)
{
  qDebug() << "ImageWidget::zoomTo(" << a_zoom << ")";

  m_zoomFactor = a_zoom;
  if(m_zoomFactor < 1.01)
  {
    m_zoomFactor = 1.0;
    resize(imageBaseSize());
  }
  else
  {
    QSize sizeBase = imageBaseSize();
    QSize sizeNew = sizeBase * m_zoomFactor;
    if(sizeNew.width() > m_imageOriginal.width() * MAX_ZOOM_FACTOR)
    {
      sizeNew = m_imageOriginal.size() * MAX_ZOOM_FACTOR;
      m_zoomFactor = qreal(sizeNew.width()) / qreal(sizeBase.width());
    }
    resize(sizeNew);
  }
}

//-----------------------------------------------------------------------------
void ImageWidget::rotate()
{
  qDebug() << "ImageWidget::rotate()";

  emit toggleBusy(true);

  QSize size = m_imageOriginal.size();
  QPixmap rotated(size.height(), size.width());
  QPainter painter(&rotated);
  painter.translate(size.height()/2,size.height()/2);
  painter.rotate(90);
  painter.translate(-size.height()/2,-size.height()/2);
  painter.drawPixmap(0, 0, m_imageOriginal);
  painter.end();

  m_imageOriginal = rotated;
  m_imageDrawing = QPixmap();

  m_rectCrop = QRectF(QPointF(DEFAULT_CROP_LEFT, DEFAULT_CROP_TOP), QPointF(DEFAULT_CROP_RIGHT, DEFAULT_CROP_BOTTOM));
  zoomTo(1.0);

  // Update drawing
  update();

  emit toggleBusy(false);
}

//-----------------------------------------------------------------------------
void ImageWidget::resetCrop()
{
  qDebug() << "ImageWidget::resetCrop()";

  // Get the current visible rect
  QRect rectVisible = countVisibleRect();

  // Calculate crop rect that fits the visible rect
  qreal cropLeft = qreal(rectVisible.left()) / qreal(width());
  qreal cropTop = qreal(rectVisible.top()) / qreal(height());
  qreal cropRight = qreal(rectVisible.right()) / qreal(width());
  qreal cropBottom = qreal(rectVisible.bottom()) / qreal(height());
  qreal cropWidth = cropRight - cropLeft;
  qreal cropHeight = cropBottom - cropTop;

  m_rectCrop = QRectF( QPointF(cropLeft + cropWidth * DEFAULT_CROP_LEFT, cropTop + cropHeight * DEFAULT_CROP_TOP),
                       QPointF(cropLeft + cropWidth * DEFAULT_CROP_RIGHT, cropTop + cropHeight * DEFAULT_CROP_BOTTOM) );

  // Update drawing
  update();
}

//-----------------------------------------------------------------------------
QRect ImageWidget::countVisibleRect() const
{
  // Get parent viewport corners
  QPoint viewportTopLeft = mapFromParent(parentWidget()->rect().topLeft());
  QPoint viewportBottomRight = mapFromParent(parentWidget()->rect().bottomRight());

  // Adjust empty space
  int visibleLeft = viewportTopLeft.x() >= rect().left() ? viewportTopLeft.x() : rect().left();
  int visibleTop = viewportTopLeft.y() >= rect().top() ? viewportTopLeft.y() : rect().top();
  int visibleRight = viewportBottomRight.x() <= rect().right() ? viewportBottomRight.x() : rect().right();
  int visibleBottom = viewportBottomRight.y() <= rect().bottom() ? viewportBottomRight.y() : rect().bottom();

  return QRect(QPoint(visibleLeft, visibleTop), QPoint(visibleRight, visibleBottom));
}

//-----------------------------------------------------------------------------
QSize ImageWidget::imageBaseSize() const
{
  if(!m_hasImage)
    return m_maxSize;

  QSize sizeBaseSize = m_imageOriginal.size();
  sizeBaseSize.scale(m_maxSize, Qt::KeepAspectRatio);

  return sizeBaseSize;
}

//-----------------------------------------------------------------------------
QPolygon ImageWidget::cropPolygon(const QSize& a_size) const
{
  QRect rectCrop = Utils::toAbsoluteFromRelative(a_size, m_rectCrop);

  // Translate crop center to origo
  QPoint cropCenter = rectCrop.center();
  rectCrop.translate(-cropCenter);

  // Rotate crop
  QTransform transformCrop;
  transformCrop.rotate(m_cropRotation);

  // Create crop polygon
  QPolygon cropPoly = transformCrop.mapToPolygon(rectCrop);
  cropPoly.translate(cropCenter);

  return cropPoly;
}

//-----------------------------------------------------------------------------
void ImageWidget::translateCrop(const QPointF& a_translation)
{
  QPointF trans = a_translation;

  // Check x bounds
  if(m_rectCrop.left() + trans.x() < 0.0)
    trans.setX(0.0-m_rectCrop.left());
  else if(m_rectCrop.right() + trans.x() > 1.0)
    trans.setX(1.0-m_rectCrop.right());

  // Check y bounds
  if(m_rectCrop.top() + trans.y() < 0.0)
    trans.setY(0.0-m_rectCrop.top());
  else if(m_rectCrop.bottom() + trans.y() > 1.0)
    trans.setY(1.0-m_rectCrop.bottom());

  m_rectCrop.translate(trans);
}

//-----------------------------------------------------------------------------
void ImageWidget::resizeCrop(const QPoint& a_translation)
{
/*
  QTransform transformRotate;
  transformRotate.rotate(-m_cropRotation);
  QPointF trans = Utils::toRelativeFromAbsolute(size(), transformRotate.map(a_translation));



  // Resize
  m_rectCrop.setBottomRight(m_rectCrop.bottomRight() + trans);
*/


  QPointF trans = Utils::toRelativeFromAbsolute(size(), a_translation);
  QRectF rectCrop = m_rectCrop;

  // Do not let go outside from right or bottom
  if(m_rectCrop.right() + trans.x() > 1.0)
    trans.setX(1.0-m_rectCrop.right());
  if(m_rectCrop.bottom() + trans.y() > 1.0)
    trans.setY(1.0-m_rectCrop.bottom());
  rectCrop.setBottomRight(rectCrop.bottomRight() + trans);

  // Do not let go too small
  QPoint adjustTransAbs(0, 0);
  QRect rectAbs(Utils::toAbsoluteFromRelative(size(), rectCrop));
  if(rectAbs.width() < CROP_MIN_WIDTH)
    adjustTransAbs.setX(CROP_MIN_WIDTH - rectAbs.width());
  if(rectAbs.height() < CROP_MIN_HEIGHT)
    adjustTransAbs.setY(CROP_MIN_HEIGHT - rectAbs.height());
  QPointF adjustTrans(Utils::toRelativeFromAbsolute(size(), adjustTransAbs));

  // Resize
  m_rectCrop.setBottomRight(m_rectCrop.bottomRight() + trans + adjustTrans);
}

//-----------------------------------------------------------------------------
void ImageWidget::saveTo(const QString& a_path, const QString& a_type)
{
  qDebug() << "ImageWidget::resetCurrent()";

  QPixmap imageCrop;

  // We need to do a bit more operations if we have a rotated crop
  if(m_cropRotation != 0.0)
  {
    QRect rectCrop = Utils::toAbsoluteFromRelative(m_imageOriginal.size(), m_rectCrop);

    // Translate crop center to origo
    QPoint cropCenter = rectCrop.center();
    rectCrop.translate(-cropCenter);

    // Rotate crop
    QTransform transformCrop;
    transformCrop.rotate(m_cropRotation);

    // Get crop bounding rect
    QPolygon cropPoly = transformCrop.mapToPolygon(rectCrop);
    QRect rectOriginalCopy = cropPoly.boundingRect();
    rectOriginalCopy.translate(cropCenter);

    // Copy crop bounds area from original
    imageCrop = m_imageOriginal.copy(rectOriginalCopy);

    // Rotate image backwards
    QTransform transformRotate;
    transformRotate.rotate(-m_cropRotation);
    imageCrop = imageCrop.transformed(transformRotate, Qt::SmoothTransformation);

    // Crop rotated image to correct size
    QRect rectFinalCopy = rectCrop.translated(imageCrop.rect().center());
    imageCrop = imageCrop.copy(rectFinalCopy);
  }
  // Non-rotated crop is easy
  else
  {
    // Copy crop area from original
    QRect rectCrop = Utils::toAbsoluteFromRelative(m_imageOriginal.size(), m_rectCrop);
    imageCrop = m_imageOriginal.copy(rectCrop);
  }

  // Save crop area to file

  bool save = imageCrop.save(a_path, a_type.toAscii(), 100);

}

//-----------------------------------------------------------------------------
void ImageWidget::visibleRectChanged()
{
  QRect rectDrawing(m_pointDrawing, m_imageDrawing.size());
  QRect rectVisible = countVisibleRect();
  if(!rectDrawing.contains(rectVisible))
  {
    emit toggleBusy(true);

    qDebug() << "ImageWidget::visibleRectChanged(): need to update drawing image";
    updateDrawingImage();

    emit toggleBusy(false);
  }
}

//-----------------------------------------------------------------------------
void ImageWidget::mousePressEvent(QMouseEvent* a_event)
{
  if(m_hasImage)
  {
    // Clicked inside crop area?
    QPolygon cropPoly = cropPolygon(size());
    if(cropPoly.containsPoint(a_event->pos(), Qt::OddEvenFill))
    {
      if(QLineF(cropPoly[2].x(), cropPoly[2].y(), a_event->x(), a_event->y()).length() < CROP_RESIZE_AREA)
      {
        qDebug() << "ImageWidget::mousePressEvent(): start resize crop";
        m_mouseState = MouseState_Resize;
      }
      else
      {
        qDebug() << "ImageWidget::mousePressEvent(): start move crop";
        m_mouseState = MouseState_Move;
      }
      m_pointLastMousePos = a_event->pos();

      a_event->accept();
      return;
    }
  }

  a_event->ignore();
}

//-----------------------------------------------------------------------------
void ImageWidget::mouseMoveEvent(QMouseEvent* a_event)
{
  if(m_mouseState == MouseState_Resize || m_mouseState == MouseState_Move)
  {
    if(a_event->pos() != m_pointLastMousePos)
    {
      if(m_mouseState == MouseState_Resize)
        resizeCrop(a_event->pos() - m_pointLastMousePos);
      else if(m_mouseState == MouseState_Move)
        translateCrop(Utils::toRelativeFromAbsolute(size(), a_event->pos() - m_pointLastMousePos));
      m_pointLastMousePos = a_event->pos();
      update();
    }

    a_event->accept();
    return;
  }

  a_event->ignore();
}

//-----------------------------------------------------------------------------
void ImageWidget::mouseReleaseEvent(QMouseEvent* a_event)
{
  if(m_mouseState == MouseState_Resize || m_mouseState == MouseState_Move)
  {
    if(a_event->pos() != m_pointLastMousePos)
    {
      if(m_mouseState == MouseState_Resize)
        resizeCrop(a_event->pos() - m_pointLastMousePos);
      else if(m_mouseState == MouseState_Move)
        translateCrop(Utils::toRelativeFromAbsolute(size(), a_event->pos() - m_pointLastMousePos));
      update();
    }
    m_mouseState = MouseState_None;
    a_event->accept();
    return;
  }

  a_event->ignore();
}

//-----------------------------------------------------------------------------
void ImageWidget::resizeEvent(QResizeEvent* a_event)
{
  qDebug() << "ImageWidget::resizeEvent(" << a_event->size() << ")";

  if(!m_hasImage)
    return;

  updateDrawingImage();
}

//-----------------------------------------------------------------------------
void ImageWidget::updateDrawingImage()
{
  // Use image portion
  if(size().width() > 1200)
  {
    // Get the current visible rect
    QRect rectVisible = countVisibleRect();
    QSize sizeVisible = rectVisible.size();

    // Make visible rect larger to ease moving
    QPoint pointResize(sizeVisible.width(), sizeVisible.height());
    rectVisible.setTopLeft(rectVisible.topLeft() - pointResize);
    rectVisible.setBottomRight(rectVisible.bottomRight() + pointResize);
    rectVisible = rectVisible.intersected(rect());

    // Calculate the visible rect in original image
    qreal factor = qreal(width()) / qreal(m_imageOriginal.width());
    QRect rectOriginal( QPoint(int(qreal(rectVisible.left()) / factor), int(qreal(rectVisible.top()) / factor)),
                        QPoint(int(qreal(rectVisible.right()) / factor), int(qreal(rectVisible.bottom()) / factor)) );

    // Copy & scale portion for drawing
    qDebug() << "scaling...";
    m_pointDrawing = rectVisible.topLeft();
    m_imageDrawing = m_imageOriginal.copy(rectOriginal).scaled(rectVisible.size(), Qt::IgnoreAspectRatio, Qt::FastTransformation);
    qDebug() << "finish scaling!";
  }
  // Use whole image scaled
  else
  {
    // Scale whole image
    qDebug() << "scaling whole image...";
    m_pointDrawing = QPoint(0, 0);
    m_imageDrawing = m_imageOriginal.scaled(size(), Qt::IgnoreAspectRatio, Qt::FastTransformation);
    qDebug() << "finish scaling!";
  }
}

//-----------------------------------------------------------------------------
void ImageWidget::paintEvent(QPaintEvent* a_event)
{
  QPainter painter(this);
  QRect rectClip = a_event->rect();
  painter.setClipRect(rectClip);

  if(!m_hasImage)
    return;

  // Draw image
  if(!m_imageDrawing.isNull())
    painter.drawPixmap(rectClip, m_imageDrawing, QRect(rectClip.topLeft() - m_pointDrawing, rectClip.size()));

  //----------------
  // Visualize crop

  // Translate crop rect to origo
  QRect rectCrop = Utils::toAbsoluteFromRelative(size(), m_rectCrop);
  QPoint rectCenter = rectCrop.center();
  rectClip.translate(-rectCenter);
  rectCrop.translate(-rectCenter);
  painter.translate(rectCenter);

  // Rotate
  painter.rotate(m_cropRotation);

  // Draw outside crop grey
  painter.setBrush(QBrush(QColor(0x40, 0x40, 0x40, 0xA0)));
  painter.setPen(Qt::NoPen);
  // Up
  if(rectClip.top() < rectCrop.top()-1)
    painter.drawRect(QRect(rectClip.topLeft(), QPoint(rectClip.right(), rectCrop.top()-1)));
  // Down
  if(rectClip.bottom() > rectCrop.bottom())
    painter.drawRect(QRect(QPoint(rectClip.left(), rectCrop.bottom()+1), rectClip.bottomRight()));
  // Left
  if(rectClip.left() < rectCrop.left())
    painter.drawRect(QRect(QPoint(rectClip.left(), rectCrop.top()), rectCrop.bottomLeft()));
  // Right
  if(rectClip.right() > rectCrop.right())
    painter.drawRect(QRect(QPoint(rectCrop.right()+1, rectCrop.top()), QPoint(rectClip.right(), rectCrop.bottom())));

  // Draw crop rect
  painter.setBrush(QBrush(Qt::NoBrush));
  painter.setPen(QPen(QBrush(QColor(0xFF, 0xFF, 0xFF, 0xFF)), 1.0));
  painter.drawRect(rectCrop);

  // Draw resize handle
  painter.setBrush(QBrush(QColor(0xFF, 0xFF, 0xFF, 0x40)));
  painter.setPen(QPen(QBrush(QColor(0xFF, 0xFF, 0xFF, 0xFF)), 1.0, Qt::SolidLine, Qt::RoundCap, Qt::BevelJoin));

  painter.translate(rectCrop.bottomRight() - QPoint(1, 1));

  QPolygonF polyHandle1;
  const qreal h1s =  5.0;
  const qreal h1b = 12.0;
  polyHandle1 << QPointF(-h1s, 0.0);
  polyHandle1 << QPointF(-h1b, 0.0);
  polyHandle1 << QPointF(0.0, -h1b);
  polyHandle1 << QPointF(0.0, -h1s);
  painter.drawPolygon(polyHandle1);

  QPolygonF polyHandle2;
  const qreal h2s = 20.0;
  const qreal h2b = 27.0;
  polyHandle2 << QPointF(-h2s, 0.0);
  polyHandle2 << QPointF(-h2b, 0.0);
  polyHandle2 << QPointF(0.0, -h2b);
  polyHandle2 << QPointF(0.0, -h2s);
  painter.drawPolygon(polyHandle2);

  QPolygonF polyHandle3;
  const qreal h3s = 35.0;
  const qreal h3b = 42.0;
  polyHandle3 << QPointF(-h3s, 0.0);
  polyHandle3 << QPointF(-h3b, 0.0);
  polyHandle3 << QPointF(0.0, -h3b);
  polyHandle3 << QPointF(0.0, -h3s);
  painter.drawPolygon(polyHandle3);

  painter.resetTransform();
}

//-----------------------------------------------------------------------------
