//-----------------------------------------------------------------------------
#include "Translator_Google.h"
//-----------------------------------------------------------------------------

#include "common/Language.h"

#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTextCodec>
#include <QtScript/QtScript>
#include <qmessagebox.h>

#include <QDebug>

//-----------------------------------------------------------------------------
Translator_Google::Translator_Google(QObject* a_parent):
  Translator(a_parent)
{
  m_network = new QNetworkAccessManager(this);
  m_reply = 0;
  m_cancelled = false;

  // Map languages
  m_mapLanguage.insert(int(Language_Albanian), "sq");
  m_mapLanguage.insert(int(Language_Arabic), "ar");
  m_mapLanguage.insert(int(Language_Bulgarian), "bg");
  m_mapLanguage.insert(int(Language_Catalan), "ca");
  m_mapLanguage.insert(int(Language_Chinese_Simplified), "zh-CN");
  m_mapLanguage.insert(int(Language_Chinese_Traditional), "zh-TW");
  m_mapLanguage.insert(int(Language_Croatian), "hr");
  m_mapLanguage.insert(int(Language_Czech), "cs");
  m_mapLanguage.insert(int(Language_Danish), "da");
  m_mapLanguage.insert(int(Language_Dutch), "nl");
  m_mapLanguage.insert(int(Language_English), "en");
  m_mapLanguage.insert(int(Language_Estonian), "et");
  m_mapLanguage.insert(int(Language_Filipino), "tl");
  m_mapLanguage.insert(int(Language_Finnish), "fi");
  m_mapLanguage.insert(int(Language_French), "fr");
  m_mapLanguage.insert(int(Language_Galician), "gl");
  m_mapLanguage.insert(int(Language_German), "de");
  m_mapLanguage.insert(int(Language_Greek), "el");
  m_mapLanguage.insert(int(Language_Hebrew), "iw");
  m_mapLanguage.insert(int(Language_Hindi), "hi");
  m_mapLanguage.insert(int(Language_Hungarian), "hu");
  m_mapLanguage.insert(int(Language_Indonesian), "id");
  m_mapLanguage.insert(int(Language_Italian), "it");
  m_mapLanguage.insert(int(Language_Japanese), "ja");
  m_mapLanguage.insert(int(Language_Korean), "ko");
  m_mapLanguage.insert(int(Language_Latvian), "lv");
  m_mapLanguage.insert(int(Language_Lithuanian), "lt");
  m_mapLanguage.insert(int(Language_Maltese), "mt");
  m_mapLanguage.insert(int(Language_Norwegian), "no");
  m_mapLanguage.insert(int(Language_Polish), "pl");
  m_mapLanguage.insert(int(Language_Portuguese), "pt");
  m_mapLanguage.insert(int(Language_Romanian), "ro");
  m_mapLanguage.insert(int(Language_Russian), "ru");
  m_mapLanguage.insert(int(Language_Serbian), "sr");
  m_mapLanguage.insert(int(Language_Slovak), "sk");
  m_mapLanguage.insert(int(Language_Slovenian), "sl");
  m_mapLanguage.insert(int(Language_Spanish), "es");
  m_mapLanguage.insert(int(Language_Swedish), "sv");
  m_mapLanguage.insert(int(Language_Thai), "th");
  m_mapLanguage.insert(int(Language_Turkish), "tr");
  m_mapLanguage.insert(int(Language_Ukrainian), "uk");
  m_mapLanguage.insert(int(Language_Vietnamese), "vi");
}

//-----------------------------------------------------------------------------
Translator_Google::~Translator_Google()
{
  if(m_reply)
  {
    m_reply->abort();
    m_reply->deleteLater();
  }
  m_reply = 0;
}

//-----------------------------------------------------------------------------
QList<int> Translator_Google::supportedLanguages() const
{
  return m_mapLanguage.keys();
}

//-----------------------------------------------------------------------------
void Translator_Google::run(const QString& a_text, int a_langFrom, int a_langTo)
{
  if(m_reply)
  {
    qDebug() << "Translator_Google::run(): Old reply still existing!!!";
    m_reply->abort();
    m_reply->deleteLater();
  }
  m_reply = 0;

  QUrl url("http://ajax.googleapis.com/ajax/services/language/translate");
  url.addQueryItem("v", "1.0");
  url.addQueryItem("q", a_text);
  url.addQueryItem("langpair", m_mapLanguage.value(a_langFrom, "") + QString("|") + m_mapLanguage.value(a_langTo, ""));

  qDebug() << "Translator_Google::run(): URL:" << url.toString();

  m_cancelled = false;
  m_reply = m_network->get(QNetworkRequest(url));
  connect(m_reply, SIGNAL(finished()), this, SLOT(replyFinished()));
}

//-----------------------------------------------------------------------------
bool Translator_Google::cancel()
{
  if(m_reply)
  {
    qDebug() << "Translator_Google::cancel(): running";

    m_cancelled = true;

    m_reply->abort();
    m_reply->deleteLater();
    m_reply = 0;

    return true;
  }

  qDebug() << "Translator_Google::cancel(): not running";

  return false;
}

//-----------------------------------------------------------------------------
void Translator_Google::replyFinished()
{
  if(!m_reply)
  {
    qDebug() << "Translator_Google::replyFinished(): No reply!!!";
    return;
  }

  if(m_reply->error() == QNetworkReply::NoError)
    parseReply();
  else
    parseError();

  m_reply->deleteLater();
  m_reply = 0;
}

//-----------------------------------------------------------------------------
void Translator_Google::parseReply()
{
  if(m_cancelled)
  {
    qDebug() << "Translator_Google::parseReply(): cancelled -> skip";
    return;
  }

  qDebug() << "Translator_Google::parseReply()";

  QString strText;

  if(QTextCodec* codec = QTextCodec::codecForName("UTF-8"))
    strText = codec->toUnicode(m_reply->readAll());

  QScriptValue sc;
  QScriptEngine engine;

  sc = engine.evaluate("value = " + strText);
  strText = "";

  if(!sc.isValid() || !sc.isObject())
  {
      QMessageBox::critical(NULL, "PhotoTranslator", "Invalid response received from translator.");
      return;
  }
  else
  {
      //at this point we should have 3 propeties there
      //1. "responseData" object
      //2. "responseDetails" : null | string-on-error
      //3.  "responseStatus" : 200 | error-code
      QScriptValue responseStatus = sc.property("responseStatus");
      if(!responseStatus.isNumber())
      {         
          QMessageBox::critical(NULL, "PhotoTranslator", "Invalid status received from translator.");
          return;
      }
      else if(responseStatus.toNumber() != 200)
      {
          QScriptValue responseDetails = sc.property("responseDetails");
          QString strDetails;
          if(!responseDetails.isNull() && responseDetails.isString())
              strDetails = responseDetails.toString();

          QString error;
          error.setNum(responseStatus.toNumber());
          error = "An error occurred.\nError Status: " + error +
                  "\nError details " + strDetails;
          QMessageBox::critical(NULL, "PhotoTranslator", error);
      }
      else
      {
          //status ok here
          QScriptValue responseData = sc.property("responseData");
          if(responseData.isObject())
          {
              QScriptValue translatedText = responseData.property("translatedText");
              if(translatedText.isString())
              {
                  strText = translatedText.toString();                  
              }
          }
      }
  }

  // TODO: Done with cheapo-way! To do this properly the returned string should be parsed.
  // QVariantMap mapReply;
  //strText.replace(QString("{\"responseData\": {\"translatedText\":\""), "");
  //strText.replace(QString("\"}, \"responseDetails\": null, \"responseStatus\": 200}"), "");

  // Replace special signs
  strText.replace(QString("\\u0026"), "&");
  strText.replace(QString("\\u003d"), "=");
  strText.replace(QString("&quot;"), "\"");
  strText.replace(QString("&amp;"), "&");
  strText.replace(QString("&lt;"), "<");
  strText.replace(QString("&gt;"), ">");

  emit done(strText);
}

//-----------------------------------------------------------------------------
void Translator_Google::parseError()
{
  if(m_cancelled)
  {
    qDebug() << "Translator_Google::parseError(): cancelled -> skip";
    return;
  }

  qDebug() << "Translator_Google::parseError()";

  emit error(tr("Translation failed.\nNetwork error: %1").arg(m_reply->errorString()));
}

//-----------------------------------------------------------------------------
