//-----------------------------------------------------------------------------
#include "View_Translate.h"
#include "ui_View_Translate.h"
//-----------------------------------------------------------------------------

#include "Context.h"
#include "Definitions.h"
#include "common/Language.h"
#include "logic/OCR.h"
#include "logic/Translator.h"
#include "settings/Settings.h"
#include "settings/SettingsDialog.h"

//#ifdef Q_WS_MAEMO_5
//  #include <QtMaemo5/QMaemo5KineticScroller>
//  #include <QtMaemo5/QMaemo5ValueButton>
//#endif

#include <QDebug>

//-----------------------------------------------------------------------------
View_Translate::View_Translate(Context* a_context, QWidget* a_parent):
  View(a_context, a_parent),
  m_ui(new Ui::View_Translate)
{
  qDebug() << "View_Translate::View_Translate()";

  m_ui->setupUi(this);

 /* #ifdef Q_WS_MAEMO_5
    new QMaemo5KineticScroller(m_ui->textSource);
    new QMaemo5KineticScroller(m_ui->textTarget);
  #else*/
    m_ui->textSource->setProperty("FingerScrollable", true);
    m_ui->textTarget->setProperty("FingerScrollable", true);
 // #endif

  QString strSourceLabel(tr("Source language"));
  QString strTargetLabel(tr("Target language"));
  m_buttonSourceLang = NULL;
  m_buttonTargetLang = NULL;
 /* #ifdef Q_WS_MAEMO_5
    QMaemo5ValueButton* buttonSourceLang = new QMaemo5ValueButton(strSourceLabel, this);
    buttonSourceLang->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    m_buttonSourceLang = buttonSourceLang;
    QMaemo5ValueButton* buttonTargetLang = new QMaemo5ValueButton(strTargetLabel, this);
    buttonTargetLang->setValueLayout(QMaemo5ValueButton::ValueUnderTextCentered);
    m_buttonTargetLang = buttonTargetLang;
  #else
  */
    m_buttonSourceLang = new QPushButton(strSourceLabel, this);
    m_buttonTargetLang = new QPushButton(strTargetLabel, this);
//  #endif
  refreshLanguageButtons();
  m_ui->layoutSource->insertWidget(0, m_buttonSourceLang);
  m_ui->layoutTarget->insertWidget(0, m_buttonTargetLang);

  connect(m_context->ocr(), SIGNAL(done(QString)), this, SLOT(doneOcr(QString)));
  connect(m_context->ocr(), SIGNAL(error(QString)), this, SLOT(errorOcr(QString)));
  connect(m_context->translator(), SIGNAL(done(QString)), this, SLOT(doneTranslator(QString)));
  connect(m_context->translator(), SIGNAL(error(QString)), this, SLOT(errorTranslator(QString)));
  connect(m_context->settings(), SIGNAL(settingsChanged(quint32)), this, SLOT(onSettingsChanged(quint32)));

  connect(m_buttonSourceLang, SIGNAL(clicked()), this, SLOT(onSourceLang()));
  connect(m_buttonTargetLang, SIGNAL(clicked()), this, SLOT(onTargetLang()));
  connect(m_ui->translate, SIGNAL(clicked()), this, SLOT(onTranslate()));
  connect(m_ui->cancel, SIGNAL(clicked()), this, SLOT(onCancel()));
  connect(m_ui->textSource, SIGNAL(textChanged()), this, SLOT(onSourceTextChanged()));

  m_fromImage = false;
  m_processing = false;
}

//-----------------------------------------------------------------------------
View_Translate::~View_Translate()
{
  qDebug() << "View_Translate::~View_Translate()";

  delete m_ui;
}

//-----------------------------------------------------------------------------
void View_Translate::initialize(bool a_fromImage)
{
  qDebug() << "View_Translate::initialize(" << a_fromImage << ")";

  m_fromImage = a_fromImage;

  if(m_fromImage)
    onOcr();
  else
  {
    m_ui->cancel->hide();
    m_ui->translate->show();
    m_ui->translate->setEnabled(false);

    m_ui->progressSource->hide();
    m_ui->textSource->show();

    m_ui->progressTarget->hide();
    m_ui->textTarget->show();
  }
}

//-----------------------------------------------------------------------------
void View_Translate::onOcr()
{
  qDebug() << "View_Translate::onOcr()";

  m_processing = true;

  m_ui->cancel->show();
  m_ui->translate->hide();

  m_ui->textSource->hide();
  m_ui->textSource->clear();
  m_ui->progressSource->show();

  m_ui->textTarget->hide();
  m_ui->textTarget->clear();
  m_ui->progressTarget->show();

  // Run ocr
  m_context->ocr()->run( IMAGE_FILE_NAME,
                         m_context->settings()->langSource() );
}

//-----------------------------------------------------------------------------
void View_Translate::onTranslate()
{
  qDebug() << "View_Translate::onTranslate()";

  if(m_ui->textSource->toPlainText().simplified().isEmpty())
  {
    qDebug() << "View_Translate::onTranslate(): nothing to translate";
    doneTranslator(QString(""));
    return;
  }

  m_processing = true;

  m_ui->cancel->show();
  m_ui->translate->hide();

  m_ui->textTarget->hide();
  m_ui->textTarget->clear();
  m_ui->progressTarget->show();

  // Translate text
  m_context->translator()->run( m_ui->textSource->toPlainText(),
                                m_context->settings()->langSource(),
                                m_context->settings()->langTarget() );
}

//-----------------------------------------------------------------------------
void View_Translate::onCancel()
{
  if(!m_processing)
  {
    qDebug() << "View_Translate::onCancel(): not processing";
    return;
  }

  qDebug() << "View_Translate::onCancel(): start";

  toggleBusy(true);

  // Cancel ocr
  if(m_context->ocr()->cancel())
    m_ui->textSource->setPlainText("");
  m_ui->progressSource->hide();
  m_ui->textSource->show();

  // Cancel translator
  if(m_context->translator()->cancel())
    m_ui->textTarget->setPlainText("");
  m_ui->progressTarget->hide();
  m_ui->textTarget->show();

  m_ui->cancel->hide();
  m_ui->translate->show();
  m_ui->translate->setEnabled(!m_ui->textSource->toPlainText().simplified().isEmpty());

  m_processing = false;

  toggleBusy(false);

  qDebug() << "View_Translate::onCancel(): end";
}

//-----------------------------------------------------------------------------
void View_Translate::doneOcr(QString a_text)
{
  qDebug() << "View_Translate::doneOcr(" << a_text << ")";

  m_ui->progressSource->hide();
  m_ui->textSource->setPlainText(a_text);
  m_ui->textSource->show();

  // Translate text
  onTranslate();
}

//-----------------------------------------------------------------------------
void View_Translate::errorOcr(QString a_error)
{
  qDebug() << "View_Translate::errorOcr(" << a_error << ")";

  m_ui->cancel->hide();
  m_ui->translate->show();
  m_ui->translate->setEnabled(false);

  m_ui->progressSource->hide();
  m_ui->textSource->setPlainText("");
  m_ui->textSource->show();

  m_ui->progressTarget->hide();
  m_ui->textTarget->setPlainText("");
  m_ui->textTarget->show();

  m_processing = false;

  showError(a_error);
}

//-----------------------------------------------------------------------------
void View_Translate::doneTranslator(QString a_text)
{
  qDebug() << "View_Translate::doneTranslator(" << a_text << ")";

  m_ui->cancel->hide();
  m_ui->translate->show();
  m_ui->translate->setEnabled(!m_ui->textSource->toPlainText().simplified().isEmpty());

  m_ui->progressTarget->hide();
  m_ui->textTarget->setPlainText(a_text);
  m_ui->textTarget->show();

  m_processing = false;
}

//-----------------------------------------------------------------------------
void View_Translate::errorTranslator(QString a_error)
{
  qDebug() << "View_Translate::errorTranslator(" << a_error << ")";

  m_ui->cancel->hide();
  m_ui->translate->show();
  m_ui->translate->setEnabled(!m_ui->textSource->toPlainText().simplified().isEmpty());

  m_ui->progressTarget->hide();
  m_ui->textTarget->setPlainText("");
  m_ui->textTarget->show();

  m_processing = false;

  showError(a_error);
}

//-----------------------------------------------------------------------------
void View_Translate::onSourceLang()
{
  SettingsDialog::changeSourceLanguage(this, m_context->settings());
}

//-----------------------------------------------------------------------------
void View_Translate::onTargetLang()
{
  SettingsDialog::changeTargetLanguage(this, m_context->settings());
}

//-----------------------------------------------------------------------------
void View_Translate::onSettingsChanged(quint32 a_changed)
{
  // Refresh language buttons
  refreshLanguageButtons();

  // Cancel on going operations first
  if(m_processing)
    onCancel();

  // Re process image and/or translation
  if(!m_processing)
  {
    if(m_fromImage)
    {
      if(a_changed & Settings::ChangeFlag_LangSource)
        onOcr();
      else if(a_changed & Settings::ChangeFlag_LangTarget)
        onTranslate();
    }
    else
    {
      if(a_changed & Settings::ChangeFlag_LangSource || a_changed & Settings::ChangeFlag_LangTarget)
        onTranslate();
    }
  }
}

//-----------------------------------------------------------------------------
void View_Translate::onSourceTextChanged()
{
  m_ui->translate->setEnabled(!m_processing && !m_ui->textSource->toPlainText().simplified().isEmpty());
}

//-----------------------------------------------------------------------------
void View_Translate::refreshLanguageButtons()
{
 /* #ifdef Q_WS_MAEMO_5
    if(QMaemo5ValueButton* button = qobject_cast<QMaemo5ValueButton*>(m_buttonSourceLang))
      button->setValueText(Language::toString(m_context->settings()->langSource()));
    if(QMaemo5ValueButton* button = qobject_cast<QMaemo5ValueButton*>(m_buttonTargetLang))
      button->setValueText(Language::toString(m_context->settings()->langTarget()));
  #else*/
    QString strSourceLabel(tr("Source language"));
    QString strTargetLabel(tr("Target language"));
    m_buttonSourceLang->setText(strSourceLabel + QString(": ") + Language::toString(m_context->settings()->langSource()));
    m_buttonTargetLang->setText(strTargetLabel + QString(": ") + Language::toString(m_context->settings()->langTarget()));
  //#endif
}

//-----------------------------------------------------------------------------
