#include "s60trail.h"
#include <qboxlayout.h>
#include <QPixmap>
#include <QLabel>
#include <QDebug>
#include <QStackedWidget>
#include <QAction>
#include <QScrollArea>
#include <QToolButton>
#include <QApplication>
#include <QHBoxLayout>

#ifdef Q_WS_MAEMO_5
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>
#endif

#include "traillogin.h"
#include "trailmainmenu.h"
#include "trailtripslist.h"
#include "trailtripview.h"
#include "trailgpxrecorder.h"
#include "TrailCreateTrip.h"
#include "TrailTripSearch.h"
#include "TrailGPXView.h"

S60Trail::S60Trail(QWidget *parent) : QWidget(parent)
{
    qDebug("S60Trail");
    QCoreApplication::setOrganizationName("PGZ");
    QCoreApplication::setOrganizationDomain("piggz.co.uk");
    QCoreApplication::setApplicationName("EveryTrail");

    setWindowTitle("PGZ Every Trail");

    setContentsMargins(0,0,0,0);
    m_layout = new QVBoxLayout();
    m_titleLayout = new QHBoxLayout();

    m_logo = new QPixmap(":images/logo2.png");

    QLabel *lab = new QLabel();
    lab->setPixmap(*m_logo);

    lab->setAlignment(Qt::AlignCenter);

    m_titleLayout->addWidget(lab);
    m_layout->addLayout(m_titleLayout);

    //Create a stack widget to house all the main widgets, and add it to the scroll area
    m_stack = new QStackedWidget(this);
    m_stack->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    //m_scroll->setWidget(m_stack);
    m_layout->addWidget(m_stack);
    setLayout(m_layout);

    connect(&m_interface, SIGNAL(busy(bool)), this, SLOT(slotBusy(bool)));

    //Create widgets for various screens
    m_login = new TrailLogin(&m_interface, this);
    m_gpxMapView = new TrailGPXView(&m_interface, this);
    m_menu = new TrailMainMenu(&m_interface, this);
    m_mytrips = new TrailTripsList(&m_interface, this);
    m_tripview = new TrailTripView(&m_interface, m_gpxMapView, this);
    m_gpxrecorder = new TrailGPXRecorder(&m_interface, this);
    m_createtrip = new TrailCreateTrip(&m_interface, this);
    m_tripsearch = new TrailTripSearch(&m_interface, this);

    //Create a scroll area for the view trip widget
    m_viewTripScroll = new QScrollArea(this);
    m_viewTripScroll->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_viewTripScroll->setWidgetResizable(true);
    m_viewTripScroll->setFrameShape(QFrame::NoFrame);
    m_viewTripScroll->setWidget(m_tripview);

    //Create a scroll area for the create trip widget
    m_createTripScroll = new QScrollArea(this);
    m_createTripScroll->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_createTripScroll->setWidgetResizable(true);
    m_createTripScroll->setFrameShape(QFrame::NoFrame);
    m_createTripScroll->setWidget(m_createtrip);

    m_stack->addWidget(m_login);
    m_stack->addWidget(m_menu);
    m_stack->addWidget(m_mytrips);
    m_stack->addWidget(m_viewTripScroll);
    m_stack->addWidget(m_gpxrecorder);
    m_stack->addWidget(m_createTripScroll);
    m_stack->addWidget(m_tripsearch);
    m_stack->addWidget(m_gpxMapView);

    //Navigation Signals
    connect(m_login, SIGNAL(loggedIn()), this, SLOT(slotLoggedIn()));
    connect(m_menu, SIGNAL(navigateRecordTrip()), this, SLOT(slotRecordTrip()));
    connect(m_menu, SIGNAL(navigateCreateTrip()), this, SLOT(slotCreateTrip()));
    connect(m_menu, SIGNAL(navigateSearch()), this, SLOT(slotSearch()));
    connect(m_mytrips, SIGNAL(viewTrip(int)), this, SLOT(slotViewTrip(int)));
    connect(&m_interface, SIGNAL(tripsLoaded(bool)), this, SLOT(slotGotTrips(bool)));
    connect(m_tripview, SIGNAL(navigateViewMap()), this, SLOT(slotViewMap()));

    //Back Button
    m_backAction = new QAction(QIcon(":/images/dialog-close.png"), "Back", this);
    m_backAction->setSoftKeyRole(QAction::NegativeSoftKey);
    connect(m_backAction, SIGNAL(triggered()), this, SLOT(slotBack()));
    addAction(m_backAction);

    //Minimize Button
    m_minimizeAction = new QAction(QIcon(":/images/gps-red.png"), "Minimize", this);
    m_minimizeAction->setSoftKeyRole(QAction::NegativeSoftKey);
    connect(m_minimizeAction, SIGNAL(triggered()), this, SLOT(slotMinimize()));
    addAction(m_minimizeAction);

#ifndef Q_OS_SYMBIAN
    m_backButton = new QToolButton(this);
    m_backButton->setDefaultAction(m_backAction);
    m_backButton->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred);
    m_titleLayout->addWidget(m_backButton);
#endif

    m_minimizeButton = new QToolButton(this);
    m_minimizeButton->setDefaultAction(m_minimizeAction);
    m_minimizeButton->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred);
    m_titleLayout->insertWidget(0, m_minimizeButton);

    connect(m_interface.location(), SIGNAL(positionUpdated(const QGeoPositionInfo&)), this, SLOT(positionUpdated(const QGeoPositionInfo&)));
    connect(m_minimizeAction, SIGNAL(triggered()), this, SLOT(slotMinimize()));

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif

#ifdef Q_OS_SYMBIAN
    m_progress = new QProgressDialog("Please Wait", QString(), 0, 0, this);
    m_progress->setWindowModality(Qt::WindowModal);
#endif

    //toggleRightSoftKey(false, this);
    m_login->setFocus();
}

S60Trail::~S60Trail()
{

}

void S60Trail::slotLoggedIn()
{
    qDebug() << "Logged In!";
    m_previous.push(m_stack->currentWidget());
    m_backAction->setIcon(QIcon(":/images/go-previous-view.png"));
    m_stack->setCurrentWidget(m_menu);
}

void S60Trail::slotGotTrips(bool ok)
{
    qDebug() << "slotGotTrips";
    if (ok) {
        m_previous.push(m_stack->currentWidget());
        m_backAction->setIcon(QIcon(":/images/go-previous-view.png"));
        m_stack->setCurrentWidget(m_mytrips);
    }
    //toggleRightSoftKey(true, this);
}

void S60Trail::slotViewTrip(int t)
{
    qDebug() << "slotViewTrip";
    m_tripview->setTrip(*(m_interface.trip(t)));
    m_previous.push(m_stack->currentWidget());
    m_backAction->setIcon(QIcon(":/images/go-previous-view.png"));
    m_stack->setCurrentWidget(m_viewTripScroll);
}

void S60Trail::slotRecordTrip()
{
    m_previous.push(m_stack->currentWidget());
    m_backAction->setIcon(QIcon(":/images/go-previous-view.png"));
    m_stack->setCurrentWidget(m_gpxrecorder);
}

void S60Trail::slotCreateTrip()
{
    m_previous.push(m_stack->currentWidget());
    m_backAction->setIcon(QIcon(":/images/go-previous-view.png"));
    m_stack->setCurrentWidget(m_createTripScroll);
}

void S60Trail::slotSearch()
{
    m_previous.push(m_stack->currentWidget());
    m_backAction->setIcon(QIcon(":/images/go-previous-view.png"));
    m_stack->setCurrentWidget(m_tripsearch);
}

void S60Trail::slotBack()
{
    if (!m_previous.isEmpty()) {
        QWidget *previous = m_previous.pop();
        if (previous)
            m_stack->setCurrentWidget(previous);

        if (m_previous.isEmpty()) {
            m_backAction->setIcon(QIcon(":/images/dialog-close.png"));
        }
    } else {
        QApplication::quit();
    }
    return;
}

void S60Trail::toggleRightSoftKey( bool back, QWidget* focusWidget )
{
    qDebug() << "toggleRightSoftKey";
    if( back ) {
      connect(m_backAction, SIGNAL(triggered()), focusWidget, SLOT(slotBack()));
      addAction( m_backAction );
    } else {
      disconnect( m_backAction );
      removeAction( m_backAction );
    }
}

void S60Trail::slotViewMap()
{
    m_previous.push(m_stack->currentWidget());
    m_stack->setCurrentWidget(m_gpxMapView);
}

void S60Trail::slotBusy(bool busy)
{
    if (busy)
        showBusyIndicator();
    else
        hideBusyIndicator();
}

void S60Trail::showBusyIndicator()
{
#if defined(Q_WS_MAEMO_5)
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
    setEnabled(false);
#elif defined(Q_OS_SYMBIAN)
    m_progress->show();
#else
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
    setEnabled(false);
#endif
}

void S60Trail::hideBusyIndicator()
{
#if defined(Q_WS_MAEMO_5)
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
    setEnabled(true);
#elif defined(Q_OS_SYMBIAN)
    m_progress->hide();
#else
    QApplication::restoreOverrideCursor();
    setEnabled(true);
#endif
}

void S60Trail::positionUpdated(const QGeoPositionInfo& pos)
{
    if (pos.attribute(QGeoPositionInfo::HorizontalAccuracy) <= 50) {
        m_minimizeAction->setIcon(QIcon(":/images/gps-green.png"));
    } else if (pos.attribute(QGeoPositionInfo::HorizontalAccuracy) <= 100) {
        m_minimizeAction->setIcon(QIcon(":/images/gps-amber.png"));
    } else {
        m_minimizeAction->setIcon(QIcon(":/images/gps-red.png"));
    }
}

void S60Trail::slotMinimize()
{
#ifdef Q_WS_MAEMO_5
    QDBusConnection c = QDBusConnection::sessionBus();
    QDBusMessage m = QDBusMessage::createSignal("/", "com.nokia.hildon_desktop", "exit_app_view");
    c.send(m);
    c.send(m);
#else
    showMinimized();
#endif

}
