#include "trailgpxrecorder.h"
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QPushButton>
#include <QTableView>
#include <QHeaderView>
#include <QtLocation/QGeoPositionInfoSource>
#include <QtLocation/QGeoPositionInfo>
#include <QDebug>
#include <QFile>
#include <QTextStream>
#include <QResizeEvent>
#include <QFileDialog>
#include "trailinterface.h"

TrailGPXRecorder::TrailGPXRecorder(TrailInterface *interface, QWidget *parent) :
    QWidget(parent), m_model(0), m_interface(interface)
{
    //Layout all the widgets
    m_layout = new QVBoxLayout(this);

    m_curPos = new QLabel("<b>Current Position:</b>", this);
    m_layout->addWidget(m_curPos);
    
    m_curLongitude = new QLabel("<b>Longitude: </b>unavailable", this);
    m_curLongitude->setFocusPolicy(Qt::TabFocus);
    m_layout->addWidget(m_curLongitude);
    
    m_curLatitude = new QLabel("<b>Latitude: </b>unavailable", this);
    m_curLatitude->setFocusPolicy(Qt::TabFocus);
    m_layout->addWidget(m_curLatitude);

    m_btnLayout = new QHBoxLayout();
    
    m_record = new QPushButton(QIcon(":/images/media-record.png"), QString(), this);
    m_stop = new QPushButton(QIcon(":/images/media-playback-stop.png"), QString(), this);
    m_save = new QPushButton(QIcon(":/images/document-save.png"),QString(), this);
   
    m_btnLayout->addWidget(m_record);
    m_btnLayout->addWidget(m_stop);
    m_btnLayout->addWidget(m_save);
    m_layout->addLayout(m_btnLayout);
    
    m_gpxList = new QTableView(this);
    m_gpxList->horizontalHeader()->setVisible(true);
    m_gpxList->horizontalHeader()->setResizeMode(QHeaderView::Stretch);
    m_layout->addWidget(m_gpxList);
    
    connect(m_save, SIGNAL(clicked(bool)), this, SLOT(slotSave()));
    connect(m_record, SIGNAL(clicked(bool)), this, SLOT(slotRecord()));
    connect(m_stop, SIGNAL(clicked(bool)), this, SLOT(slotStop()));
    
    //Create the location source
    m_location = m_interface->location();
    
    if (m_location) {
        m_model = new TrailTrackModel(this);
        
        m_gpxList->setModel(m_model);

        connect(m_location, SIGNAL(positionUpdated(const QGeoPositionInfo&)), m_model, SLOT(positionUpdated(const QGeoPositionInfo&)));
        connect(m_location, SIGNAL(positionUpdated(const QGeoPositionInfo&)), this, SLOT(positionUpdated(const QGeoPositionInfo&)));
    }
}

TrailGPXRecorder::~TrailGPXRecorder()
{

}

void TrailGPXRecorder::slotRecord()
{
    if (m_model) {
        m_model->clear();
        m_model->setRecord(true);
    }
}

void TrailGPXRecorder::slotSave()
{
    if (!m_model) {
        return;
    }
    qDebug() << "Saving";

    QString filename = QFileDialog::getSaveFileName(this, tr("Save File"), QString(), QString("*.gpx"));

    if (!filename.isEmpty()) {
        if (!filename.endsWith(".gpx", Qt::CaseInsensitive)) {
            filename += ".gpx";
        }

        QFile data(filename);
        if (data.open(QFile::WriteOnly | QFile::Truncate)) {
            QTextStream t(&data);
            t << m_model->toText();
            data.close();
        }
    }
}

void TrailGPXRecorder::slotStop()
{
    if (m_model) {
        m_model->setRecord(false);
    }
}

void TrailGPXRecorder::positionUpdated(const QGeoPositionInfo& pos)
{
    if (width() > height()) {
        m_curPos->setText(QString("<b>Current Position: </b>%1:%2").arg(pos.coordinate().latitude()).arg(pos.coordinate().longitude()));
    } else {
        m_curLatitude->setText(QString("<b>Latitude: </b>%1").arg(pos.coordinate().latitude()));
        m_curLongitude->setText(QString("<b>Longitude: </b>%1").arg(pos.coordinate().longitude()));
    }
}

void TrailGPXRecorder::resizeEvent(QResizeEvent *event)
{
    if (event->size().width() > event->size().height()) {
        //Landscape
        m_curLongitude->hide();
        m_curLatitude->hide();
    } else {
        //Portrait
        m_curLongitude->show();
        m_curLatitude->show();
        m_curPos->setText("<b>Current Position:</b>");
    }
}
