/*
 * This file is part of Personal Photo Frame.
 *
 * Copyright (C) 2009 Andrew Olmsted. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
*  
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/* personal-photo-frame.c */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "personal-photo-frame.h"
#include <hildon/hildon.h>
#include <hildon/hildon-fm.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <libintl.h>
//#include <conic.h>

#define HOME_DIR g_get_home_dir()

#define _(String) dgettext("hildon-libs", String)

#define PERSONAL_PHOTO_FRAME_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj,\
							                         PERSONAL_TYPE_PHOTO_FRAME,\
							                         PersonalPhotoFramePrivate))
													 
struct _PersonalPhotoFramePrivate
{
    GtkWidget *homeWidget;
	GtkWidget *event;
	
	gboolean singleMode;
	gchar *singleFilename;
	gchar *slideshowDirname;
	gchar *slideshowFilename;
	gint slideshowNum;
	gint slideshowDuration;
	gboolean slideshowRandom;
	gboolean showSlideshowControls;
	gboolean isPressed;
	
	GtkWidget *singleModeBox;
	GtkWidget *slideshowModeBox;
	
	GdkPixbuf *image;
	GdkPixbuf *forwardButton;
	GdkPixbuf *backButton;
	
	GtkWidget *settingsDialog;
	
	gboolean isShown;
	guint eventID;
};

HD_DEFINE_PLUGIN_MODULE (PersonalPhotoFrame, personal_photo_frame, HD_TYPE_HOME_PLUGIN_ITEM);

gboolean personal_photo_frame_timeout (PersonalPhotoFrame *self);
void personal_photo_frame_update_content (PersonalPhotoFrame *self);

void personal_photo_frame_remove_timers (PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	if (self->priv->eventID != 0) {
		g_source_remove (self->priv->eventID);
		self->priv->eventID = 0;
	}
}

void personal_photo_frame_add_timers (PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	if (self->priv->eventID == 0) {
		if (self->priv->slideshowDuration == 1) {
			self->priv->eventID = g_timeout_add_seconds (30, (GSourceFunc)personal_photo_frame_timeout, self);
		} else if (self->priv->slideshowDuration == 2) {
			self->priv->eventID = g_timeout_add_seconds (60, (GSourceFunc)personal_photo_frame_timeout, self);
		} else if (self->priv->slideshowDuration == 3) {
			self->priv->eventID = g_timeout_add_seconds (300, (GSourceFunc)personal_photo_frame_timeout, self);
		} else if (self->priv->slideshowDuration == 4) {
			self->priv->eventID = g_timeout_add_seconds (600, (GSourceFunc)personal_photo_frame_timeout, self);
		} else if (self->priv->slideshowDuration == 5) {
			self->priv->eventID = g_timeout_add_seconds (1800, (GSourceFunc)personal_photo_frame_timeout, self);
		} else if (self->priv->slideshowDuration == 6) {
			self->priv->eventID = g_timeout_add_seconds (3600, (GSourceFunc)personal_photo_frame_timeout, self);
		}
	}
}

void personal_photo_frame_read_settings ( PersonalPhotoFrame *self )
{
	//g_warning("%s", __PRETTY_FUNCTION__);
    gchar *filename;
    gboolean fileExists;
    GKeyFile *keyFile;

    keyFile = g_key_file_new();
    filename = g_strconcat (HOME_DIR, "/.personal_photo_frame", NULL);
    fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);

    if (fileExists) {
        GError *error=NULL;

		self->priv->singleMode = g_key_file_get_boolean (keyFile, "config", "singleMode", &error);
		if (error) {
			self->priv->singleMode = TRUE;
			g_error_free (error);
			error = NULL;
		}
        self->priv->singleFilename = g_key_file_get_string (keyFile, "config", "singleFilename", &error);
        if (error) {
            self->priv->singleFilename = "/usr/share/icons/hicolor/scalable/hildon/personal-photo-frame.png";
            g_error_free (error);
            error = NULL;
        }
		self->priv->slideshowDirname = g_key_file_get_string (keyFile, "config", "slideshowDirname", &error);
		if (error) {
			self->priv->slideshowDirname = "";
			g_error_free (error);
			error = NULL;
		}
		self->priv->slideshowNum = g_key_file_get_integer (keyFile, "config", "slideshowNum", &error);
		if (error) {
			self->priv->slideshowNum = 0;
			g_error_free (error);
			error = NULL;
		}
		self->priv->slideshowDuration = g_key_file_get_integer (keyFile, "config", "slideshowDuration", &error);
		if (error) {
			self->priv->slideshowDuration = 0;
			g_error_free (error);
			error = NULL;
		}
		self->priv->slideshowRandom = g_key_file_get_boolean (keyFile, "config", "slideshowRandom", &error);
		if (error) {
			self->priv->slideshowRandom = FALSE;
			g_error_free (error);
			error = NULL;
		}
		self->priv->showSlideshowControls = g_key_file_get_boolean (keyFile, "config", "showSlideshowControls", &error);
		if (error) {
			self->priv->showSlideshowControls = TRUE;
			g_error_free (error);
			error = NULL;
		}
    } else {
		self->priv->singleMode = TRUE;
        self->priv->singleFilename = "/usr/share/icons/hicolor/scalable/hildon/personal-photo-frame.png";
		self->priv->slideshowDirname = "";
		self->priv->slideshowNum = 0;
		self->priv->slideshowDuration = 0;
		self->priv->slideshowRandom = FALSE;
		self->priv->showSlideshowControls = TRUE;
    }
	
    g_key_file_free (keyFile);
    g_free (filename);
}

void personal_photo_frame_write_settings (PersonalPhotoFrame *self)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	GKeyFile *keyFile;
	gchar *fileData;
	FILE *iniFile;
	gsize size;
	gchar *filename;
	
	keyFile = g_key_file_new();
	
	g_key_file_set_boolean (keyFile, "config", "singleMode", self->priv->singleMode);
	g_key_file_set_string (keyFile, "config", "singleFilename", self->priv->singleFilename);
	g_key_file_set_string (keyFile, "config", "slideshowDirname", self->priv->slideshowDirname);
	g_key_file_set_integer (keyFile, "config", "slideshowNum", self->priv->slideshowNum);
	g_key_file_set_integer (keyFile, "config", "slideshowDuration", self->priv->slideshowDuration);
	g_key_file_set_boolean (keyFile, "config", "slideshowRandom", self->priv->slideshowRandom);
	g_key_file_set_boolean (keyFile, "config", "showSlideshowControls", self->priv->showSlideshowControls);
	
	filename = g_strconcat (HOME_DIR, "/.personal_photo_frame", NULL);
	fileData = g_key_file_to_data (keyFile, &size, NULL);
	iniFile = fopen (filename, "w");
	fputs (fileData, iniFile);
	fclose (iniFile);
	g_key_file_free (keyFile);
	g_free (fileData);
	g_free (filename);
}

void personal_photo_frame_button_press (GtkWidget *widget, GdkEventButton *event, PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	self->priv->isPressed = TRUE;
	gtk_widget_queue_draw (GTK_WIDGET (self));
}

void personal_photo_frame_button_release (GtkWidget *widget, GdkEventButton *event, PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	self->priv->isPressed = FALSE;
	
	if (!self->priv->singleMode && self->priv->showSlideshowControls) {
		gdouble x_coord, y_coord;
	
		if (gdk_event_get_coords (event, &x_coord, &y_coord)) {
			if (y_coord > 310.0 - 48.0) {
				if (x_coord > 310.0 - 48.0) {
					//self->priv->slideshowNum++;
					personal_photo_frame_update_content (self);
					personal_photo_frame_remove_timers (self);
					personal_photo_frame_add_timers (self);
				} else if (x_coord < 5.0 + 48.0) {
					self->priv->slideshowNum-=2;
					personal_photo_frame_update_content (self);
					personal_photo_frame_remove_timers (self);
					personal_photo_frame_add_timers (self);
				} else {
					DBusConnection *conn = dbus_bus_get(DBUS_BUS_SESSION, NULL);
					if (self->priv->singleMode) {
						hildon_mime_open_file (conn, self->priv->singleFilename);
					} else {
						hildon_mime_open_file (conn, self->priv->slideshowFilename);
					}
					dbus_connection_unref(conn);
				}
			} else {
				DBusConnection *conn = dbus_bus_get(DBUS_BUS_SESSION, NULL);
				if (self->priv->singleMode) {
					hildon_mime_open_file (conn, self->priv->singleFilename);
				} else {
					hildon_mime_open_file (conn, self->priv->slideshowFilename);
				}
				dbus_connection_unref(conn);
			}
		}
	} else {
		DBusConnection *conn = dbus_bus_get(DBUS_BUS_SESSION, NULL);
		if (self->priv->singleMode) {
			hildon_mime_open_file (conn, self->priv->singleFilename);
		} else {
			hildon_mime_open_file (conn, self->priv->slideshowFilename);
		}
		dbus_connection_unref(conn);
	}
	
	gtk_widget_queue_draw (GTK_WIDGET (self));
}

void personal_photo_frame_leave_event (GtkWidget *widget, GdkEventCrossing *event, PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	self->priv->isPressed = FALSE;
	gtk_widget_queue_draw (GTK_WIDGET (self));
}

void personal_photo_frame_content_create (PersonalPhotoFrame *self)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	self->priv->event = gtk_event_box_new ();
	gtk_event_box_set_visible_window (GTK_EVENT_BOX (self->priv->event), FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (self->priv->event), 5);

	gtk_box_pack_start (GTK_BOX (self->priv->homeWidget), self->priv->event, TRUE, TRUE, 0);
	
	self->priv->image = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 305, 305);
	gdk_pixbuf_fill (self->priv->image, 0x00000000);
	self->priv->backButton = gdk_pixbuf_new_from_file ("/usr/share/icons/hicolor/48x48/hildon/general_back.png", NULL);
	self->priv->forwardButton = gdk_pixbuf_new_from_file ("/usr/share/icons/hicolor/48x48/hildon/general_forward.png", NULL);
	
	g_signal_connect (self->priv->event, "button-release-event", G_CALLBACK (personal_photo_frame_button_release), self);
	g_signal_connect (self->priv->event, "button-press-event", G_CALLBACK (personal_photo_frame_button_press), self);
	g_signal_connect (self->priv->event, "leave-notify-event", G_CALLBACK (personal_photo_frame_leave_event), self);
	
    gtk_widget_show_all (self->priv->homeWidget);
}

gboolean personal_photo_frame_timeout (PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	personal_photo_frame_update_content (self);
	
	if (!self->priv->isShown) {
		personal_photo_frame_remove_timers (self);
	}
	return self->priv->isShown;
}

void personal_photo_frame_update_content (PersonalPhotoFrame *self)
{
	//g_warning("%s", __PRETTY_FUNCTION__);

	if (self->priv->singleMode) {
		g_object_unref (self->priv->image);
		self->priv->image = gdk_pixbuf_new_from_file_at_scale (self->priv->singleFilename, 305, 305, TRUE, NULL);
	} else {
		GDir *dir = g_dir_open (self->priv->slideshowDirname, 0, NULL);
		
		if (dir) {
			gboolean found = FALSE;
			gint looped = 0;
			
			while ((!found) && (looped < 2)) {
				int i;
				for (i=0; i<self->priv->slideshowNum; i++) {
					g_dir_read_name (dir);
				}
				self->priv->slideshowNum++;
				gchar *filename = g_strdup (g_dir_read_name (dir));
				if (!filename) {
					self->priv->slideshowNum = 0;
					looped++;
					g_dir_rewind (dir);
					g_object_unref (self->priv->image);
					self->priv->image = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, 305, 305);
					gdk_pixbuf_fill (self->priv->image, 0x00000000);
					gtk_widget_queue_draw (self);
				} else {
					self->priv->slideshowFilename = g_strconcat (self->priv->slideshowDirname, "/", filename, NULL);

					GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file_at_scale (self->priv->slideshowFilename, 305, 305, TRUE, NULL);
					if (pixbuf) {
						g_object_unref (self->priv->image);
						self->priv->image = gdk_pixbuf_copy (pixbuf);
						g_object_unref (pixbuf);
						found = TRUE;
						personal_photo_frame_write_settings (self);
						gtk_widget_queue_draw (self);
					}
					g_free (filename);
				}
			}
			g_dir_close (dir);
		} else {
			g_object_unref (self->priv->image);
			self->priv->image = gdk_pixbuf_new_from_file_at_scale ("/usr/share/icons/hicolor/scalable/hildon/personal-photo-frame.png", 305, 305, TRUE, NULL);
		}
	}
}

void personal_photo_from_single_photo_select (GtkWidget *button, PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	GtkWidget *dialog = hildon_file_chooser_dialog_new (GTK_WINDOW (self->priv->settingsDialog), GTK_FILE_CHOOSER_ACTION_OPEN);
	
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		gchar *filename = gtk_file_chooser_get_filename (dialog);
		hildon_button_set_value (HILDON_BUTTON (button), filename);
		g_free (filename);
	}
	gtk_widget_destroy (dialog);
}

void personal_photo_frame_slideshow_dir_select (GtkWidget *button, PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	GtkWidget *dialog = hildon_file_chooser_dialog_new (GTK_WINDOW (self->priv->settingsDialog), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);
	
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		gchar *dirname = gtk_file_chooser_get_filename (dialog);
		hildon_button_set_value (HILDON_BUTTON (button), dirname);
		g_free (dirname);
	}
	gtk_widget_destroy (dialog);
}

void personal_photo_frame_mode_toggle (GtkToggleButton *toggleButton, PersonalPhotoFrame *self)
{
	//g_warning ("%s", __PRETTY_FUNCTION__);
	if (gtk_toggle_button_get_active (toggleButton)) {
		gtk_widget_show_all (self->priv->singleModeBox);
		gtk_widget_hide_all (self->priv->slideshowModeBox);
	} else {
		gtk_widget_hide_all (self->priv->singleModeBox);
		gtk_widget_show_all (self->priv->slideshowModeBox);
	}
}

void personal_photo_frame_settings (HDHomePluginItem *hitem, PersonalPhotoFrame *self)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	self->priv->settingsDialog = gtk_dialog_new_with_buttons ("Personal Photo Frame Settings",
        NULL, 0, _("wdgt_bd_save"), GTK_RESPONSE_ACCEPT, NULL);
    GtkWidget *content_area = gtk_dialog_get_content_area (GTK_DIALOG (self->priv->settingsDialog));
	GtkSizeGroup *group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));
	
	self->priv->singleModeBox = gtk_vbox_new (FALSE, 0);
	self->priv->slideshowModeBox = gtk_vbox_new (FALSE, 0);
	
	GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
	GtkWidget *singleModeToggle = hildon_gtk_radio_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, NULL);
	gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (singleModeToggle), FALSE);
	gtk_button_set_label (GTK_BUTTON (singleModeToggle), "Single Photo Mode");
	GtkWidget *slideshowModeToggle = hildon_gtk_radio_button_new_from_widget (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, GTK_RADIO_BUTTON (singleModeToggle));
	gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (slideshowModeToggle), FALSE);
	gtk_button_set_label (GTK_BUTTON (slideshowModeToggle), "Slide Show Mode");
	gtk_box_pack_start (GTK_BOX (hbox), singleModeToggle, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), slideshowModeToggle, FALSE, FALSE, 0);
	
	GtkWidget *singlePhotoButton = hildon_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title (HILDON_BUTTON (singlePhotoButton), "Select photo");
	hildon_button_set_value (HILDON_BUTTON (singlePhotoButton), self->priv->singleFilename);
	gtk_box_pack_start (GTK_BOX (self->priv->singleModeBox), singlePhotoButton, FALSE, FALSE, 0);
	g_signal_connect (singlePhotoButton, "clicked", G_CALLBACK (personal_photo_from_single_photo_select), self);
	
	GtkWidget *slideshowDirButton = hildon_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title (HILDON_BUTTON (slideshowDirButton), "Select directory");
	hildon_button_set_value (HILDON_BUTTON (slideshowDirButton), self->priv->slideshowDirname);
	gtk_box_pack_start (GTK_BOX (self->priv->slideshowModeBox), slideshowDirButton, FALSE, FALSE, 0);
	g_signal_connect (slideshowDirButton, "clicked", G_CALLBACK (personal_photo_frame_slideshow_dir_select), self);
	
	GtkWidget *selector = hildon_touch_selector_new_text ();
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "when changing desktops");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "every 30 seconds");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "every minute");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "every 5 minutes");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "every 10 minutes");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "every 30 minutes");
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "every hour");
	hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, self->priv->slideshowDuration);
	
	GtkWidget *slideshowDuration = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title (HILDON_BUTTON (slideshowDuration), "Change photos");
	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (slideshowDuration), HILDON_TOUCH_SELECTOR (selector));
	gtk_box_pack_start (GTK_BOX (self->priv->slideshowModeBox), slideshowDuration, FALSE, FALSE, 0);
	
	GtkWidget *slideshowRandom = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label (GTK_BUTTON (slideshowRandom), "Display in random order");
	hildon_check_button_set_active (HILDON_CHECK_BUTTON (slideshowRandom), self->priv->slideshowRandom);
	//gtk_box_pack_start (GTK_BOX (self->priv->slideshowModeBox), slideshowRandom, FALSE, FALSE, 0);
	
	GtkWidget *showSlideshowControls = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label (GTK_BUTTON (showSlideshowControls), "Display slide show controls");
	hildon_check_button_set_active (HILDON_CHECK_BUTTON (showSlideshowControls), self->priv->showSlideshowControls);
	gtk_box_pack_start (GTK_BOX (self->priv->slideshowModeBox), showSlideshowControls, FALSE, FALSE, 0);
	
	gtk_box_pack_start (GTK_BOX (content_area), hbox, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (content_area), self->priv->slideshowModeBox, FALSE, FALSE, 0);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (singleModeToggle), self->priv->singleMode);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (slideshowModeToggle), !self->priv->singleMode);
	g_signal_connect (singleModeToggle, "toggled", G_CALLBACK (personal_photo_frame_mode_toggle), self);
	
	gtk_widget_show_all (self->priv->settingsDialog);
	gtk_box_pack_start (GTK_BOX (content_area), self->priv->singleModeBox, FALSE, FALSE, 0);
	personal_photo_frame_mode_toggle (singleModeToggle, self);
	
	if (gtk_dialog_run (GTK_DIALOG (self->priv->settingsDialog)) == GTK_RESPONSE_ACCEPT) {
		gchar *compare = g_strdup (self->priv->slideshowDirname);
		self->priv->singleMode = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (singleModeToggle));
		self->priv->singleFilename = g_strdup (hildon_button_get_value (HILDON_BUTTON (singlePhotoButton)));
		self->priv->slideshowDirname = g_strdup (hildon_button_get_value (HILDON_BUTTON (slideshowDirButton)));
		self->priv->slideshowDuration = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (selector), 0);
		self->priv->slideshowRandom = hildon_check_button_get_active (HILDON_CHECK_BUTTON (slideshowRandom));
		self->priv->showSlideshowControls = hildon_check_button_get_active (HILDON_CHECK_BUTTON (showSlideshowControls));
		
		if (!g_str_equal (compare, self->priv->slideshowDirname)) {
			self->priv->slideshowNum = 0;
		}
		g_free (compare);
		
		personal_photo_frame_write_settings (self);
		personal_photo_frame_update_content (self);
		
		personal_photo_frame_remove_timers (self);
		personal_photo_frame_add_timers (self);
	}
	gtk_widget_destroy (self->priv->settingsDialog);
}

static void personal_photo_frame_check_desktop (GObject *gobject, GParamSpec *pspec, PersonalPhotoFrame *self)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	gchar *name = pspec->name;

	g_object_get (gobject, name, &self->priv->isShown, NULL);
	if (self->priv->isShown) {
		if (self->priv->eventID == 0) {
			if (!self->priv->singleMode) {
				if (self->priv->slideshowDuration == 0) {
					personal_photo_frame_update_content (self);
				}
				personal_photo_frame_add_timers (self);
			}
		}
	}
}

static void personal_photo_frame_dispose (GObject *object)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	PersonalPhotoFrame *self = PERSONAL_PHOTO_FRAME (object);

	G_OBJECT_CLASS (personal_photo_frame_parent_class)->dispose (object);
}

static void personal_photo_frame_finalize (GObject *object)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	PersonalPhotoFrame *self = PERSONAL_PHOTO_FRAME (object);

	if (self->priv->eventID) {
		g_source_remove (self->priv->eventID);
	}
	g_object_unref (self->priv->forwardButton);
	g_object_unref (self->priv->backButton);
	
	personal_photo_frame_write_settings (self);

	G_OBJECT_CLASS (personal_photo_frame_parent_class)->finalize (object);
}

static void personal_photo_frame_realize (GtkWidget *widget)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	GdkScreen *screen;

	screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	GTK_WIDGET_CLASS (personal_photo_frame_parent_class)->realize (widget);
}

static gboolean personal_photo_frame_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	PersonalPhotoFrame *self = PERSONAL_PHOTO_FRAME (widget);
	cairo_t *cr;
	
	cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));

	GdkColor color;
	if (!self->priv->isPressed) {
		gtk_style_lookup_color (gtk_rc_get_style(widget), "DefaultBackgroundColor", &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.75);
	} else {
		gtk_style_lookup_color (gtk_rc_get_style(widget), "SelectionColor", &color);
		cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.6);
	}
	
	gint width, height, x, y;
	gint radius = 5;
	width = widget->allocation.width;
	height = widget->allocation.height;
	x = widget->allocation.x;
	y = widget->allocation.y;

	cairo_move_to(cr, x + radius, y);
	cairo_line_to(cr, x + width - radius, y);
	cairo_curve_to(cr, x + width - radius, y, x + width, y, x + width,
				y + radius);
	cairo_line_to(cr, x + width, y + height - radius);
	cairo_curve_to(cr, x + width, y + height - radius, x + width,
				y + height, x + width - radius, y + height);
	cairo_line_to(cr, x + radius, y + height);
	cairo_curve_to(cr, x + radius, y + height, x, y + height, x,
				y + height - radius);
	cairo_line_to(cr, x, y + radius);
	cairo_curve_to(cr, x, y + radius, x, y, x + radius, y);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	cairo_fill_preserve(cr);
	
	gtk_style_lookup_color (gtk_rc_get_style(widget), "ActiveTextColor", &color);
	cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);
	cairo_set_line_width (cr, 1);
	cairo_stroke (cr);
	
	gint pixbufWidth=gdk_pixbuf_get_width(self->priv->image);
	gint pixbufHeight=gdk_pixbuf_get_height(self->priv->image);
	gdk_cairo_set_source_pixbuf (cr, self->priv->image, 5 + ((305-pixbufWidth)/2), 5 + ((305-pixbufHeight)/2));
	cairo_set_operator (cr, CAIRO_OPERATOR_OVER);
	if (!self->priv->isPressed) {
		cairo_paint (cr);
	} else {
		cairo_paint_with_alpha (cr, 0.75);
	}
	
	cairo_destroy(cr);
	
	if (!self->priv->singleMode && self->priv->showSlideshowControls) {
		cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));
		gdk_cairo_set_source_pixbuf (cr, self->priv->backButton, 5, 310 - 48);
		cairo_paint_with_alpha (cr, 0.5);
		
		gdk_cairo_set_source_pixbuf (cr, self->priv->forwardButton, 310 -48, 310 - 48);
		cairo_paint_with_alpha (cr, 0.5);
		cairo_destroy (cr);
	}
	
	return GTK_WIDGET_CLASS (personal_photo_frame_parent_class)->expose_event (widget, event);
}

static void personal_photo_frame_class_init (PersonalPhotoFrameClass *klass)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	object_class->dispose = personal_photo_frame_dispose;
	object_class->finalize = personal_photo_frame_finalize;
	
	widget_class->realize = personal_photo_frame_realize;
	widget_class->expose_event = personal_photo_frame_expose_event;

	g_type_class_add_private (klass, sizeof (PersonalPhotoFramePrivate));
}

static void personal_photo_frame_class_finalize (PersonalPhotoFrameClass *klass G_GNUC_UNUSED)
{
}

static void personal_photo_frame_init (PersonalPhotoFrame *self)
{
	//g_warning("%s", __PRETTY_FUNCTION__);
	self->priv = PERSONAL_PHOTO_FRAME_GET_PRIVATE (self);
	self->priv->eventID = 0;
	self->priv->isPressed = FALSE;
	
	hd_home_plugin_item_set_settings (&self->parent, TRUE);
	g_signal_connect (&self->parent, "show-settings", G_CALLBACK (personal_photo_frame_settings), self);

	gtk_window_set_default_size (GTK_WINDOW (self), 315, 315);
	
	self->priv->homeWidget = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (self), self->priv->homeWidget);
	gtk_widget_show (self->priv->homeWidget);

	GdkGeometry hints;
	hints.min_width = 315;
	hints.min_height = 315;
	hints.max_width = 315;
	hints.max_height = 315;
	gtk_window_set_geometry_hints (GTK_WINDOW (self), self->priv->homeWidget, &hints, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	
	personal_photo_frame_read_settings (self);
	personal_photo_frame_content_create (self);
	personal_photo_frame_add_timers (self);
	personal_photo_frame_update_content (self);
	
	g_signal_connect (self, "notify::is-on-current-desktop", G_CALLBACK (personal_photo_frame_check_desktop), self);
}

PersonalPhotoFrame* personal_photo_frame_new (void)
{
  return g_object_new (PERSONAL_TYPE_PHOTO_FRAME, NULL);
}
