/*
 * This file is part of Personal Dataplan Monitor.
 *
 * Copyright (C) 2009 Andrew Olmsted. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
*  
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */

/* personal-gprs-mon.c */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "personal-gprs-mon.h"
#include <hildon/hildon.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <libintl.h>
#include <gconf/gconf-client.h>

#define HOME_DIR g_get_home_dir()

#define GPRS_GCONF_PATH "/system/osso/connectivity/network_type/GPRS"
#define GPRS_GCONF_RX GPRS_GCONF_PATH  "/gprs_rx_bytes"
#define GPRS_GCONF_TX GPRS_GCONF_PATH  "/gprs_tx_bytes"
#define GPRS_GCONF_RESET GPRS_GCONF_PATH  "/gprs_reset_time"

#define _(String) dgettext("hildon-libs", String)

#define PERSONAL_GPRS_MON_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE (obj,\
							                         PERSONAL_TYPE_GPRS_MON,\
							                         PersonalGprsMonPrivate))

struct _PersonalGprsMonPrivate
{
    GtkWidget *homeWidget;
    GtkWidget *contents;
	
	GtkWidget *titleLabel;
	GtkWidget *uploadLabel;
	GtkWidget *uploadContent;
	GtkWidget *downloadLabel;
	GtkWidget *downloadContent;
	
	GtkWidget *resetLabel;
	GtkWidget *resetContent;
	GtkWidget *resetBox;
	
	GConfClient *client;
	gint refresh;
	gboolean binaryPrefix;
	gboolean showResetDate;
	
	guint eventID;
};

HD_DEFINE_PLUGIN_MODULE (PersonalGprsMon, personal_gprs_mon, HD_TYPE_HOME_PLUGIN_ITEM);

gboolean personal_gprs_mon_update_content (PersonalGprsMon *self);

void personal_gprs_mon_read_settings ( PersonalGprsMon *self )
{
	//g_warning("personal_gprs_mon_read_settings");	
    gchar *filename;
    gboolean fileExists;
    GKeyFile *keyFile;

    keyFile = g_key_file_new();
    filename = g_strconcat (HOME_DIR, "/.personal_gprs_mon", NULL);
    fileExists = g_key_file_load_from_file (keyFile, filename, G_KEY_FILE_KEEP_COMMENTS, NULL);

    if (fileExists) {
        GError *error=NULL;

        self->priv->refresh = g_key_file_get_integer (keyFile, "config", "refresh", &error);
        if (error) {
            self->priv->refresh = 5;
            g_error_free (error);
            error = NULL;
        }
		self->priv->binaryPrefix = g_key_file_get_boolean (keyFile, "config", "binaryPrefix", &error);
		if (error) {
			self->priv->binaryPrefix = FALSE;
			g_error_free (error);
			error = NULL;
		}
		self->priv->showResetDate = g_key_file_get_boolean (keyFile, "config", "showResetDate", &error);
		if (error) {
			self->priv->showResetDate = FALSE;
			g_error_free (error);
			error = NULL;
		}
    } else {
        self->priv->refresh = 5;
		self->priv->binaryPrefix = FALSE;
		self->priv->showResetDate = FALSE;
    }

    g_key_file_free (keyFile);
    g_free (filename);
}

void personal_gprs_mon_write_settings (PersonalGprsMon *self)
{
	//g_warning ("personal_gprs_mon_write_settings");
	GKeyFile *keyFile;
	gchar *fileData;
	FILE *iniFile;
	gsize size;
	GtkTreeIter iter;
	gboolean exists;
	gchar *filename;
	gint i=0;
	
	keyFile = g_key_file_new();
		
	g_key_file_set_integer (keyFile, "config", "refresh", self->priv->refresh);
	g_key_file_set_boolean (keyFile, "config", "binaryPrefix", self->priv->binaryPrefix);
	g_key_file_set_boolean (keyFile, "config", "showResetDate", self->priv->showResetDate);
	
	filename = g_strconcat (HOME_DIR, "/.personal_gprs_mon", NULL);
	fileData = g_key_file_to_data (keyFile, &size, NULL);
	iniFile = fopen (filename, "w");
	fputs (fileData, iniFile);
	fclose (iniFile);
	g_key_file_free (keyFile);
	g_free (fileData);
	g_free (filename);
}

void personal_gprs_mon_content_create (PersonalGprsMon *self)
{
	//g_warning ("personal_gprs_mon_content_create");
	self->priv->contents = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (self->priv->contents), 0);
	GtkSizeGroup *group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));

	GtkWidget *image = gtk_image_new_from_icon_name ("personal-gprs-mon", GTK_ICON_SIZE_BUTTON);
    self->priv->titleLabel = gtk_label_new ("Dataplan Monitor");
	hildon_helper_set_logical_font (self->priv->titleLabel, "EmpSystemFont");
	GtkWidget *titleBox = gtk_hbox_new (FALSE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (titleBox), 0);
	gtk_box_pack_start (GTK_BOX (titleBox), image, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (titleBox), self->priv->titleLabel, TRUE, TRUE, 0);
	gtk_misc_set_alignment (GTK_MISC (image), 0.5, 1);
	gtk_misc_set_padding (GTK_MISC (image), 2, 0);
	gtk_misc_set_alignment (GTK_MISC (self->priv->titleLabel), 0, 0.8);
	
	self->priv->uploadLabel = gtk_label_new ("Uploaded");
	gtk_size_group_add_widget (group, self->priv->uploadLabel);
	self->priv->uploadContent = gtk_label_new ("0.000 B");
	GtkWidget *uploadBox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (uploadBox), self->priv->uploadLabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (uploadBox), self->priv->uploadContent, TRUE, TRUE, 0);
	hildon_helper_set_logical_font (self->priv->uploadLabel, "SmallSystemFont");
	hildon_helper_set_logical_font (self->priv->uploadContent, "SmallSystemFont");
	hildon_helper_set_logical_color (self->priv->uploadContent, GTK_RC_FG, GTK_STATE_NORMAL, "ActiveTextColor");
	gtk_misc_set_alignment (GTK_MISC (self->priv->uploadLabel), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (self->priv->uploadLabel), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_HALF);
	gtk_misc_set_alignment (GTK_MISC (self->priv->uploadContent), 1, 0.5);
	gtk_misc_set_padding (GTK_MISC (self->priv->uploadContent), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_HALF);
	
	self->priv->downloadLabel = gtk_label_new ("Downloaded");
	gtk_size_group_add_widget (group, self->priv->downloadLabel);
	self->priv->downloadContent = gtk_label_new ("0.000 B");
	GtkWidget *downloadBox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (downloadBox), self->priv->downloadLabel, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (downloadBox), self->priv->downloadContent, TRUE, TRUE, 0);	
	hildon_helper_set_logical_font (self->priv->downloadLabel, "SmallSystemFont");
	hildon_helper_set_logical_font (self->priv->downloadContent, "SmallSystemFont");
	hildon_helper_set_logical_color (self->priv->downloadContent, GTK_RC_FG, GTK_STATE_NORMAL, "ActiveTextColor");
	gtk_misc_set_alignment (GTK_MISC (self->priv->downloadLabel), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (self->priv->downloadLabel), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_HALF);
	gtk_misc_set_alignment (GTK_MISC (self->priv->downloadContent), 1, 0.5);
	gtk_misc_set_padding (GTK_MISC (self->priv->downloadContent), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_HALF);
	
	self->priv->resetLabel = gtk_label_new ("Last Reset");
	gtk_size_group_add_widget (group, self->priv->resetLabel);
	self->priv->resetContent = gtk_label_new ("Could not retrieve");
	self->priv->resetBox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (self->priv->resetBox), self->priv->resetLabel, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (self->priv->resetBox), self->priv->resetContent, TRUE, TRUE, 0);
	hildon_helper_set_logical_font (self->priv->resetLabel, "SmallSystemFont");
	hildon_helper_set_logical_font (self->priv->resetContent, "SmallSystemFont");
	hildon_helper_set_logical_color (self->priv->resetContent, GTK_RC_FG, GTK_STATE_NORMAL, "ActiveTextColor");
	gtk_misc_set_alignment (GTK_MISC (self->priv->resetLabel), 0, 0.5);
	gtk_misc_set_padding (GTK_MISC (self->priv->resetLabel), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_HALF);
	gtk_misc_set_alignment (GTK_MISC (self->priv->resetContent), 1, 0.5);
	gtk_misc_set_padding (GTK_MISC (self->priv->resetContent), HILDON_MARGIN_DEFAULT, HILDON_MARGIN_HALF);
	
	gtk_box_pack_start (GTK_BOX (self->priv->contents), titleBox, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (self->priv->contents), uploadBox, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (self->priv->contents), downloadBox, FALSE, FALSE, 0);	
	gtk_box_pack_start (GTK_BOX (self->priv->contents), self->priv->resetBox, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (self->priv->homeWidget), GTK_WIDGET (self->priv->contents), FALSE, FALSE, 0);
	
    gtk_widget_show_all (self->priv->homeWidget);
	if (!self->priv->showResetDate) {
		gtk_widget_hide_all (self->priv->resetBox);
	}
}

gboolean personal_gprs_mon_update_content (PersonalGprsMon *self)
{
	//g_warning ("personal_gprs_mon_update_content");
	gchar *uploadString, *downloadString;
	gchar *resetString;
	gint upload, download, resetTime;
	GError *error = NULL;

	uploadString = gconf_client_get_string (self->priv->client, GPRS_GCONF_TX, &error);
	if (error) {
		g_warning
			("Error trying to get gconf variable %s", GPRS_GCONF_TX);
		uploadString = "0";
		g_error_free (error);
		error = NULL;
	}
	downloadString = gconf_client_get_string (self->priv->client, GPRS_GCONF_RX, &error);
	if (error) {
		g_warning
			("Error trying to get gconf variable %s", GPRS_GCONF_RX);
		downloadString = "0";
		g_error_free (error);
		error = NULL;
	}
	resetString = gconf_client_get_string (self->priv->client, GPRS_GCONF_RESET, &error);
	if (error) {
		g_warning
			("Error trying to get gconf variable %s", GPRS_GCONF_RESET);
		resetString = "0";
		g_error_free (error);
		error = NULL;
	}
	
	upload = g_ascii_strtoll (uploadString, NULL, 10);
	download = g_ascii_strtoll (downloadString, NULL, 10);
	resetTime = g_ascii_strtoll (resetString, NULL, 10);
	
	gdouble uploadSize;
	gdouble downloadSize;
	gchar *uploadUnits, *downloadUnits;
	
	if (self->priv->binaryPrefix) {
		if (upload > 1024 && upload <= 1048576) {
			uploadSize = upload / 1024.0;
			uploadUnits = " KiB";
		} else if (upload > 1048576 && upload <= 1073741824) {
			uploadSize = upload / 1048576.0;
			uploadUnits = " MiB";
		} else if (upload > 1073741824) {
			uploadSize = upload / 1073741824.0;
			uploadUnits = " GiB";
		} else {
			uploadSize = upload / 1.0;
			uploadUnits = " B";
		}
		
		if (download > 1024 && download <= 1048576) {
			downloadSize = download / 1024.0;
			downloadUnits = " KiB";
		} else if (download > 1048576 && download <= 1073741824) {
			downloadSize = download / 1048576.0;
			downloadUnits = " MiB";
		} else if (download > 1073741824) {
			downloadSize = download / 1073741824.0;
			downloadUnits = " GiB";
		} else {
			downloadSize = download / 1.0;
			downloadUnits = " B";
		}
	} else {
		if (upload > 1000 && upload <= 1000000) {
			uploadSize = upload / 1000.0;
			uploadUnits = " kB";
		} else if (upload > 1000000 && upload <= 1000000000) {
			uploadSize = upload / 1000000.0;
			uploadUnits = " MB";
		} else if (upload > 1000000000) {
			uploadSize = upload / 1000000000.0;
			uploadUnits = " GB";
		} else {
			uploadSize = upload /1.0;
			uploadUnits = " B";
		}
		
		if (download > 1000 && download <= 1000000) {
			downloadSize = download / 1000.0;
			downloadUnits = " kB";
		} else if (download > 1000000 && download <= 1000000000) {
			downloadSize = download / 1000000.0;
			downloadUnits = " MB";
		} else if (upload > 1000000000) {
			downloadSize = download / 1000000000.0;
			downloadUnits = " GB";
		} else {
			downloadSize = download / 1.0;
			downloadUnits = " B";
		}
	}
	
	GTimeVal timeVal;
	timeVal.tv_sec = resetTime;
	timeVal.tv_usec = 0;
	GDate *date = g_date_new ();
	g_date_set_time_val (date, &timeVal);
	gchar dateString[256];
	g_date_strftime (dateString, 256, "%a, %d %b %Y", date);
	gtk_label_set_text (GTK_LABEL (self->priv->resetContent), dateString);
	
	gtk_label_set_text (GTK_LABEL (self->priv->uploadContent), g_strconcat (g_strdup_printf ("%.3f", uploadSize), uploadUnits, NULL));
	gtk_label_set_text (GTK_LABEL (self->priv->downloadContent), g_strconcat (g_strdup_printf ("%.3f", downloadSize), downloadUnits, NULL));
	
	if (self->priv->showResetDate) {
		gtk_widget_show_all (self->priv->resetBox);
		gtk_widget_set_size_request (GTK_WIDGET (self), 315, 155);
		gtk_window_resize (GTK_WINDOW (self), 315, 155);
	} else {
		gtk_widget_hide (self->priv->resetBox);
		gtk_widget_set_size_request (GTK_WIDGET (self), 315, 120);
		gtk_window_resize (GTK_WINDOW (self), 315, 120);
	}
		
	return TRUE;
}

void personal_gprs_mon_settings (HDHomePluginItem *hitem, PersonalGprsMon *self)
{
	//g_warning ("personal_gprs_mon_settings");
	GtkWidget *dialog = gtk_dialog_new_with_buttons ("Personal Dataplan Monitor Settings",
        NULL, 0,
        _("wdgt_bd_save"), GTK_RESPONSE_ACCEPT, NULL);
    GtkWidget *content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
	GtkSizeGroup *group = GTK_SIZE_GROUP (gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL));
	
	GtkWidget *selector = hildon_touch_selector_new_text ();
	
	int i;
	for (i=0; i<=60; i++) {
		gchar *text = g_strdup_printf ("%d seconds(s)", i);
		hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), text);
		g_free (text);
	}
	self->priv->refresh;
	hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (selector), 0, self->priv->refresh);
	
	GtkWidget *refreshButton = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
	hildon_button_set_title (HILDON_BUTTON (refreshButton), "Update every");
	hildon_button_add_title_size_group (HILDON_BUTTON (refreshButton), group);
	hildon_button_set_alignment (HILDON_BUTTON (refreshButton), 0, 0.5, 0, 0);

	hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (refreshButton),
                                     HILDON_TOUCH_SELECTOR (selector));
	gtk_container_add (GTK_CONTAINER (content_area), refreshButton);
	
	GtkWidget *refreshLabel = gtk_label_new ("'0' will refresh only when switching to the desktop");
	gtk_label_set_markup (GTK_LABEL (refreshLabel), "<small>'0' will refresh only when switching to the desktop</small>");
	gtk_container_add (GTK_CONTAINER (content_area), refreshLabel);
	
	GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
	GtkWidget *binaryPrefix = hildon_gtk_radio_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, NULL);
	gtk_button_set_label (GTK_BUTTON (binaryPrefix), "Binary prefixes (KiB)");
	GtkWidget *siPrefix = hildon_gtk_radio_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, gtk_radio_button_get_group (GTK_RADIO_BUTTON (binaryPrefix)));
	gtk_button_set_label (GTK_BUTTON (siPrefix), "SI prefixes (kB)");
	gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (binaryPrefix), FALSE);
	gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (siPrefix), FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (binaryPrefix), self->priv->binaryPrefix);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (siPrefix), !self->priv->binaryPrefix);
	gtk_container_add (GTK_CONTAINER (hbox), binaryPrefix);
	gtk_container_add (GTK_CONTAINER (hbox), siPrefix);
	gtk_container_add (GTK_CONTAINER (content_area), hbox);
	
	GtkWidget *showResetDate = hildon_check_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
	gtk_button_set_label (GTK_BUTTON (showResetDate), "Display the last reset date");
	gtk_container_add (GTK_CONTAINER (content_area), showResetDate);
	hildon_check_button_set_active (HILDON_CHECK_BUTTON (showResetDate), self->priv->showResetDate);
	
	gtk_widget_show_all (dialog);
	
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
		self->priv->refresh = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (selector), 0);
		self->priv->binaryPrefix = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (binaryPrefix));
		self->priv->showResetDate = hildon_check_button_get_active (HILDON_CHECK_BUTTON (showResetDate));
		
		personal_gprs_mon_write_settings (self);
		personal_gprs_mon_update_content (self);
	}
	gtk_widget_destroy (dialog);
}

static void personal_gprs_mon_check_desktop (GObject *gobject, GParamSpec *pspec, PersonalGprsMon *self)
{
	//g_warning ("personal_gprs_mon_check_desktop");
	gchar *name = pspec->name;
	gboolean status;
	g_object_get (gobject, name, &status, NULL);
	if (status) {
		personal_gprs_mon_update_content (self);
		if (self->priv->eventID == 0) {
			self->priv->eventID = g_timeout_add (1*1000*60, (GSourceFunc)personal_gprs_mon_update_content, self);
		}
	} else if (self->priv->eventID != 0) {
		g_source_remove (self->priv->eventID);
		self->priv->eventID = 0;
	}
}

static void personal_gprs_mon_dispose (GObject *object)
{
	//g_warning ("personal_gprs_mon_dispose");
	PersonalGprsMon *self = PERSONAL_GPRS_MON (object);

	G_OBJECT_CLASS (personal_gprs_mon_parent_class)->dispose (object);
}

static void personal_gprs_mon_finalize (GObject *object)
{
	//g_warning ("personal_gprs_mon_finalize");
	PersonalGprsMon *self = PERSONAL_GPRS_MON (object);

	if (self->priv->eventID) {
		g_source_remove (self->priv->eventID);
	}
	g_object_unref (self->priv->client);

	G_OBJECT_CLASS (personal_gprs_mon_parent_class)->finalize (object);
}

static void personal_gprs_mon_realize (GtkWidget *widget)
{
	//g_warning ("personal_gprs_mon_realize");
	GdkScreen *screen;

	screen = gtk_widget_get_screen (widget);
	gtk_widget_set_colormap (widget, gdk_screen_get_rgba_colormap (screen));

	gtk_widget_set_app_paintable (widget, TRUE);

	GTK_WIDGET_CLASS (personal_gprs_mon_parent_class)->realize (widget);
}

static gboolean personal_gprs_mon_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
	//g_warning ("personal_gprs_mon_expose_event");
	PersonalGprsMon *self = PERSONAL_GPRS_MON (widget);
	cairo_t *cr;
	
	cr = gdk_cairo_create(GDK_DRAWABLE (widget->window));

	GdkColor color;
	gtk_style_lookup_color (gtk_rc_get_style(widget), "ContentFrameColor", &color);
	cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);
	
	gint width, height, x, y;
	gint radius = 5;
	gint iconOffset = 16;
	gint iconSize = 56;
	width = widget->allocation.width;
	height = iconSize - iconOffset;
	x = widget->allocation.x;
	y = widget->allocation.y + iconOffset;

	cairo_move_to(cr, x + radius, y);
	cairo_line_to(cr, x + width - radius, y);
	cairo_curve_to(cr, x + width - radius, y, x + width, y, x + width,
				y + radius);
	cairo_line_to(cr, x + width, y + height);
	cairo_curve_to(cr, x + width, y + height, x + width,
				y + height, x + width, y + height);
	cairo_line_to(cr, x, y + height);
	cairo_curve_to(cr, x, y + height, x, y + height, x,
				y + height);
	cairo_line_to(cr, x, y);
	cairo_curve_to(cr, x, y + radius, x, y, x + radius, y);

	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);

	cairo_fill_preserve(cr);
	
	gtk_style_lookup_color (gtk_rc_get_style(widget), "ActiveTextColor", &color);
	cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);
	cairo_set_line_width (cr, 1);
	cairo_stroke (cr);
	
	width = widget->allocation.width;
	height = widget->allocation.height - iconSize;
	x = widget->allocation.x;
	y = iconSize;
	
	gtk_style_lookup_color (gtk_rc_get_style(widget), "DefaultBackgroundColor", &color);
	cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);

	cairo_move_to(cr, x + radius, y);
	cairo_line_to(cr, x + width, y);
	cairo_line_to(cr, x + width, y + height - radius);
	cairo_curve_to(cr, x + width, y + height - radius, x + width,
				y + height, x + width - radius, y + height);
	cairo_line_to(cr, x + radius, y + height);
	cairo_curve_to(cr, x + radius, y + height, x, y + height, x,
				y + height - radius);
	cairo_line_to(cr, x, y);

	cairo_fill_preserve (cr);
	
	gtk_style_lookup_color (gtk_rc_get_style(widget), "ActiveTextColor", &color);
	cairo_set_source_rgba (cr, color.red/65535.0, color.green/65335.0, color.blue/65535.0, 0.5);
	cairo_set_line_width (cr, 1);
	cairo_stroke (cr);
	
	cairo_destroy(cr);

	return GTK_WIDGET_CLASS (personal_gprs_mon_parent_class)->expose_event (widget, event);
}

static void personal_gprs_mon_class_init (PersonalGprsMonClass *klass)
{
	//g_warning ("personal_gprs_mon_class_init");
	GObjectClass *object_class = G_OBJECT_CLASS (klass);
	GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

	object_class->dispose = personal_gprs_mon_dispose;
	object_class->finalize = personal_gprs_mon_finalize;
	
	widget_class->realize = personal_gprs_mon_realize;
	widget_class->expose_event = personal_gprs_mon_expose_event;

	g_type_class_add_private (klass, sizeof (PersonalGprsMonPrivate));
}

static void personal_gprs_mon_class_finalize (PersonalGprsMonClass *klass G_GNUC_UNUSED)
{
}

static void personal_gprs_mon_init (PersonalGprsMon *self)
{
	//g_warning ("personal_gprs_mon_init");
	self->priv = PERSONAL_GPRS_MON_GET_PRIVATE (self);
	self->priv->eventID = 0;
	
	hd_home_plugin_item_set_settings (&self->parent, TRUE);
	g_signal_connect (&self->parent, "show-settings", G_CALLBACK (personal_gprs_mon_settings), self);

	gtk_window_set_default_size (GTK_WINDOW (self), 315, 120);
	
	self->priv->homeWidget = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (self), self->priv->homeWidget);
	gtk_widget_show (self->priv->homeWidget);

	GdkGeometry hints;
	hints.min_width = 315;
	hints.min_height = 120;
	hints.max_width = 315;
	hints.max_height = 155;
	gtk_window_set_geometry_hints (GTK_WINDOW (self), self->priv->homeWidget, &hints, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
	
	g_type_init ();
	self->priv->client = gconf_client_get_default ();
	
	personal_gprs_mon_read_settings (self);
	
	personal_gprs_mon_content_create (self);
	
	g_signal_connect (self, "notify::is-on-current-desktop", G_CALLBACK (personal_gprs_mon_check_desktop), self);
}

PersonalGprsMon* personal_gprs_mon_new (void)
{
  return g_object_new (PERSONAL_TYPE_GPRS_MON, NULL);
}
